package ru.yandex.analyzer;

import org.apache.lucene.analysis.TokenFilter;
import org.apache.lucene.analysis.TokenStream;
import org.apache.lucene.analysis.tokenattributes.CharTermAttribute;

import java.io.IOException;

/**
 * A {@link TokenFilter} that pads numbers to the specified size.
 */
public final class PaddingFilter extends TokenFilter {
    private final int width;
    private final CharTermAttribute termAtt;
    private final String maxString;
    private final long max;

    public PaddingFilter(final TokenStream in, final int width) {
        super(in);
        this.width = width;
        termAtt = addAttribute(CharTermAttribute.class);
        StringBuilder sb = new StringBuilder(width);
        for (int i = 0; i < width; ++i) {
            sb.append('9');
        }
        maxString = new String(sb);
        long max;
        try {
            max = Long.parseLong(maxString);
        } catch (NumberFormatException e) {
            max = Long.MAX_VALUE;
        }
        this.max = max;
    }

    private static boolean allDigits(final char[] buf, final int len) {
        for (int i = len; i-- > 0;) {
            char c = buf[i];
            if (c < '0' || c > '9') {
                return false;
            }
        }
        return true;
    }

    @Override
    public final boolean incrementToken() throws IOException {
        if (input.incrementToken()) {
            int len = termAtt.length();
            char[] buf = termAtt.buffer();
            if (allDigits(buf, len)) {
                if (len > width) {
                    int nulls = 0;
                    for (int i = 0; i < len && buf[i] == '0'; ++i) {
                        ++nulls;
                    }
                    if (len - nulls > width) {
                        termAtt.setLength(width);
                        for (int i = 0; i < width; ++i) {
                            buf[i] = '9';
                        }
                    } else {
                        int start = len - width;
                        for (int i = start, pos = 0; i < len; ++i, ++pos) {
                            buf[pos] = buf[i];
                        }
                    }
                } else if (len < width) {
                    buf = termAtt.resizeBuffer(width);
                    termAtt.setLength(width);
                    int offset = width - len;
                    for (int i = width; i > offset;) {
                        --i;
                        buf[i] = buf[i - offset];
                    }
                    for (int i = 0; i < offset; ++i) {
                        buf[i] = '0';
                    }
                }
            } else {
                try {
                    double value = Double.parseDouble(new String(buf, 0, len));
                    if (value > max) {
                        termAtt.setEmpty();
                        termAtt.append(maxString);
                    } else {
                        buf = termAtt.resizeBuffer(width);
                        termAtt.setLength(width);
                        long longValue = Math.max((long) value, 0L);
                        for (int i = width; i-- > 0;) {
                            buf[i] = (char) ((longValue % 10L) + '0');
                            longValue /= 10L;
                        }
                    }
                } catch (NumberFormatException e) {
                    termAtt.setEmpty();
                    termAtt.append(maxString);
                }
            }
            return true;
        } else {
            return false;
        }
    }
}
