package ru.yandex.analyzer;

import java.util.ArrayList;
import java.util.concurrent.ConcurrentHashMap;

import org.apache.lucene.analysis.TokenFilter;
import org.apache.lucene.analysis.TokenStream;
import org.apache.lucene.analysis.tokenattributes.CharTermAttribute;
import org.apache.lucene.analysis.tokenattributes.PositionIncrementAttribute;

import java.io.IOException;

import ru.yandex.util.string.HexStrings;

/**
 * A {@link TokenFilter} that multiplies and takes floor of the number.
 */
public final class PermuteFilter extends TokenFilter {
    private static final ConcurrentHashMap<Long, int[][]> CACHE =
        new ConcurrentHashMap<>();
    private final CharTermAttribute termAtt;
    private final PositionIncrementAttribute posAtt;
    private final int n;
    private final int r;
    private final int[][] permutations;
    private int currentPermutation = -1;
    private char[][] blocks;
    private int len;

    public PermuteFilter(final TokenStream in, final int n, final int r) {
        super(in);
        termAtt = addAttribute(CharTermAttribute.class);
        posAtt = addAttribute(PositionIncrementAttribute.class);
        this.n = n;
        this.r = r;
        permutations = loadPermutations(n, r);
    }

    private static int[][] loadPermutations(final int n, final int r) {
        long key = n + (r << 32);
        int[][] perms = CACHE.get(key);
        if (perms == null) {
            perms = generate(n, r);
            CACHE.put(key, perms);
        }
        return perms;
    }

    /**
    * Generate all combinations of r elements from a set
    * @param n the number of elements in input set
    * @param r the number of elements in a combination
    * @return the list containing all combinations
    */
    public static int[][] generate(int n, int r) {
        ArrayList<int[]> combinations = new ArrayList<>();
        int[] combination = new int[r];
        // initialize with lowest lexicographic combination
        for (int i = 0; i < r; i++) {
            combination[i] = i;
        }
        while (combination[r - 1] < n) {
            combinations.add(combination.clone());
            // generate next combination in lexicographic order
            int t = r - 1;
            while (t != 0 && combination[t] == n - r + t) {
                t--;
            }
            combination[t]++;
            for (int i = t + 1; i < r; i++) {
                combination[i] = combination[i - 1] + 1;
            }
        }
        ArrayList<int[]> combs = new ArrayList<>();
        int k = n - r;
        int[] allNums = new int[n];
        for (int i = 0; i < n; i++) {
            allNums[i] = i;
        }
        for (int i = 0; i < combinations.size(); i++) {
            int[] newComb = new int[n];
            int[] comb = combinations.get(i);
            for (int j = n - r, t = 0; j < n; j++, t++) {
                newComb[j] = comb[t];
            }
            int minStart = 0;
            for (int j = 0; j < k; j++) {
                for (int t = minStart; t < n; t++) {
                    int num = allNums[t];
                    boolean contains = false;
                    for (int u = 0; u < comb.length; u++) {
                        if (comb[u] == num) {
                            contains = true;
                            break;
                        }
                    }
                    if (!contains) {
                        newComb[j] = num;
                        minStart = t + 1;
                        break;
                    }
                }
            }
            combs.add(newComb);
        }
        return combs.toArray(new int[0][]);
    }

    @Override
    public final boolean incrementToken() throws IOException {
        if (currentPermutation == -1
            || currentPermutation >= permutations.length)
        {
            if (input.incrementToken()) {
                this.len = termAtt.length();
                char[] buffer = termAtt.buffer();
                blocks = new char[n][];
                for (int i = 0; i < n; i++) {
                    int start = i * len / n;
                    int end = (i + 1) * len / n;
                    char[] block = new char[end - start];
                    System.arraycopy(buffer, start, block, 0, end - start);
                    blocks[i] = block;
                }
                currentPermutation = 0;
                int plen = generatePermutedBlock(currentPermutation++, buffer);
                termAtt.setLength(plen);
                return true;
            } else {
                return false;
            }
        } else {
            char[] buffer = termAtt.buffer();
            int len = generatePermutedBlock(currentPermutation++, buffer);
            termAtt.setLength(len);
            posAtt.setPositionIncrement(0);
            return true;
        }
    }

    private int generatePermutedBlock(
        final int permutationIdx,
        final char[] buffer)
    {
        int[] perm = permutations[permutationIdx];
        int len = 0;
        int k = n - r;
        for (int i = 0; i < k; i++) {
            int blockIdx = perm[i];
            char[] block = blocks[blockIdx];
            System.arraycopy(block, 0, buffer, len, block.length);
            len += block.length;
        }
        return len;
    }
}
