package ru.yandex.analyzer;

import org.apache.lucene.analysis.TokenFilter;
import org.apache.lucene.analysis.TokenStream;
import org.apache.lucene.analysis.tokenattributes.CharTermAttribute;
import org.apache.lucene.analysis.tokenattributes.PositionIncrementAttribute;

import java.io.IOException;

/**
 * A {@link TokenFilter} that pads numbers to the specified size.
 */
public final class PositionSuffixFilter extends TokenFilter {
    private final CharTermAttribute termAtt;
    private final PositionIncrementAttribute posAtt;
    private final StringBuilder sb;
    private int pos = 0;

    public PositionSuffixFilter(final TokenStream in) {
        super(in);
        termAtt = addAttribute(CharTermAttribute.class);
        posAtt = addAttribute(PositionIncrementAttribute.class);
        sb = new StringBuilder(10);
        sb.append('@');
    }

    @Override
    public final boolean incrementToken() throws IOException {
        if (input.incrementToken()) {
            char[] buf = termAtt.buffer();
            int len = termAtt.length();
            int inc = posAtt.getPositionIncrement();
            posAtt.setPositionIncrement(0);
            pos += inc;
            sb.setLength(1);
            sb.append(pos);
            if (sb.length() + len  > buf.length) {
                buf = termAtt.resizeBuffer((sb.length() << 1) + len);
            }
            int end = sb.length();
            for (int i = len, j = 0; j < end; j++, i++) {
                buf[i] = sb.charAt(j);
            }
            termAtt.setLength(len + end);
            return true;
        } else {
            return false;
        }
    }

    @Override
    public void reset() throws IOException {
        pos = 0;
        super.reset();
    }
}
