package ru.yandex.analyzer;

import java.io.Closeable;
import java.io.IOException;
import java.util.function.Function;

import org.apache.lucene.util.CloseableThreadLocal;

import ru.yandex.msearch.DefaultAnalyzerFactory;
import ru.yandex.msearch.AnalyzerProvider;
import ru.yandex.msearch.Config;
import ru.yandex.msearch.PrefixingAnalyzerWrapper;
import ru.yandex.msearch.config.DatabaseConfig;
import ru.yandex.search.prefix.Prefix;

public class ThreadLocalAnalyzerProvider implements Closeable, AnalyzerProvider {
    private final CloseableThreadLocal<PrefixingAnalyzerWrapper> searchAnalyzer
            = new CloseableThreadLocal<>();

    private final CloseableThreadLocal<PrefixingAnalyzerWrapper> indexAnalyzer
            = new CloseableThreadLocal<>();

    private final Function<DatabaseConfig, PrefixingAnalyzerWrapper> searchAnalyzerFactory;
    private final Function<DatabaseConfig, PrefixingAnalyzerWrapper> indexAnalyzerFactory;
    private final DatabaseConfig config;

    public ThreadLocalAnalyzerProvider(final DatabaseConfig config) {
        this(config, DefaultAnalyzerFactory.INDEX, DefaultAnalyzerFactory.SEARCH);
    }

    public ThreadLocalAnalyzerProvider(
            final DatabaseConfig config,
            final Function<DatabaseConfig, PrefixingAnalyzerWrapper> indexAnalyzerFactory,
            final Function<DatabaseConfig, PrefixingAnalyzerWrapper> searchAnalyzerFactory)
    {
        this.searchAnalyzerFactory = searchAnalyzerFactory;
        this.indexAnalyzerFactory = indexAnalyzerFactory;
        this.config = config;
    }

    @Override
    public void close() throws IOException {
        searchAnalyzer.close();
        indexAnalyzer.close();
    }

    @Override
    public PrefixingAnalyzerWrapper indexAnalyzer(final Prefix prefix) {
        PrefixingAnalyzerWrapper wrapper = getOrCreate(indexAnalyzer, indexAnalyzerFactory);
        wrapper.setPrefix(prefix.toString());
        return wrapper;
    }

    @Override
    public PrefixingAnalyzerWrapper searchAnalyzer(final Prefix prefix) {
        PrefixingAnalyzerWrapper wrapper = getOrCreate(searchAnalyzer, searchAnalyzerFactory);
        wrapper.setPrefix(prefix.toString());
        return wrapper;
    }

    @Override
    public PrefixingAnalyzerWrapper indexAnalyzer() {
        return getOrCreate(indexAnalyzer, indexAnalyzerFactory);
    }

    @Override
    public PrefixingAnalyzerWrapper searchAnalyzer() {
        return getOrCreate(searchAnalyzer, searchAnalyzerFactory);
    }

    protected PrefixingAnalyzerWrapper getOrCreate(
            final CloseableThreadLocal<PrefixingAnalyzerWrapper> threadLocal,
            final Function<DatabaseConfig, PrefixingAnalyzerWrapper> factory)
    {
        PrefixingAnalyzerWrapper current = threadLocal.get();
        if (current == null) {
            current = factory.apply(config);
            threadLocal.set(current);
        }

        return current;
    }
}
