package ru.yandex.analyzer;

/**
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

import java.io.IOException;

import org.apache.lucene.analysis.LowerCaseFilter; // for javadoc
import org.apache.lucene.analysis.TokenFilter;
import org.apache.lucene.analysis.TokenStream;
import org.apache.lucene.analysis.tokenattributes.CharTermAttribute;
import org.apache.lucene.analysis.tokenattributes.PositionIncrementAttribute;

/**
 * A {@link TokenFilter} that stems Russian words.
 * <p>
 * The implementation was inspired by GermanStemFilter.
 * The input should be filtered by {@link LowerCaseFilter} before passing it to RussianStemFilter ,
 * because RussianStemFilter only works with lowercase characters.
 * </p>
 */
public final class YandexLemmFilter extends TokenFilter
{
    private String[] tokensList = YandexLemmer.NO_TOKENS;
    private int tokensPos = 0;
    private CharTermAttribute termAtt;
    private PositionIncrementAttribute posAtt;

    public YandexLemmFilter(TokenStream in) {
        super(in);
        termAtt = addAttribute(CharTermAttribute.class);
        posAtt = addAttribute(PositionIncrementAttribute.class);
    }
    /**
     * Returns the next token in the stream, or null at EOS
     */
    @Override
    public final boolean incrementToken() throws IOException {
        if (tokensPos < tokensList.length) {
            termAtt.setEmpty();
            termAtt.append(tokensList[tokensPos++]);
            posAtt.setPositionIncrement(0);
            return true;
        } else {
            if (input.incrementToken()) {
                // All tokens will be alnum, no need to check < '0' case
                if (termAtt.buffer()[0] > '9' && termAtt.length() <= 55) {
                    tokensList =
                        YandexLemmer.lemm(termAtt.buffer(), termAtt.length());
                    tokensPos = 0;
                }
                if (tokensPos < tokensList.length) {
                    termAtt.append('"');
                }
                return true;
            } else {
                return false;
            }
        }
    }

    @Override
    public void reset() throws IOException {
        tokensPos = 0;
        tokensList = YandexLemmer.NO_TOKENS;
        super.reset();
    }
}
