package ru.yandex.analyzer;

import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import com.google.common.cache.Weigher;

import java.util.Arrays;
import java.util.concurrent.ExecutionException;

import ru.yandex.lemmer.Lemmer;

/**
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

public class YandexLemmer
{
    private static final int CACHE_ENTRY_WEIGHT_OVERHEAD = 80;
    private static final int AVERAGE_OBJECT_SIZE = 20;
    public static final String[] NO_TOKENS = new String[0];

    private static final int DEFAULT_LEMMER_CACHE_CONCURRENCY = 16;
    private static final int DEFAULT_LEMMER_CACHE_WEIGHT = 100000000;

    private static LoadingCache<CharBuffer, String[]> cache;

    private static final Weigher<CharBuffer, String[]> lemmWeigher =
        new Weigher<CharBuffer, String[]>() {
            @Override
            public int weigh(final CharBuffer token, final String[] lemmas) {
                int weight = CACHE_ENTRY_WEIGHT_OVERHEAD;
                weight += AVERAGE_OBJECT_SIZE + (token.length() << 1);
                for (String lemma: lemmas) {
                    weight += AVERAGE_OBJECT_SIZE + (lemma.length() << 1);
                }
                return weight;
            }
        };

    private static final CacheLoader<CharBuffer, String[]> lemmLoader =
        new CacheLoader<CharBuffer, String[]>() {
            @Override
            public String[] load(final CharBuffer input) {
                input.unshare();
                String[] lemmas = input.analyze();
                if (lemmas == null) {
                    lemmas = NO_TOKENS;
                }
                return lemmas;
            }
        };

    static {
        setCacheParams(DEFAULT_LEMMER_CACHE_CONCURRENCY,
            DEFAULT_LEMMER_CACHE_WEIGHT);
    }

    public static void setCacheParams(final int concurrency, final int weight)
    {
        cache = CacheBuilder.newBuilder()
            .concurrencyLevel(concurrency)
            .maximumWeight(weight)
            .weigher(lemmWeigher)
            .build(lemmLoader);
    }

    public static String[] lemm(final char[] buf) {
        return lemm(buf, buf.length);
    }

    public static String[] lemm(final char[] buf, final int len) {
        try {
            return cache.get(new CharBuffer(buf, len));
        } catch (ExecutionException e) {
            return NO_TOKENS;
        }
    }

    private static class CharBuffer {
        private final int len;
        private char[] buf;
        private int hash = 0;

        public CharBuffer(final char[] buf, final int len) {
            this.buf = buf;
            this.len = len;
        }

        public int length() {
            return len;
        }

        public String[] analyze() {
            return Lemmer.analyzeWord(buf, len);
        }

        public void unshare() {
            buf = Arrays.copyOf(buf, len);
        }

        @Override
        public int hashCode() {
            int hash = this.hash;
            if (hash == 0) {
                for (int i = 0; i < len; ++i) {
                    hash *= 31;
                    hash += buf[i];
                }
                this.hash = hash;
            }
            return hash;
        }

        @Override
        public boolean equals(final Object o) {
            if (o instanceof CharBuffer) {
                CharBuffer cbuf = (CharBuffer) o;
                if (len == cbuf.len) {
                    for (int i = 0; i < len; ++i) {
                        if (buf[i] != cbuf.buf[i]) {
                            return false;
                        }
                    }
                    return true;
                }
            }
            return false;
        }
    }
}
