package ru.yandex.analyzer.preprocessor;

import java.io.IOException;
import java.io.Reader;
import java.io.StringReader;
import java.util.List;

import ru.yandex.json.parser.JsonException;
import ru.yandex.json.parser.JsonParser;
import ru.yandex.json.xpath.PathComponent;
import ru.yandex.json.xpath.PathMatcher;
import ru.yandex.json.xpath.PrimitiveHandler;
import ru.yandex.json.xpath.XPathContentHandler;

public class JsonSelectorPreprocessor implements FieldPreprocessor {
    private final PathMatcher matcher;
    private final DuplicationPolicy onDuplicate;

    public JsonSelectorPreprocessor(
        final PathMatcher matcher,
        final DuplicationPolicy onDuplicate)
    {
        this.matcher = matcher;
        this.onDuplicate = onDuplicate;
    }

    @Override
    public Reader preprocess(final Reader reader) throws IOException {
        Handler handler = new Handler(matcher, onDuplicate);
        JsonParser parser =
            new JsonParser(new XPathContentHandler(handler));
        try {
            parser.parse(reader);
        } catch (JsonException e) {
            Throwable cause = e.getCause();
            if (cause instanceof FieldPreprocessorException) {
                throw (FieldPreprocessorException) cause;
            } else {
                throw new FieldPreprocessorException(e);
            }
        }
        if (handler.value == null) {
            return Reader.nullReader();
        } else {
            return new StringReader(handler.value);
        }
    }

    private static class Handler implements PrimitiveHandler {
        private final PathMatcher matcher;
        private final DuplicationPolicy onDuplicate;
        private String value = null;

        Handler(
            final PathMatcher matcher,
            final DuplicationPolicy onDuplicate)
        {
            this.matcher = matcher;
            this.onDuplicate = onDuplicate;
        }

        @Override
        public void handle(
            final List<PathComponent> path,
            final Object value)
            throws JsonException
        {
            if (value != null && matcher.matches(path)) {
                if (this.value == null) {
                    this.value = value.toString();
                } else {
                    try {
                        this.value =
                            onDuplicate.apply(this.value, value.toString());
                    } catch (FieldPreprocessorException e) {
                        throw new JsonException(e);
                    }
                }
            }
        }
    }
}

