package ru.yandex.lympho;

import java.io.IOException;
import java.net.URI;
import java.net.URISyntaxException;
import java.util.logging.Level;
import java.util.logging.Logger;

import org.apache.http.HttpStatus;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.entity.ByteArrayEntity;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClients;

import ru.yandex.function.GenericAutoCloseable;
import ru.yandex.json.writer.JsonType;
import ru.yandex.parser.string.URIParser;

public abstract class AbstractHttpOutput
        extends LymphoJsonWriter
        implements GenericAutoCloseable<IOException>
{
    private final URI uri;
    private final int batchSize;
    private final CloseableHttpClient client;
    protected int batchRecords = 0;
    protected int totalRecords = 0;
    protected boolean newBatch = true;
    protected int batchNum = 0;
    private volatile boolean stopped = false;

    public AbstractHttpOutput(
            final String uri,
            final int batchSize)
            throws URISyntaxException
    {
        super(JsonType.HUMAN_READABLE, batchSize * 1000);

        this.uri = URIParser.INSTANCE.apply(uri);
        this.batchSize = batchSize;
        this.client = HttpClients.createDefault();
    }


    protected abstract void writeBeforeBatch() throws IOException;

    protected abstract void writeAfterBatch() throws IOException;

    @Override
    public void startObject() throws IOException {
        if (level == 1 && newBatch) {
            // do prefix
            System.out.println("Level 1 , before batch");
            writeBeforeBatch();
            newBatch = false;

            System.out.println("Before batch printed, level " + level);
        }
        super.startObject();
    }

    @Override
    public void endObject() throws IOException {
        super.endObject();

        if (level == 1) {
            batchRecords += 1;
            totalRecords += 1;
            System.out.println("Level 1 endObj");
            if (batchRecords >= batchSize) {
                System.out.println("Level 1 , sending batch");
                writeAfterBatch();
                wrapped.close();
                wrapped.reset();

                send();

                level = 1;
                baos.reset();
                batchRecords = 0;
                batchNum += 1;
                newBatch = true;
            }
            // record ended
        }
    }

    @Override
    public void startArray() throws IOException {
        if (level == 1) {
            throw new IOException("Array is not available on this level");
        }

        super.startArray();
    }

    protected abstract Logger logger();

    protected void send() throws IOException {
        HttpPost post = new HttpPost(uri);
        post.setEntity(new ByteArrayEntity(baos.toByteArray()));
        try {
            while (!stopped) {
                int status = -1;
                try (CloseableHttpResponse response = client.execute(post)) {
                    status = response.getStatusLine().getStatusCode();
                    if (status == HttpStatus.SC_OK) {
                        break;
                    }

                    logger().warning("Batch send failed with status " + status);
                } catch (IOException ioe) {
                    logger().log(Level.WARNING, "Send batch error", ioe);
                }

                Thread.sleep(1000);
            }
        } catch (InterruptedException ie) {
            throw new IOException("Interrupted", ie);
        }
    }

    public void finish() throws IOException {
        if (batchRecords > 0) {
            writeAfterBatch();
            send();
            batchRecords = 0;
            level = 1;
            newBatch = true;
        }
    }

    @Override
    public void close() throws IOException {
        finish();
        stopped = true;
    }
}
