package ru.yandex.lympho;

import java.io.File;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.TimeUnit;
import java.util.logging.Level;
import java.util.stream.Collectors;

import jdk.nashorn.api.scripting.ScriptObjectMirror;
import org.apache.lucene.queryParser.ParseException;
import org.apache.lucene.queryParser.QueryParser;
import org.apache.lucene.search.Query;
import org.apache.lucene.util.Version;

import ru.yandex.function.GenericAutoCloseableChain;
import ru.yandex.msearch.Index;
import ru.yandex.msearch.PrefixingAnalyzerWrapper;
import ru.yandex.msearch.Searcher;
import ru.yandex.msearch.config.DatabaseConfig;
import ru.yandex.msearch.printkeys.PrintKeysParams;
import ru.yandex.parser.uri.CgiParams;
import ru.yandex.queryParser.QueryParserFactory;
import ru.yandex.queryParser.YandexQueryParserFactory;
import ru.yandex.search.NullScorerFactory;
import ru.yandex.search.prefix.Prefix;
import ru.yandex.util.timesource.TimeSource;

public abstract class AbstractLymphoContext implements LymphoContext {
    protected final Index index;
    protected final QueryParserFactory parserFactory;
    protected final GenericAutoCloseableChain<IOException> chain;

    public AbstractLymphoContext(final Index index) {
        this.index = index;
        this.chain = new GenericAutoCloseableChain<>();

        this.parserFactory = new NullScorerFactory(index.config());
    }

    protected Query parseQuery(
        final String text,
        final String prefix)
        throws ParseException
    {
        PrefixingAnalyzerWrapper analyzer = index.searchAnalyzer();
        QueryParser parser = parserFactory
            .create(Version.LUCENE_40, null, analyzer);
        parser.setLowercaseExpandedTerms(true);
        parser.setReplaceEExpandedTerms(true);
        parser.useOldWildcatdQuery(true);
        parser.setAllowLeadingWildcard(true);

        analyzer.setPrefix(prefix);

        return parser.parse(text);
    }

    public long currentTimeMillis() {
        return TimeSource.INSTANCE.currentTimeMillis();
    }

    @Override
    public void close() throws IOException {
        chain.close();
    }

    public Searcher searcher(final int shard, final boolean reverse) throws IOException {
        Searcher searcher;
        if (reverse) {
            searcher = index.getDiskSearcher(shard);
        } else {
            searcher = index.getSearcher(shard, false);
        }

        this.chain.add(searcher::close);
        return searcher;
    }

    public Searcher searcher(final Prefix prefix, final boolean reverse) throws IOException {
        Searcher searcher;
        if (reverse) {
            searcher = index.getDiskSearcher(prefix);
        } else {
            searcher = index.getSearcher(prefix, false);
        }

        this.chain.add(searcher::close);
        return searcher;
    }

    public Iterable<LymphoTerm> keys(final String field) throws Exception {
        Map<String, Collection<String>> params = new LinkedHashMap<>();
        //params.put("shard", Collections.singletonList(String.valueOf(shard)));
        params.put("field", Collections.singletonList(field));

        return keys(params);
    }

    public Iterable<LymphoTerm> keys(final String field, final boolean freqs) throws Exception {
        Map<String, Collection<String>> params = new LinkedHashMap<>();
        params.put("print-freqs", Collections.singletonList(String.valueOf(freqs)));
        params.put("field", Collections.singletonList(field));

        return keys(params);
    }

    @SuppressWarnings("deprecation")
    public Iterable<LymphoTerm> keys(final Map<String, ?> obj) throws Exception {
        Map<String, Collection<String>> params = new LinkedHashMap<>(obj.size() << 1);
        for (Map.Entry<String, ?> entry: obj.entrySet()) {
            List<String> value;
            if (entry.getValue() instanceof ScriptObjectMirror) {
                ScriptObjectMirror nestedObj = ((ScriptObjectMirror) entry.getValue());
                if (nestedObj.isArray()) {
                    value = new ArrayList<>(nestedObj.size());
                    for (Object item: nestedObj.values()) {
                        value.add(String.valueOf(item));
                    }
                } else {
                    value = Collections.singletonList(String.valueOf(nestedObj));
                }
            } else if (entry.getValue() instanceof Collection) {
                Collection<?> collection = ((Collection<?>) entry.getValue());
                value = new ArrayList<>(collection.size());
                for (Object item: collection) {
                    value.add(String.valueOf(item));
                }
            } else {
                value = Collections.singletonList(String.valueOf(entry.getValue()));
            }
            params.put(entry.getKey(), value);
        }
        Map<String, QueryParserFactory> queryParserFactory = new LinkedHashMap<>();
        for (Map.Entry<String, DatabaseConfig> entry: index.daemonConfig().databasesConfigs().entrySet()) {
            queryParserFactory.put(entry.getKey(), new YandexQueryParserFactory(entry.getValue()));
        }

        PrintKeysParams printKeysParams = new PrintKeysParams(
            queryParserFactory,
            index,
            new CgiParams(params),
            null);
        return new LymphoKeysIterator(this, printKeysParams);
    }

    public Iterator<String> sbr(final String id, final String name) throws Exception {
        Path tempDir = Files.createTempDirectory(Path.of("/tmp"), "sbr_");
        tempDir.toFile().deleteOnExit();
        logger().log(Level.WARNING, "Downloading resource to " + tempDir);
        Process process = null;
        try  {
            ProcessBuilder pb = new ProcessBuilder("sky get sbr:" + id);
            pb.directory(tempDir.toFile());
            process = pb.start();
            if (!process.waitFor(10, TimeUnit.MINUTES)) {
                throw new IOException("Timout on waiting sbr resource");
            }

            return Files.lines(tempDir.resolve(name)).iterator();
        } finally {
            if (process != null) {
                process.destroy();
                process.destroyForcibly();
            }
            List<Path> children = Files.list(tempDir).collect(Collectors.toList());
            for (Path path: children) {
                if (!path.toFile().delete()) {
                    logger().log(Level.WARNING, "Failed to remove " + path);
                }
            }
            if (!tempDir.toFile().delete()) {
                logger().log(Level.WARNING, "Failed to remove temp dir" + tempDir);
            }
        }

        //return null;
    }
}
