package ru.yandex.lympho;

import java.io.IOException;
import java.net.URI;
import java.net.URISyntaxException;
import java.util.logging.Level;

import org.apache.http.HttpStatus;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.entity.ByteArrayEntity;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClients;

import ru.yandex.function.GenericAutoCloseable;
import ru.yandex.json.writer.JsonType;
import ru.yandex.parser.string.URIParser;

public class HttpOutput extends LymphoJsonWriter implements GenericAutoCloseable<IOException> {
    private final URI uri;
    private final int batchSize;
    private final CloseableHttpClient client;
    private final LymphoWorkerTask task;
    private int batchRecords = 0;
    private int totalRecords = 0;
    private int batchNum = 0;
    private volatile boolean stopped = false;

    public HttpOutput(
        final LymphoWorkerTask task,
        final String uri,
        final int batchSize)
        throws URISyntaxException
    {
        super(JsonType.HUMAN_READABLE, batchSize * 1000);

        this.task = task;
        this.uri = URIParser.INSTANCE.apply(uri);
        this.batchSize = batchSize;
        this.client = HttpClients.createDefault();
    }

    protected void writeMeta() throws IOException {
        wrapped.startObject();
        wrapped.key("config");
        wrapped.startObject();
        wrapped.key("jobId");
        wrapped.value(task.config().jobId());
        wrapped.key("taskId");
        wrapped.value(task.config().taskId());
        wrapped.key("shard");
        wrapped.value(task.shard());
        wrapped.endObject();
        //wrapped.value(task.config());
        wrapped.key("meta");
        wrapped.startObject();
        wrapped.key("records_processed");
        wrapped.value(totalRecords);
        wrapped.key("batch_num");
        wrapped.value(batchNum);
        wrapped.endObject();
        wrapped.key("records");
        wrapped.startArray();
    }

    @Override
    public void startObject() throws IOException {
        if (level == 1) {
            // do prefix
            writeMeta();
        }
        super.startObject();
    }

    @Override
    public void endObject() throws IOException {
        super.endObject();

        if (level == 1) {
            batchRecords += 1;
            totalRecords += 1;

            if (batchRecords >= batchSize) {
                wrapped.endArray();
                wrapped.endObject();
                wrapped.close();
                wrapped.reset();

                send();

                level = 1;
                baos.reset();
                batchRecords = 0;
                batchNum += 1;
            }
            // record ended
        }
    }

    @Override
    public void startArray() throws IOException {
        if (level == 1) {
            throw new IOException("Array is not available on this level");
        }

        super.startArray();
    }

    protected void send() throws IOException {
        HttpPost post = new HttpPost(uri);
        post.setEntity(new ByteArrayEntity(baos.toByteArray()));
        try {
            while (!stopped) {
                int status = -1;
                try (CloseableHttpResponse response = client.execute(post)) {
                    status = response.getStatusLine().getStatusCode();
                    if (status == HttpStatus.SC_OK) {
                        break;
                    }

                    task.logger().warning("Batch send failed with status " + status);
                } catch (IOException ioe) {
                    task.logger().log(Level.WARNING, "Send batch error", ioe);
                }

                Thread.sleep(1000);
            }
        } catch (InterruptedException ie) {
            throw new IOException("Interrupted", ie);
        }
    }

    public void finish() throws IOException {
        if (batchRecords > 0) {
            send();
            batchRecords = 0;
        }
    }

    @Override
    public void close() throws IOException {
        finish();
        stopped = true;
    }
}
