package ru.yandex.lympho;

import java.io.IOException;
import java.util.Iterator;

import org.apache.lucene.index.DocsAndPositionsEnum;
import org.apache.lucene.index.DocsEnum;
import org.apache.lucene.index.Fields;
import org.apache.lucene.index.IndexReader;
import org.apache.lucene.index.MultiFields;
import org.apache.lucene.index.Terms;
import org.apache.lucene.index.TermsEnum;
import org.apache.lucene.search.DocIdSetIterator;
import org.apache.lucene.util.Bits;
import org.apache.lucene.util.BytesRef;

import ru.yandex.msearch.Searcher;
import ru.yandex.msearch.collector.PruningCollector;
import ru.yandex.msearch.printkeys.PrintKeysParams;
import ru.yandex.msearch.util.InvertBits;
import ru.yandex.msearch.util.OrNotBits;
import ru.yandex.msearch.util.SparseBitSet;

public class LymphoKeysIterator implements Iterator<LymphoTerm>, Iterable<LymphoTerm> {
    private final AbstractLymphoContext context;
    private final PrintKeysParams params;
    private LymphoTerm next;
    private SparseBitSet queryDocs;
    private final Bits skipDocs;
    private final IndexReader reader;
    private final Fields fields;
    private final Terms terms;
    private Iterator<LymphoTerm> singlePrefixIterator;
    private final Iterator<BytesRef> prefixesIterator;

    public LymphoKeysIterator(final AbstractLymphoContext context, final PrintKeysParams params) throws IOException {
        this.params = params;
        this.context = context;

        Searcher searcher;
        if (params.shard() != null) {
            searcher = context.searcher(params.shard(), params.reverse());
        } else {
            searcher = context.searcher(params.user(), params.reverse());
        }
        reader = searcher.reader();
        fields = MultiFields.getFields(reader);
        if (fields != null) {
            terms = fields.terms(params.field());
            if (terms != null) {
                if (params.skipDeleted()) {
                    Bits skipDocs = MultiFields.getDeletedDocs(reader);
                    if (queryDocs != null) {
                        if (skipDocs == null) {
                            skipDocs = new InvertBits(queryDocs);
                        } else {
                            skipDocs =
                                new OrNotBits(skipDocs, queryDocs);
                        }
                    }
                    this.skipDocs = skipDocs;
                } else {
                    // queryDocs != null => skipDeleted
                    skipDocs = null;
                }

                prefixesIterator = params.prefixRefs().iterator();
                nextWithException();
            } else {
                prefixesIterator = null;
                skipDocs = null;
            }
        } else {
            prefixesIterator = null;
            skipDocs = null;
            terms = null;
        }
    }

    @Override
    public boolean hasNext() {
        return next != null;
    }

    private LymphoTerm nextWithException() throws IOException {
        LymphoTerm result = next;
        if (terms == null) {
            return result;
        }

        next = null;
        if (singlePrefixIterator != null && singlePrefixIterator.hasNext()) {
            next = singlePrefixIterator.next();
        } else {
            while (prefixesIterator.hasNext()) {
                BytesRef prefix = prefixesIterator.next();
                singlePrefixIterator = new PrefixTermIterator(terms, prefix);
                if (singlePrefixIterator.hasNext()) {
                    next = singlePrefixIterator.next();
                    break;
                }
            }
        }

        return result;
    }

    @Override
    public Iterator<LymphoTerm> iterator() {
        return this;
    }

    @Override
    public LymphoTerm next() {
        if (!hasNext()) {
            return null;
        }

        try {
            return nextWithException();
        } catch (IOException ioe) {
            return null;
        }
    }

    private class PrefixTermIterator implements Iterator<LymphoTerm> {
        private final Terms terms;
        private final BytesRef prefix;
        private final TermsEnum termsEnum;
        private DocsEnum docsEnum;
        private BytesRef term;
        final int docsMax;
        final int docsOffset;
        long pos;
        LymphoTerm nextTerm;

        public PrefixTermIterator(final Terms terms, final BytesRef prefix) throws IOException {
            this.terms = terms;
            TermsEnum termsEnum;
            if (params.reverse()) {
                termsEnum = terms.reverseIterator();
            } else {
                termsEnum = terms.iterator();
            }

            this.termsEnum = termsEnum;
            this.prefix = prefix;
            long pos = 0;
            DocsEnum docsEnum = null;
            DocsAndPositionsEnum docsAndPosEnum = null;
            docsMax = params.docsOffset() + params.docsLength();
            docsOffset = params.docsOffset();
            nextWithException();
        }

        @Override
        public boolean hasNext() {
            return nextTerm != null;
        }

        private LymphoTerm nextWithException() throws IOException {
            LymphoTerm result = nextTerm;

            if (pos == 0) {
                if (prefix == null) {
                    term = termsEnum.next();
                } else {
                    if (params.reverse()) {
                        BytesRef seekPrefix = new BytesRef(prefix);
                        seekPrefix.append(PruningCollector.REVERSE_SEEK_SUFFIX);
                        termsEnum.seek(seekPrefix);
                    } else {
                        termsEnum.seek(prefix);
                    }
                    term = termsEnum.term();
                }
            } else {
                ++pos;
                term = termsEnum.next();
            }
            nextTerm = null;

            while (true) {
                if (!keepIteratingTerms(term)) {
                    return result;
                }

                if (validTerm(term)) {
                    break;
                }

                ++pos;
                term = termsEnum.next();
            }

            while (keepIteratingTerms(term)) {
                int freq;
                int totalFreq = 0;
                if (params.printTotalFreqs() || skipDocs != null) {
                    // Check that at least on doc position is not deleted
                    // for this term, otherwise skip term
                    freq = 0;
                    docsEnum = termsEnum.docs(skipDocs, docsEnum);
                    for (int docId = docsEnum.nextDoc();
                         docId != DocIdSetIterator.NO_MORE_DOCS;
                         docId = docsEnum.nextDoc()) {
                        ++freq;
                        totalFreq += docsEnum.freq();
                        if (freq == params.maxFreq()) {
                            break;
                        }
                    }
                    if (freq == 0) {
//                            ++pos;
                        term = termsEnum.next();
                        continue;
                    } else if (pos < params.offset()) {
                        ++pos;
                        term = termsEnum.next();
                        continue;
                    }
                } else if (params.printFreqs()) {
                    freq = termsEnum.docFreq();
                } else {
                    freq = 0;
                }

                if (nextTerm == null) {
                    nextTerm = new LymphoTerm();
                }
                nextTerm.reset();
                nextTerm.term(term);
                nextTerm.freq(freq, totalFreq);
                return result;
            }

            return result;
        }


        private boolean validTerm(final BytesRef term) {
            return (pos >= params.offset() || skipDocs != null)
                       && (!params.exact() || term.endsWith((byte) '"'));
        }

        private boolean keepIteratingTerms(final BytesRef term) {
            return term != null
                && pos < params.offset() + params.length()
                && (prefix == null || term.startsWith(prefix));
        }
        @Override
        public LymphoTerm next() {
            if (!hasNext()) {
                return null;
            }

            try {
                return nextWithException();
            } catch (IOException ioe) {
                return null;
            }
        }
    }
}
