package ru.yandex.lympho;

import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.util.LinkedHashSet;
import java.util.Set;
import java.util.logging.Level;
import java.util.stream.Collectors;

import org.apache.http.HttpException;
import org.apache.http.HttpStatus;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.entity.StringEntity;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClients;

import ru.yandex.function.GenericAutoCloseable;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.BadResponseException;
import ru.yandex.http.util.CharsetUtils;
import ru.yandex.io.StringBuilderWriter;
import ru.yandex.json.dom.JsonList;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.TypesafeValueContentHandler;
import ru.yandex.json.writer.JsonType;
import ru.yandex.json.writer.JsonWriter;
import ru.yandex.logger.PrefixedLogger;
import ru.yandex.msearch.Config;
import ru.yandex.msearch.Daemon;
import ru.yandex.msearch.config.DatabaseConfig;
import ru.yandex.parser.uri.QueryConstructor;

public class LymphoNodeService implements Runnable, GenericAutoCloseable<IOException> {
    private final LymphoNodeWorker worker;
    private static final long TASK_FETCH_DELAY = 10000;
    private static final long TASK_STATUS_CHECK_DELAY = 200;
    private static final long PREFIX = 0L;
    private final Daemon daemon;
    private final Thread thread;
    private final PrefixedLogger logger;
    private final CloseableHttpClient client;
    private volatile boolean stopped = false;

    public LymphoNodeService(final Daemon daemon) {
        client = HttpClients.createDefault();
        logger = daemon.httpSearchServer().logger();
        this.daemon = daemon;
        this.thread = new Thread(this, "LymphoService");
        thread.setDaemon(true);

        this.worker = new LymphoNodeWorker(daemon);
    }

    public void start() {
        worker.start();
        this.start();
    }

    @Override
    public void close() throws IOException {
        worker.close();
        this.close();
    }

    protected WorkerTaskConfig nextTask() throws IOException, HttpException {
        HttpGet get = new HttpGet(
            "http://localhost:" + daemon.jsonServerPort()
                + "/search?text=lympho_create_time:*&get=*&sort=lympho_create_time&length=1");

        try (CloseableHttpResponse response = client.execute(get)) {
            if (HttpStatus.SC_OK != response.getStatusLine().getStatusCode()) {
                throw new BadResponseException(get, response);
            }

            JsonMap tasksMap =
                TypesafeValueContentHandler.parse(CharsetUtils.content(response.getEntity())).asMap();

            JsonList tasks = tasksMap.getList("hitsArray");
            if (tasks.size() <= 0) {
                return null;
            }

            return new WorkerTaskConfig(tasks.get(0).asMap(), daemon.index().config());
        } catch (Exception je) {
            logger.log(Level.WARNING, "Failed to parse task", je);
            throw new IOException(je);
        }
    }

    @Override
    public void run() {
        try {
            WorkerTaskConfig task = null;
            while (!stopped) {
                try {
                    task = nextTask();
                } catch (Exception e) {
                    // do something on broken job
                    logger.log(Level.WARNING, "Can not get new task");
                }

                if (task == null) {
                    Thread.sleep(TASK_FETCH_DELAY);
                    continue;
                }

                if (task.deadline() > System.currentTimeMillis()) {
                    logger.info("Task over deadlin, cleaning up " + task.taskId());
                    removeTaskLoop(task);
                    Thread.sleep(TASK_FETCH_DELAY);
                    continue;
                }

                logger.info("New task, scheduling " + task.taskId());
                try {
                    executeTask(task);
                } catch (IOException ioe) {
                    logger.warning("Lympho task failed " + task.taskId());
                } catch (InterruptedException e) {
                    logger.warning("Lympho service thread was interrupted");
                    stopped = true;
                    break;
                }

                Thread.sleep(TASK_FETCH_DELAY);
            }
        } catch (InterruptedException ie) {
            logger.log(Level.WARNING, "Thread interrupted", ie);
        }
    }

    public void executeTask(final WorkerTaskConfig config) throws IOException, InterruptedException{
        Set<Integer> shards = new LinkedHashSet<>(config.backendShards());
        shards.removeAll(config.finishedShards());
        logger.info("Scheduling task " + config.taskId() + " for shards " + shards);

        for (Integer shard: shards) {
            logger.info("Scheduling shard " + shard + " for task " + config.taskId());
            LymphoWorkerTask task = new LymphoWorkerTask(config, logger, shard);
            boolean scheduled = worker.schedule(task);
            if (!scheduled) {
                logger.warning("Failed to schedule task, timeout");
                continue;
            }

            LymphoWorkerTaskStatus status = task.status();
            while (!stopped) {
                status = task.status();
                if (status != LymphoWorkerTaskStatus.PENDING
                    && status != LymphoWorkerTaskStatus.RUNNING) {
                    break;
                }

                Thread.sleep(TASK_STATUS_CHECK_DELAY);
            }

            logger.info(
                "Task finished shard " + shard + " for task " + config.taskId() + " with status " + status);

            if (status == LymphoWorkerTaskStatus.COMPLETED_OK) {
                //finish shard
                shardCompleted(config, shard);
            } else {
                logger.warning("Shard processing failed " + shard + " for task " + task.config().taskId());
            }
        }

        shards.removeAll(config.finishedShards());
        if (shards.isEmpty()) {
            // finished
            taskCompleted(config);
        } else {
            if (config.incRetries() >= config.maxRetries()) {
                taskFailed(config);
            }
        }
    }

    public void removeTaskLoop(final TaskConfigBuilder task) throws InterruptedException {
        while (!stopped) {
            try {
                removeTask(task);
            } catch (Exception e) {
                logger.log(Level.SEVERE, "Failed to save task", e);
            }

            Thread.sleep(3000);
        }
    }

    public DatabaseConfig indexConfig() {
        return daemon.index().config();
    }

    public void removeTask(final TaskConfigBuilder taskConfig) throws HttpException, IOException {
        StringBuilderWriter sbw = new StringBuilderWriter();
        try (JsonWriter writer = JsonType.NORMAL.create(sbw)) {
            writer.startObject();
            writer.key("prefix");
            writer.value(PREFIX);
            writer.key("docs");
            writer.startArray();
            writer.startObject();
            for (String key: daemon.index().config().primaryKey()) {
                writer.key(key);
                writer.value(taskConfig.taskId());
            }
            writer.endObject();
            writer.endArray();
            writer.endObject();
        }

        HttpPost post = new HttpPost(
            "http://localhost:" + daemon.jsonIndexerServer().port()
                + "/delete?&lympho&" + taskConfig.taskId());

        try (CloseableHttpResponse response = client.execute(post)) {
            if (HttpStatus.SC_OK != response.getStatusLine().getStatusCode()) {
                throw new BadResponseException(post, response);
            }
        }
    }

    public void saveTaskLoop(final TaskConfigBuilder task) throws InterruptedException {
        while (!stopped) {
            try {
                saveTask(task);
            } catch (Exception e) {
                logger.log(Level.SEVERE, "Failed to save task", e);
            }

            Thread.sleep(3000);
        }
    }

    public void saveTask(final TaskConfigBuilder taskConfig) throws HttpException, IOException {
        StringBuilderWriter sbw = new StringBuilderWriter();
        try (JsonWriter writer = JsonType.NORMAL.create(sbw)) {
            writer.startObject();
            writer.key("prefix");
            writer.value(PREFIX);
            writer.key("docs");
            writer.startArray();
            writer.value(taskConfig);
            writer.endArray();
            writer.endObject();
        }

        HttpPost post = new HttpPost(
            "http://localhost:" + daemon.jsonIndexerServer().port()
                + "/modify?save-task&id=" + taskConfig.taskId());
        post.setEntity(new StringEntity(sbw.toString(), StandardCharsets.UTF_8));
        try (CloseableHttpResponse response = client.execute(post)) {
            if (HttpStatus.SC_OK != response.getStatusLine().getStatusCode()) {
                throw new BadResponseException(post, response);
            }
        }
    }

    protected void shardCompleted(
        final WorkerTaskConfig task,
        final int shard)
        throws InterruptedException, IOException
    {
        task.finishShard(shard);
        // first save to index
        saveTaskLoop(task);
    }

    protected void taskFailed(final WorkerTaskConfig task) throws InterruptedException, IOException {
        // first save to index
        saveTaskLoop(task);

        // then notify scheduler
        QueryConstructor qc = new QueryConstructor(task.schedulerUri().toString());
        try {
            qc.append("jobId", task.jobId());
            qc.append("taskId", task.taskId());
            qc.append(
                "shards",
                task.backendShards().stream().map(Object::toString).collect(
                    Collectors.joining(",")));
            qc.append("status", "failed");
        } catch (BadRequestException bre) {
            throw new IOException(bre);
        }

        int retries = 0;
        boolean finished = false;
        while (retries < 10) {
            HttpGet get = new HttpGet(qc.toString());
            try (CloseableHttpResponse response = client.execute(get)) {
                int status = response.getStatusLine().getStatusCode();
                if (status == HttpStatus.SC_OK) {
                    finished = true;
                    break;
                }

                retries += 1;
                logger.log(
                    Level.WARNING,
                    "Failed to notify scheduler about running task, status " + status);
            }

            Thread.sleep(60000);
        }

        if (finished) {
            removeTaskLoop(task);
        }
    }

    protected void taskCompleted(final WorkerTaskConfig task) throws InterruptedException, IOException {
        // first save to index
        saveTaskLoop(task);

        // then notify scheduler
        QueryConstructor qc = new QueryConstructor(task.schedulerUri().toString());
        try {
            qc.append("jobId", task.jobId());
            qc.append("taskId", task.taskId());
            qc.append(
                "shards",
                task.backendShards().stream().map(Object::toString).collect(
                    Collectors.joining(",")));
            qc.append("status", "completed");
        } catch (BadRequestException bre) {
            throw new IOException(bre);
        }

        int retries = 0;
        boolean finished = false;
        while (retries < 10) {
            HttpGet get = new HttpGet(qc.toString());
            try (CloseableHttpResponse response = client.execute(get)) {
                int status = response.getStatusLine().getStatusCode();
                if (status == HttpStatus.SC_OK) {
                    finished = true;
                    break;
                }

                retries += 1;
                logger.log(
                    Level.WARNING,
                    "Failed to notify scheduler about running task, status " + status);
            }

            Thread.sleep(60000);
        }

        if (finished) {
            removeTaskLoop(task);
        }
    }
}
