package ru.yandex.lympho;

import java.io.IOException;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.TimeUnit;
import java.util.logging.Level;

import javax.script.ScriptContext;
import javax.script.ScriptEngine;
import javax.script.ScriptEngineManager;
import javax.script.SimpleScriptContext;

import ru.yandex.function.GenericAutoCloseable;
import ru.yandex.logger.PrefixedLogger;
import ru.yandex.msearch.Daemon;
import ru.yandex.msearch.Index;

public class LymphoNodeWorker
    implements GenericAutoCloseable<IOException>, Runnable
{
    private final ScriptEngine engine;
    private final Index index;
    private final Daemon daemon;
    private final Thread thread;
    private final PrefixedLogger logger;
    private volatile LymphoWorkerTask currentTask;
    private volatile boolean stopped = false;
    private final ArrayBlockingQueue<LymphoWorkerTask> tasks = new ArrayBlockingQueue<>(1);

    public LymphoNodeWorker(final Daemon daemon) {
        ScriptEngineManager factory = new ScriptEngineManager();
        engine = factory.getEngineByName("nashorn");

        this.index = daemon.index();
        this.thread = new Thread(this, "LymphoWorker");
        thread.setDaemon(true);
        this.logger = daemon.httpSearchServer().logger().replacePrefix("LymphoWorker");
        this.daemon = daemon;
    }

    public void start() {
        thread.start();
    }

    public boolean schedule(final LymphoWorkerTask task) throws InterruptedException {
        return tasks.offer(task, 1, TimeUnit.MINUTES);
    }

    public LymphoWorkerTaskStatus execute(final LymphoWorkerTask task) {
        logger.info("Executing task " + task.config().taskId() + " for shard " + task.shard());
        try (BasicLymphoContext mrContext = new BasicLymphoContext(index, task)) {
            task.logger().info("Starting");
            ScriptContext context = new SimpleScriptContext();
            context.setAttribute(
                "context",
                mrContext,
                ScriptContext.ENGINE_SCOPE);

            engine.eval(task.config().scriptBody(), context);
            task.logger().info("Completed");
        } catch (Exception e) {
            task.logger().info("Failed");
            logger.log(Level.WARNING, "Task failed " + task.config().taskId(), e);
            return LymphoWorkerTaskStatus.COMPLETED_ERROR;
        }

        return LymphoWorkerTaskStatus.COMPLETED_OK;
    }

    @Override
    public void run() {
        while (!stopped) {
            LymphoWorkerTask task = tasks.poll();
            if (task != null) {
                execute(task);
            }

            currentTask = null;
        }
    }

    @Override
    public void close() throws IOException {
        thread.interrupt();
    }
}
