package ru.yandex.lympho;

import java.io.IOException;
import java.net.URI;
import java.util.LinkedHashSet;
import java.util.Set;
import java.util.stream.Collectors;

import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.parser.JsonException;
import ru.yandex.json.writer.JsonValue;
import ru.yandex.json.writer.JsonWriterBase;
import ru.yandex.parser.string.CollectionParser;
import ru.yandex.parser.string.NonEmptyValidator;
import ru.yandex.parser.string.NonNegativeIntegerValidator;
import ru.yandex.parser.string.PositiveIntegerValidator;
import ru.yandex.parser.string.PositiveLongValidator;

public class TaskConfigBuilder implements JsonValue, TaskConfig {
    private final String taskId;
    private final String jobId;
    private final Set<Integer> backendShards;
    private final long createdTime;
    private final long deadline;
    private final String scriptBody;
    private final int heartbeatTimeout;
    private final int maxRetries;
    private final URI schedulerUri;

    public TaskConfigBuilder(final JsonMap source) throws JsonException {
        this.taskId = source.getString(TaskFields.TASK_ID.fieldName());
        this.jobId = source.getString(TaskFields.JOB_ID.fieldName());
        this.backendShards = source.get(
            TaskFields.BACKEND_SHARDS.fieldName(),
            new CollectionParser<>(
                NonNegativeIntegerValidator.INSTANCE,
                LinkedHashSet::new));
        this.createdTime =
            source.get(TaskFields.CREATED_TIME.fieldName(), PositiveLongValidator.INSTANCE);

        this.deadline =
            source.get(TaskFields.DEADLINE.fieldName(), PositiveLongValidator.INSTANCE);

        this.scriptBody =
            source.get(TaskFields.SCRIPT_BODY.fieldName(), NonEmptyValidator.TRIMMED);

        this.heartbeatTimeout =
            source.get(TaskFields.HEARTBEAT_TIMEOUT.fieldName(), PositiveIntegerValidator.INSTANCE);

        this.schedulerUri = source.getURI(TaskFields.SCHEDULER_URI.fieldName());
        this.maxRetries = source.getInt(TaskFields.MAX_RETRIES.fieldName(), 1);
    }

    protected void writeFields(final JsonWriterBase writer) throws IOException {
        writer.key(TaskFields.TASK_ID.fieldName());
        writer.value(taskId);
        writer.key(TaskFields.JOB_ID.fieldName());
        writer.value(jobId);
        writer.key(TaskFields.BACKEND_SHARDS.fieldName());
        writer.value(
            backendShards.stream().map(String::valueOf).collect(Collectors.joining(",")));
        writer.key(TaskFields.CREATED_TIME.fieldName());
        writer.value(createdTime);
        writer.key(TaskFields.DEADLINE.fieldName());
        writer.value(deadline);
        writer.key(TaskFields.SCRIPT_BODY.fieldName());
        writer.value(scriptBody);
        writer.key(TaskFields.HEARTBEAT_TIMEOUT.fieldName());
        writer.value(heartbeatTimeout);
        writer.key(TaskFields.SCHEDULER_URI.fieldName());
        writer.value(schedulerUri);
        writer.key(TaskFields.MAX_RETRIES.fieldName());
        writer.value(maxRetries);
    }

    @Override
    public void writeValue(final JsonWriterBase writer) throws IOException {
        writer.startObject();
        writeFields(writer);
        writer.endObject();
    }

    @Override
    public String taskId() {
        return taskId;
    }

    @Override
    public String jobId() {
        return jobId;
    }

    @Override
    public Set<Integer> backendShards() {
        return backendShards;
    }

    @Override
    public long createdTime() {
        return createdTime;
    }

    @Override
    public long deadline() {
        return deadline;
    }

    @Override
    public String scriptBody() {
        return scriptBody;
    }

    @Override
    public int heartbeatTimeout() {
        return heartbeatTimeout;
    }

    @Override
    public URI schedulerUri() {
        return schedulerUri;
    }

    @Override
    public int maxRetries() {
        return maxRetries;
    }
}
