package ru.yandex.lympho;

import java.io.IOException;
import java.util.Collections;
import java.util.LinkedHashSet;
import java.util.Set;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.stream.Collectors;

import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.writer.JsonWriterBase;
import ru.yandex.msearch.Config;
import ru.yandex.msearch.config.DatabaseConfig;
import ru.yandex.parser.string.CollectionParser;
import ru.yandex.parser.string.IntegerParser;
import ru.yandex.parser.string.NonNegativeIntegerValidator;
import ru.yandex.parser.string.PositiveLongValidator;
import ru.yandex.parser.string.Validator;

public class WorkerTaskConfig extends TaskConfigBuilder {
    private static final String WORKER_ADD_TS = "lympho_worker_add_ts";
    private static final String WORKER_RETRIES_COUNT = "lympho_task_retries_count";
    private static final String FINISHED_SHARDS = "lympho_worker_finished_shards";

    private final long workerAddTs;
    private Set<Integer> finishedShards;
    private final AtomicInteger retries;

    public WorkerTaskConfig(final JsonMap source, final DatabaseConfig config) throws Exception {
        super(source);

        this.finishedShards = source.get(
            FINISHED_SHARDS,
            Collections.emptySet(),
            new CollectionParser<>(
                NonNegativeIntegerValidator.INSTANCE,
                LinkedHashSet::new));

        this.workerAddTs = source.get(WORKER_ADD_TS, PositiveLongValidator.INSTANCE);
        this.retries = new AtomicInteger(source.getInt(WORKER_RETRIES_COUNT, 0));
        SearchBackendShardNumValidator validator
            = new SearchBackendShardNumValidator(config);

        for (Integer shard: backendShards()) {
            validator.validate(shard);
        }
    }

    @Override
    protected void writeFields(final JsonWriterBase writer)
        throws IOException
    {
        super.writeFields(writer);

        writer.key(WORKER_ADD_TS);
        writer.value(workerAddTs);

        writer.key(FINISHED_SHARDS);
        writer.value(
            finishedShards.stream().map(String::valueOf).collect(Collectors.joining(",")));

        writer.key(WORKER_RETRIES_COUNT);
        writer.value(retries.get());
    }

    public long workerAddTs() {
        return workerAddTs;
    }

    public Set<Integer> finishedShards() {
        return finishedShards;
    }

    public int retries() {
        return retries.get();
    }

    public int incRetries() {
        return retries.incrementAndGet();
    }

    public synchronized void finishShard(final Integer shard) {
        finishedShards.add(shard);
    }

    private static final class SearchBackendShardNumValidator extends Validator<Integer> {
        private final DatabaseConfig config;

        public SearchBackendShardNumValidator(final DatabaseConfig config) {
            super(IntegerParser.INSTANCE);

            this.config = config;
        }

        @Override
        public void validate(final Integer value) throws Exception {
            if (value < 0 || value >= config.shards()) {
                throw new IllegalArgumentException(
                    "Invalid shard number " + value + " not in range [0," + config.shards() + ')');
            }
        }
    }
}
