package ru.yandex.msearch;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStreamReader;

import java.nio.ByteBuffer;
import java.nio.channels.WritableByteChannel;
import java.nio.charset.Charset;
import java.nio.charset.CharsetDecoder;
import java.nio.charset.CodingErrorAction;
import java.nio.charset.StandardCharsets;

import java.text.ParseException;

import java.util.HashMap;
import java.util.Map;
import java.util.function.Function;

import ru.yandex.charset.Decoder;

import ru.yandex.http.util.YandexHeaders;

import ru.yandex.json.parser.ContentHandler;
import ru.yandex.json.parser.StringCollectorFactory;
import ru.yandex.json.parser.JsonException;
import ru.yandex.json.parser.JsonParser;

import ru.yandex.msearch.config.DatabaseConfig;
import ru.yandex.search.json.HandlersManager;

public abstract class AbstractJsonMessage implements JournalableMessage {
    private static final ThreadLocalDecoders DECODERS =
        new ThreadLocalDecoders();
    private static final String ZOO_QUEUE_ID_TO_CHECK =
        "\r\n" + YandexHeaders.ZOO_QUEUE_ID_TO_CHECK + ": ";
    private static final String ZOO_QUEUE_ID =
        "\r\n" + YandexHeaders.ZOO_QUEUE_ID + ": ";
    private static final String ZOO_SHARD_ID =
        "\r\n" + YandexHeaders.ZOO_SHARD_ID + ": ";
    private static final String ZOO_QUEUE =
        "\r\n" + YandexHeaders.ZOO_QUEUE + ": ";

    private final Charset charset;
    private final int priority;
    private final QueueShard queueShard;
    private final MessageQueueId queueId;
    protected final DatabaseConfig config;
    private final byte[] dump;
    private final byte[] header;
    protected final MessageContext context;

    public AbstractJsonMessage(
        final MessageContext context,
        final String uri,
        final byte[] dump,
        final Charset charset,
        final int priority,
        final QueueShard queueShard,
        final MessageQueueId queueId,
        final DatabaseConfig config,
        final boolean journalable)
    {
        this.context = context;
        this.charset = charset;
        this.priority = priority;
        this.queueShard = queueShard;
        this.queueId = queueId;
        this.config = config;
        if (journalable) {
            this.dump = dump;
        } else {
            this.dump = null;
        }
        if (journalable) {
            StringBuilder sb = new StringBuilder("POST ");
            sb.append(uri);
            sb.append(" HTTP/1.1\r\nContent-Length: ");
            sb.append(dump.length);
            if (queueId.weakCheck()) {
                sb.append(ZOO_QUEUE_ID_TO_CHECK);
            } else {
                sb.append(ZOO_QUEUE_ID);
            }
            sb.append(queueId.queueId());
            if (queueShard.shardId() != -1) {
                sb.append(ZOO_SHARD_ID);
                sb.append(queueShard.shardId());
                sb.append(ZOO_QUEUE);
                sb.append(queueShard.service());
            }
            sb.append("\r\nContent-Type: application/json; charset=");
            sb.append(charset);
            sb.append("\r\n\r\n");
            header = new String(sb).getBytes(StandardCharsets.UTF_8);
        } else {
            header = null;
        }
    }

    private static JsonParser createParser(
        final int expectedSize,
        final Function<? super HandlersManager, ? extends ContentHandler>
        contentHandlerFactory)
    {
        HandlersManager manager = new HandlersManager(
            StringCollectorFactory.INSTANCE.create(expectedSize));
        manager.push(contentHandlerFactory.apply(manager));
        return new JsonParser(manager);
    }

    protected void parse(
        final byte[] dump,
        final Function<? super HandlersManager, ? extends ContentHandler>
        contentHandlerFactory)
        throws IOException, ParseException
    {
        try {
            if (dump.length < config.inMemoryParsingLimit()) {
                Map<Charset, Decoder> decoders = DECODERS.get();
                Decoder decoder = decoders.get(charset);
                if (decoder == null) {
                    CharsetDecoder charsetDecoder = charset.newDecoder()
                        .onMalformedInput(CodingErrorAction.REPORT)
                        .onUnmappableCharacter(CodingErrorAction.REPORT);
                    decoder = new Decoder(charsetDecoder);
                    decoders.put(charset, decoder);
                }
                decoder.process(dump);
                JsonParser parser =
                    createParser(decoder.length(), contentHandlerFactory);
                decoder.processWith(parser);
                parser.eof();
            } else {
                createParser(dump.length, contentHandlerFactory).parse(
                    new InputStreamReader(
                        new ByteArrayInputStream(dump),
                        charset.newDecoder()
                            .onMalformedInput(CodingErrorAction.REPORT)
                            .onUnmappableCharacter(CodingErrorAction.REPORT)));
            }
        } catch (JsonException e) {
            ParseException ex = new ParseException(e.toString(), 0);
            ex.initCause(e);
            throw ex;
        }
    }

    @Override
    public Type type() {
        return Type.JSON;
    }

    @Override
    public int priority() {
        return priority;
    }

    @Override
    public MessageContext context() {
        return context;
    }

    @Override
    public MessageQueueId queueId() {
        return queueId;
    }

    @Override
    public QueueShard queueShard() {
        if (queueShard.shardId() == -1) {
            return new QueueShard(queueShard.service(), prefix());
        }
        return queueShard;
    }

    @Override
    public boolean journalable() {
        return dump != null;
    }

    @Override
    public int writeTo(final WritableByteChannel channel) throws IOException {
        ByteBuffer buf = ByteBuffer.wrap(header);
        while (buf.hasRemaining()) {
            channel.write(buf);
        }
        buf = ByteBuffer.wrap(dump);
        while (buf.hasRemaining()) {
            channel.write(buf);
        }
        int result = dump.length + header.length;
        return result;
    }

    private static class ThreadLocalDecoders
        extends ThreadLocal<Map<Charset, Decoder>>
    {
        @Override
        protected Map<Charset, Decoder> initialValue() {
            return new HashMap<>();
        }
    }
}
