package ru.yandex.msearch;

import java.io.IOException;

import java.text.ParseException;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.atomic.AtomicReference;
import java.util.concurrent.locks.ReadWriteLock;
import java.util.concurrent.locks.ReentrantReadWriteLock;

import java.util.logging.Level;
import java.util.logging.Logger;

import org.apache.lucene.analysis.Analyzer;
import org.apache.lucene.document.Document;
import org.apache.lucene.document.FieldSelector;
import org.apache.lucene.document.FieldSelectorResult;
import org.apache.lucene.document.Fieldable;
import org.apache.lucene.document.MapFieldSelector;
import org.apache.lucene.index.CorruptIndexException;
import org.apache.lucene.index.IndexReader;
import org.apache.lucene.index.IndexWriter;
import org.apache.lucene.index.LogByteSizeMergePolicy;
import org.apache.lucene.index.MergePolicy;
import org.apache.lucene.index.PerFieldMergingIndexReader;
import org.apache.lucene.index.Term;
import org.apache.lucene.index.codecs.CodecProvider;
import org.apache.lucene.search.Collector;
import org.apache.lucene.search.IndexSearcher;
import org.apache.lucene.search.Query;
import org.apache.lucene.search.QueryProducer;
import org.apache.lucene.search.TermQuery;
import org.apache.lucene.store.RAMDirectory;

import ru.yandex.collection.IntSet;

import ru.yandex.function.AbstractStringBuilderable;
import ru.yandex.function.StringBuilderable;

import ru.yandex.msearch.collector.BaseCollector;
import ru.yandex.msearch.collector.DocCollector;
import ru.yandex.msearch.collector.PlainCollector;
import ru.yandex.msearch.config.DatabaseConfig;
import ru.yandex.msearch.util.IOScheduler;

import ru.yandex.logger.PrefixedLogger;

import ru.yandex.search.prefix.Prefix;

public abstract class AbstractPart
    extends AbstractStringBuilderable
    implements MessageHandler
{
//    private static final boolean DEBUG = true;
    private static final boolean DEBUG = false;
//    protected static final int FLUSH_THRESHOLD = 150000;

    private final ConcurrentHashMap<QueueShard, QueueId> queueIds =
        new ConcurrentHashMap<>();
    private final ReadWriteLock partLock = new ReentrantReadWriteLock();
    private final Object reopenSearcherLock = new Object();
    private final Object getSearcherLock = new Object();
    private final AtomicLong indexGeneration = new AtomicLong(0);
    private final AtomicLong searcherGeneration = new AtomicLong(0);
    private final AtomicInteger indexedDocs = new AtomicInteger(0);
    private final AtomicInteger deletedDocs = new AtomicInteger(0);
    private final AtomicInteger refs = new AtomicInteger(0);
    private final AnalyzerProvider analyzerProvider;
    private final IndexManager indexManager;
    private final DeleteAccessorProxy indexAccessor;
    private final long version;
    private final RAMDirectory dir;
    private final IndexWriter writer;
    private volatile boolean closed = false;
    private volatile boolean dirty = false;
    public volatile boolean preFlushed = false;
    public volatile boolean setPreFlush = false;
    private volatile boolean flushed = false;
    private volatile int uses = 0;
    private boolean needFlush = false;
    private boolean deleted = false;
    private PartSearcher currentSearcher = null;
    private AtomicReference<PartSearcher> flushSearcher = new AtomicReference<PartSearcher>(null);
    private LinkedList<QueryProducer> pendingDeletes = null;
//    private LinkedList<Query> deletes = new LinkedList<Query>();
    private Journal journal;
    private int commitCount;
    protected final DatabaseConfig config;
    private int[] currentIndexingDocs = new int[1];
    protected final PrefixedLogger logger;
    protected final Logger indexLogger;
    protected final int flushThreshold;

    static final class PartStats {
        long dirSize;
        long writerSize;
        int numDocs;
        int maxDoc;
        int bufferedDeletes;
        long payloadSize;
    }

    static final class PartsStats {
        private PartStats[] stats;

        public PartsStats(PartStats[] stats) {
            this.stats = stats;
        }

        @Override
        public String toString() {
            StringBuilder sb = new StringBuilder();
            String sep = "";
            for(PartStats stat : stats) {
                sb.append(sep);
                sb.append("[ ");
                sb.append("\tdirSize: " + stat.dirSize);
                sb.append(", writerSize: " + stat.writerSize);
                sb.append(", numDocs: " + stat.numDocs);
                sb.append(", maxDoc: " + stat.maxDoc);
                sb.append(", pendingDeletes: " + stat.bufferedDeletes);
                sb.append(", payloadSize: " + stat.payloadSize);
                sb.append(" ]");
                sep = "\n";
            }
            return sb.toString();
        }
    }

    private static final class DeleteAccessorProxy implements IndexAccessor {
        private final IndexAccessor accessor;
        private ArrayList<QueryProducer> deletes = null;
        private long deletesWeight = 0;

        public DeleteAccessorProxy(final IndexAccessor accessor) {
            this.accessor = accessor;
        }

        public synchronized QueryProducer[] getDeletes() {
            if(deletes == null) return null;
            return deletes.toArray(new QueryProducer[deletes.size()]);
        }

        public synchronized int deletesCount() {
            if(deletes == null) return 0;
            return deletes.size();
        }

        public synchronized long deletesWeight() {
            return deletesWeight;
        }

        public void deleteDocuments(QueryProducer query) throws IOException {
            if (query == null) {
                throw new CorruptIndexException("AbstractPart.deleteDocuments: query = null");
            }
            long weight = query.toString().length() << 3;
            synchronized (this) {
                if( deletes == null ) {
                    deletes = new ArrayList<QueryProducer>();
                }
                deletes.add(query);
                deletesWeight += weight;
            }
            accessor.deleteDocuments(query);
        }

        public void updateQueueIds(final Map<QueueShard, QueueId> ids) {
            accessor.updateQueueIds(ids);
        }

        public long queueId(final QueueShard shard) {
            return accessor.queueId(shard);
        }

        public CloseableSearcher searcher() throws IOException {
            return accessor.searcher();
        }

        public void primaryKeyAtomicOp(PrimaryKey key, AtomicOp op) throws IOException {
            accessor.primaryKeyAtomicOp(key, op);
        }

        public void prefixReadAtomicOp(Prefix prefix, PrefixOp op) throws IOException, ParseException {
            accessor.prefixReadAtomicOp(prefix, op);
        }

        public void prefixWriteAtomicOp(Prefix prefix, PrefixOp op) throws IOException, ParseException {
            accessor.prefixWriteAtomicOp(prefix, op);
        }

        public void dirtifyQueueShard(
            final QueueShard shard,
            final long queueId)
        {
            accessor.dirtifyQueueShard(shard, queueId);
        }

        @Override
        public PrefixedLogger logger() {
            return accessor.logger();
        }

        @Override
        public Logger indexLogger() {
            return accessor.indexLogger();
        }
    }

    public AbstractPart(
        final AnalyzerProvider analyzerProvider,
        final IndexManager indexManager,
        final IndexAccessor indexAccessor,
        final Map<QueueShard, QueueId> prevQueueIds,
        final long version,
        final DatabaseConfig config)
        throws IOException
    {
        this.analyzerProvider = analyzerProvider;
        this.indexManager = indexManager;
        this.indexAccessor = new DeleteAccessorProxy(indexAccessor);
        this.version = version;
        this.config = config;
        this.logger = indexAccessor.logger();
        this.indexLogger = indexAccessor.indexLogger();
        this.flushThreshold = config.partFlushThreshold();
        dir = new RAMDirectory();
        writer = indexManager.createWriter(dir);
        journal = indexManager.createJournal();
        reopenReader(0);
        commitCount = 0;
        if (prevQueueIds != null) {
            for (final Map.Entry<QueueShard, QueueId> entry :
                prevQueueIds.entrySet())
            {
                final QueueId value = entry.getValue();
                if (value.dirty()) {
                    queueIds.put(
                        entry.getKey(),
                        new QueueId(value));
                }
            }
        }
    }

    protected abstract void cleanup();

    protected AnalyzerProvider analyzerProvider() {
        return analyzerProvider;
    }

    protected void traverseParts(PartVisitor visitor) throws IOException {
        indexManager.traverseParts(visitor, version);
    }

    protected IndexAccessor indexAccessor() {
        return indexAccessor;
    }

    protected IndexManager indexManager() {
        return indexManager;
    }

    public long version()
    {
        return version;
    }

    public void setDirty( boolean dirty )
    {
        this.dirty = dirty;
    }

    public boolean dirty()
    {
        return dirty;
    }

    public Logger logger() {
        return logger;
    }

    //return journal size
    public long journalMessage(final JournalableMessage message)
        throws IOException
    {
        long size = 0;
        dirty = true;
        if (!message.queueId().magic()) {
            final QueueShard shard = message.queueShard();
            QueueId queueId = queueIds.get(shard);
            if (queueId == null) {
                QueueId newId = new QueueId(indexAccessor.queueId(shard));
                queueId = queueIds.putIfAbsent(shard, newId);
                if (queueId == null) {
                    queueId = newId;
                }
            }
            synchronized(queueId) {
                final long messageId = message.queueId().queueId();
                final boolean weak = message.queueId().weakCheck();
                if (queueId.get() < messageId) {
                    if (journal != null) {
                        size = journal.journalMessage(message);
                    }
                    if (!weak) {
                        indexAccessor.dirtifyQueueShard(shard, messageId);
                        queueId.setId(messageId);
                    }
                } else {
                    if (!weak) {
                        indexAccessor.dirtifyQueueShard(shard, messageId);
                    }
                    throw new QueueIdOutOfOrderException("QueueId for shard<"
                        + shard + "> is less than or equals current queueId: "
                        + messageId + " < " + queueId.get()
                        + ", accessorQueueId: " + indexAccessor.queueId(shard));
                }
            }
        } else {
            if (journal != null) {
                size = journal.journalMessage(message);
            }
        }
        return size;
    }

    public synchronized void closeJournal() throws IOException {
        if (journal != null) {
            journal.close();
            journal = null;
        }
        cleanup();
    }

    public synchronized void incUses()
    {
        uses++;
    }

    public synchronized void decUses()
    {
        uses--;
        if (uses == 0 && setPreFlush) {
            preFlushed = true;
        }
        notifyAll();
    }

    public synchronized void waitFree()
    {
        while( uses > 0 )
        {
            try
            {
                wait();
            }
            catch( InterruptedException e )
            {
            }
        }
    }

    public void incRef()
    {
        refs.incrementAndGet();
    }

    private int decRef()
    {
        return refs.decrementAndGet();
    }

    public int refs()
    {
        return refs.get();
    }

    public void preFlush() {
        setPreFlush = true;
        if (partLock.writeLock().tryLock()) {
            preFlushed = true;
            partLock.writeLock().unlock();
        }
    }

    protected Document filterDocument(
        final Document document,
        final FieldSelector selector)
    {
        Document filtered = new Document(selector.maxFieldCount());
        for (Fieldable f: document.getFields()) {
            FieldSelectorResult result = selector.accept(f.name());
            if (result != null && result != FieldSelectorResult.NO_LOAD) {
                filtered.add(f);
            }
        }

        return filtered;
    }

    protected Document filterDocument(
        final Document document,
        final Collection<String> fields)
    {
        if (fields != null) {
            Map<String, FieldSelectorResult> selections = new HashMap<>();
            for (String field: fields) {
                selections.put(field, FieldSelectorResult.LOAD);
            }

            return filterDocument(document, new MapFieldSelector(selections));
        }

        return document;
    }

    public PartSearcher getSearcher() throws IOException
    {
        partLock.readLock().lock();
        try {
            long indexGen = indexGeneration.get();
            long searchGen = searcherGeneration.get();
            if( currentSearcher == null || indexGen > searchGen )
            {
                commit(indexGen);
            }
            synchronized(getSearcherLock)
            {
                PartSearcher retval = currentSearcher;
                retval.incRef();
                return retval;
            }
        } finally {
            partLock.readLock().unlock();
        }
    }

    private void reopenReader(long indexGen) throws IOException {
//        System.err.println("\t" + this + " reopenReader: indexGen=" + indexGen + ", searchGen=" + searcherGeneration.get() + ", indexedDocs=" + indexedDocs.get());
        synchronized(reopenSearcherLock)
        {
            if( indexGen <= searcherGeneration.get() && currentSearcher != null  ) return;

            IndexReader reader;
            if (currentSearcher == null) {
                //first time create empty reader;
                reader = writer.getReaderMem();
                ((PerFieldMergingIndexReader)reader).setParent(this);
            } else {
                reader = ((PerFieldMergingIndexReader)currentSearcher.reader()).clone(indexedDocs.get()-1);
//                System.err.println("\t" + this + " -> " + System.identityHashCode(reader) + " reopenReader: lastDocId=" + ((PerFieldMergingIndexReader)reader).lastDocId);
            }

            //IndexReader.open(writer, true);
            if( currentSearcher != null && currentSearcher.reader() != reader || currentSearcher == null )
            {
            PartSearcher oldSearcher = currentSearcher;
                synchronized(getSearcherLock)
                {
                    currentSearcher = new PartSearcher( new IndexSearcher(reader), reader, this );
                    currentSearcher.incRef();
                    if( oldSearcher != null )
                    {
                        ((PerFieldMergingIndexReader)oldSearcher.reader()).decDeletersRefs();
                        oldSearcher.free();
                    }
                }
            }
            searcherGeneration.set(indexGen);
        }
    }

    // TODO: remove code duplication
    public void search(final Query query, final Collector col)
        throws IOException
    {
        Searcher searcher = getSearcher();
        if (searcher == null) {
            return;
        }
        try
        {
            searcher.searcher().search(query, col, true);
        }
        finally
        {
            searcher.free();
        }
    }

    public Document[] search(final Query query, final int n)
        throws IOException
    {
        Searcher searcher = getSearcher();
        if (searcher == null) return new Document[0];
        try
        {
            PlainCollector col = new PlainCollector(n);
            searcher.searcher().search(query, col, true);
            col.flush();
            List<Document> docs = col.docs();
            return docs.toArray(new Document[docs.size()]);
        }
        finally
        {
            searcher.free();
        }
    }

    public Document[] fastSearch(final TermQuery query)
        throws IOException
    {
        final Term term = query.getTerm();
        Searcher searcher = getSearcher();
        if (searcher == null) return new Document[0];
        try {
            PerFieldMergingIndexReader reader =
                (PerFieldMergingIndexReader)searcher.reader();
            return reader.getDocsForTerm(term);
        } finally {
            searcher.free();
        }
    }

    public Document document(final int docId) throws IOException {
        Searcher searcher = getSearcher();
        if (searcher == null) return null;
        try {
            PerFieldMergingIndexReader reader =
                (PerFieldMergingIndexReader)searcher.reader();
            return reader.document(docId);
        } finally {
            searcher.free();
        }
    }

    public void commit(long indexGen) throws IOException {
        if (indexGen <= searcherGeneration.get()) {
            return;
        }
        reopenReader(indexGen);
    }

    public void freeSearcher(Searcher searcher) throws IOException {
        int searcherRefs = searcher.decRef();
        if (searcherRefs == 0) {
            searcher.close();
        }
        free();
    }

    public void free() throws IOException {
        if (decRef() == 0) {
            delete();
        }
    }

    public void close() throws IOException {
        if (logger.isLoggable(Level.INFO)) {
            StringBuilder sb = new StringBuilder();
            toStringBuilder(sb);
            sb.append(" AbstractPart.close");
            logger.info(new String(sb));
        }
//        writer.commit();
//        writer.rollback();
//        writer.close();
        closed = true;
    }

    public boolean needFlush() {
        return needFlush;
    }

    //This method will write only data to the underlying writer's directory (the most time expensive task).
    //You must then add returned SegmentInfos to the index writer manualy to complete transaction.
    public List<org.apache.lucene.index.SegmentInfo> asyncWriteTo(IndexWriter dest) throws IOException {
        IOScheduler.setThreadReadPrio(IOScheduler.IOPRIO_INDEXING);
        IOScheduler.setThreadWritePrio(IOScheduler.IOPRIO_INDEXING);
        flushed = true;
        partLock.readLock().lock();
        try {
            if (closed) return null;
        } finally {
            partLock.readLock().unlock();
        }

        synchronized (this) {
            while (uses > 0) { //Wait for all handlers to complete as they may call IndexDocument.
                try {
                    wait();
                } catch (InterruptedException ign) {
                    return null;
                }
            }
        }
//        System.err.println(this + " AbstractPart.asyncWriteTo: deletes.size() = " + deletes.size());

        RAMDirectory tmpDir = new RAMDirectory();
        IndexWriter tmpWriter = indexManager.createWriter(tmpDir);
        try {
            partLock.writeLock().lock();
            if (flushSearcher.get() == null) {
                flushSearcher.set(getSearcher());
            }
        } finally {
            partLock.writeLock().unlock();
        }
        if (flushSearcher.get().reader().numDocs() > 0) {
            tmpWriter.addIndexes(flushSearcher.get().reader());
        }
        tmpWriter.commit();
        tmpWriter.close();

        QueryProducer[] deletes = indexAccessor.getDeletes();
        if (deletes != null) {
            dest.deleteDocuments(deletes);
        }
        List<org.apache.lucene.index.SegmentInfo> ret = null;
        try {
            ret = dest.addIndexesData(tmpDir);
            closed = true;
            return ret;
        } finally {
            if (closed) {
                flushSearcher.get().free();
                flushSearcher.set(null);
            }
            tmpDir.close();
        }
    }

    public void atomicWriteTo(IndexWriter dest) throws IOException {
        if (closed) return;
        if (logger.isLoggable(Level.INFO)) {
            StringBuilder sb = new StringBuilder();
            toStringBuilder(sb);
            sb.append(" AbstractPart.atomicWriteTo");
            logger.info(new String(sb));
        }
        reopenReader(indexGeneration.incrementAndGet());

        RAMDirectory tmpDir = new RAMDirectory();
        IndexWriter tmpWriter = indexManager.createWriter(tmpDir);
        if (currentSearcher.reader().numDocs() > 0) {
            tmpWriter.addIndexes(currentSearcher.reader());
        }
        tmpWriter.commit();
        tmpWriter.close();


        QueryProducer[] deletes = indexAccessor.getDeletes();
        if (deletes != null) {
            dest.deleteDocuments(deletes);
        }
        closed = true;
        try {
            dest.addIndexes(tmpDir);
        } finally {
            tmpDir.close();
        }
    }

    public void delete() throws IOException {
        partLock.writeLock().lock();
        try {
            closeJournal();
            if (currentSearcher != null) currentSearcher.close();
            writer.rollback();
            writer.close();
            if(!closed) {
                throw new IOException( "AbstractPart deleted whithout closing" );
            }
        } finally {
            dir.close();
            deleted = true;
            partLock.writeLock().unlock();
        }
    }

    public PartStats stats() throws IOException {
        PartStats stats = new PartStats();
        stats.dirSize = dir.sizeInBytes();
        partLock.readLock().lock();
        try {
            if (!closed) {
                stats.writerSize = writer.ramSizeInBytes();
                stats.numDocs = writer.numDocs();
                stats.maxDoc = writer.maxDoc();
                stats.bufferedDeletes = indexAccessor.deletesCount();
                stats.payloadSize = payloadSize();
            }
        } finally {
            partLock.readLock().unlock();
        }
        return stats;
    }

    public long size() {
        long size = 0L;
        partLock.readLock().lock();
        try {
            if (!closed) {
                size = dir.sizeInBytes() + writer.ramSizeInBytes();
            }
        } finally {
            partLock.readLock().unlock();
        }
        return size + payloadSize();
    }

    public long payloadSize() {
        return indexAccessor.deletesWeight();
    }

    protected int indexDocument(
            final HTMLDocument doc,
            final Analyzer analyzer)
            throws IOException
    {
        partLock.readLock().lock();
        try {
            if (preFlushed && uses == 0) {
                throw new IOException( "Trying do add a document to an already closed part" );
            }
            if (indexLogger.isLoggable(Level.FINER)) {
                StringBuilder sb = new StringBuilder();
                toStringBuilder(sb);
                sb.append(" Indexing: ");
                StringBuilderable.toStringBuilder(sb, doc.primaryKey());
                indexLogger.finer(new String(sb));
            } else if (indexLogger.isLoggable(Level.FINE)) {
                StringBuilder sb = new StringBuilder(" Indexing: ");
                StringBuilderable.toStringBuilder(sb, doc.primaryKey());
                indexLogger.fine(new String(sb));
            }
            int docId = writer.addDocument(doc.getDoc(), analyzer);
            indexedDocs.incrementAndGet();
            if (indexLogger.isLoggable(Level.FINER)) {
                StringBuilder sb = new StringBuilder();
                toStringBuilder(sb);
                sb.append(" Indexed: ");
                StringBuilderable.toStringBuilder(sb, doc.primaryKey());
                sb.append(", indexedDocs=");
                sb.append(indexedDocs.get());
                sb.append(", docId=");
                sb.append(docId);
                indexLogger.finer(new String(sb));
            }
            long indexGen = indexGeneration.incrementAndGet();
            if (indexedDocs.get() > flushThreshold) {
                needFlush = true;
            }
//            if (((PerFieldMergingIndexReader)currentSearcher.reader()).needFlush && !needFlush) {
//                System.err.println("NEED FLUSH: docs=" + indexedDocs.get() + ", deletes=" + deletedDocs.get());
//                needFlush = true;
//            }
            return docId;
        } finally {
            partLock.readLock().unlock();
        }
    }

    public void deleteDocuments(final QueryProducer query) throws IOException {
        deleteDocuments(query, -1);
    }

    public void deleteDocuments(final QueryProducer query, final int docId)
        throws IOException
    {
        if (query == null) {
            throw new CorruptIndexException("AbstractPart.deleteDocuments: query = null");
        }
        if (indexLogger.isLoggable(Level.FINER)) {
            StringBuilder sb = new StringBuilder();
            toStringBuilder(sb);
            sb.append(" Deleting: ");
            sb.append(query);
            indexLogger.finer(new String(sb));
        } else if (indexLogger.isLoggable(Level.FINE)) {
            indexLogger.fine("Deleting: " + query);
        }
        partLock.readLock().lock();
        try {
            if (indexGeneration.get() > searcherGeneration.get()) {
                reopenReader(indexGeneration.get());
            }
            synchronized(reopenSearcherLock) {
                if (preFlushed && uses == 0) {
                    if (flushSearcher.get() == null) {
                        flushSearcher.compareAndSet(null, getSearcher());
                    }
                    if (pendingDeletes == null) pendingDeletes = new LinkedList<>();
                    pendingDeletes.add(query);
                } else {
                    deletedDocs.incrementAndGet();
                }
                if (indexGeneration.get() == 0) {
                    //no documents was indexed, so no deletion is posible
                    return;
                }
            }
            PartSearcher searcher;
            PerFieldMergingIndexReader reader;
            synchronized(getSearcherLock)
            {
                searcher = currentSearcher;
                searcher.incRef();
                reader = (PerFieldMergingIndexReader)searcher.reader();
                reader.incDeletersRefs();
            }
//            try {
//                Thread.sleep((int)(Math.random() * 5.));
//            } catch (InterruptedException ign) {
//            }
            int delcount = -1;
            try {
                if (docId != -1) {
                    reader.deleteDocument(docId);
                    delcount = 1;
                } else {
                    delcount = reader.deleteDocuments(query.produceQuery());
                }
            } finally {
                if (searcher != null) {
                    reader.decDeletersRefs();
                    searcher.free();
                }
            }
            indexGeneration.incrementAndGet();
//            }
            if (indexLogger.isLoggable(Level.FINER)) {
                StringBuilder sb = new StringBuilder();
                toStringBuilder(sb);
                sb.append(" Deleted: ");
                sb.append(query);
                sb.append(" , delcount=");
                sb.append(delcount);
                indexLogger.finer(new String(sb));
            }
        } finally {
            partLock.readLock().unlock();
        }
    }

    public List<QueryProducer> getPendingDeletes() {
        return pendingDeletes;
    }

    public QueryProducer[] getDeletes() {
        return indexAccessor.getDeletes();
    }

    public Map<QueueShard, QueueId> getQueueIds() {
        return queueIds;
    }

    public int numDocs() throws IOException {
        return writer.numDocs();
    }

    public long numDocsLong() throws IOException {
        return (long) numDocs();
    }

    public boolean isClosed()
    {
        return closed;
    }

    @Override
    public void reopen() {
        throw new UnsupportedOperationException(
            "Shard doesn't support reopen operation");
    }

    @Override
    public void reopen(
        final IntSet shards,
        final Boolean ro,
        final boolean doWait)
        throws IOException
    {
        throw new UnsupportedOperationException(
            "Shard doesn't support reopen operation");
    }

    @Override
    public void toStringBuilder(final StringBuilder sb) {
        sb.append("\tAP:");
        sb.append(System.identityHashCode(this));
        sb.append(", i=");
        sb.append(indexedDocs.get());
        sb.append(", d=");
        sb.append(deletedDocs.get());
        sb.append(", NF=");
        sb.append(needFlush);
    }

    @Override
    public void finalize() {
        if (!deleted) {
            System.err.println( "Finalizer: AbstractPart was not deleted and is no more referenced by anybody. Refs = " + refs.get() + " / currentSearcher.refs() = " + (currentSearcher == null ? 0 : currentSearcher.refs()) + " / this: " + currentSearcher );
            try {
                delete();
            } catch (IOException e) {
            }
        }
    }
}
