package ru.yandex.msearch;

import java.io.IOException;

import java.util.Map;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.locks.ReentrantReadWriteLock;

import org.apache.lucene.analysis.Analyzer;
import org.apache.lucene.document.Document;
import org.apache.lucene.index.CorruptIndexException;
//import org.apache.lucene.queryParser.ParseException;

import ru.yandex.msearch.config.DatabaseConfig;
import ru.yandex.msearch.util.AtomicBitsArray;

public class BloomPrimaryKeyPart extends PrimaryKeyPartBase {
    private static final int INITIAL_BITS = 2;
    private AtomicBitsArray primaryKeysBitSet;
    private final ReentrantReadWriteLock filterLock =
        new ReentrantReadWriteLock();
    private final AtomicInteger cacheHit = new AtomicInteger(0);
    private final AtomicInteger cacheMiss = new AtomicInteger(0);
    private final AtomicInteger checkCount = new AtomicInteger(1000);

    public BloomPrimaryKeyPart(
        final AnalyzerProvider analyzerProvider,
        final IndexManager indexManager,
        final IndexAccessor indexAccessor,
        final Map<QueueShard, QueueId> queueIds,
        final long version,
        final DatabaseConfig config)
        throws IOException
    {
        super(
            analyzerProvider,
            indexManager,
            indexAccessor,
            queueIds,
            version,
            config);
        primaryKeysBitSet =
            new AtomicBitsArray(flushThreshold << 3, INITIAL_BITS);
    }

    @Override
    protected void cleanup() {
        primaryKeysBitSet = null;
    }

    @Override
    public int indexDocument(final HTMLDocument doc,
        final Analyzer analyzer)
        throws IOException
    {
        int docId = super.indexDocument(doc, analyzer);
        filterPut(doc.primaryKey().cacheKey().hashCode());
        return docId;
    }

    private static final int hash(int h) {
         h ^= (h >>> 20) ^ (h >>> 12);
         return h ^ (h >>> 7) ^ (h >>> 4);
    }

    private final int posForHash(final int hashCode) {
        return Math.abs(hash(hashCode) % primaryKeysBitSet.length());
    }

    private final void filterPut(final int hashCode) {
        filterLock.readLock().lock();
        try {
            final int bitsPos = posForHash(hashCode);
            while (true) {
                int oldValue = primaryKeysBitSet.get(bitsPos);
                if (oldValue == primaryKeysBitSet.maxValuePerItem()) {
                    filterLock.readLock().unlock();
                    try {
                        growFilterBitness();
                    } finally {
                        filterLock.readLock().lock();
                    }
                } else {
                    final int newValue = oldValue + 1;
                    if (primaryKeysBitSet.compareAndSet(bitsPos, oldValue,
                        newValue))
                    {
                        return;
                    }
                }
            }
        } finally {
            filterLock.readLock().unlock();
        }
    }

    private final boolean filterRemove(final int hashCode) {
        filterLock.readLock().lock();
        try {
            final int bitsPos = posForHash(hashCode);
            while (true) {
                int oldValue = primaryKeysBitSet.get(bitsPos);
                if (oldValue == 0) {
                    return false;
                }
                final int newValue = oldValue - 1;
                if (primaryKeysBitSet.compareAndSet(bitsPos, oldValue,
                    newValue))
                {
                    return true;
                }
            }
        } finally {
            filterLock.readLock().unlock();
        }
    }

    private final void growFilterBitness() {
        filterLock.writeLock().lock();
        try {
            AtomicBitsArray newBitSet =
                new AtomicBitsArray(
                    primaryKeysBitSet.length(),
                    primaryKeysBitSet.numBits() << 1);
            for (int i = 0; i < primaryKeysBitSet.length(); i++) {
                newBitSet.set(i, primaryKeysBitSet.get(i));
            }
            primaryKeysBitSet = newBitSet;
        } finally {
            filterLock.writeLock().unlock();
        }
    }

    private final boolean filterContains(final int hashCode) {
        filterLock.readLock().lock();
        try {
            final int bitsPos = posForHash(hashCode);
            return primaryKeysBitSet.get(bitsPos) > 0;
        } finally {
            filterLock.readLock().unlock();
        }
    }

    @Override
    public boolean hasPrimaryKey(final PrimaryKey primaryKey,
        final PrimaryKeySearcher primaryKeySearcher)
        throws IOException
    {
        if (checkCount.decrementAndGet() == 0) {
            checkCount.set(10000);
        }
        if (filterContains(primaryKey.cacheKey().hashCode())) {
            Document[] docs = search(primaryKey.query(), 2);
            if (docs.length > 1) {
                throw new CorruptIndexException(
                    "Wrong document count for key: " + primaryKey
                    + ", expected 1, got:" + docs.length);
            }
            if (docs.length > 0) {
                cacheHit.incrementAndGet();
            } else {
                cacheMiss.incrementAndGet();
            }
            if (primaryKeySearcher != null && docs.length > 0) {
                primaryKeySearcher.doc(docs[0]);
            }
            return docs.length > 0;
        }
        cacheHit.incrementAndGet();
        return false;
    }

    @Override
    public Document getDocumentByPrimaryKey(final PrimaryKey primaryKey)
        throws IOException
    {
        Document[] docs = search(primaryKey.query(), 2);
        if (docs.length != 1) {
            throw new CorruptIndexException(
                "Wrong document count for key: " + primaryKey
                + ", expected 1, got:" + docs.length);
        }
        return docs[0];
    }

    @Override
    public void removePrimaryKey(final PrimaryKey primaryKey) {
        filterRemove(primaryKey.cacheKey().hashCode());
    }

    @Override
    public boolean deleteDocument(final PrimaryKey primaryKey)
        throws IOException
    {
        if (!filterRemove(primaryKey.cacheKey().hashCode())) {
            return false;
        } else {
            deleteDocuments(primaryKey.queryProducer());
            return true;
        }
    }

    @Override
    public long payloadSize() {
        filterLock.readLock().lock();
        try {
            return super.payloadSize() + primaryKeysBitSet.sizeInBytes();
        } finally {
            filterLock.readLock().unlock();
        }
    }
}

