package ru.yandex.msearch;

import java.io.IOException;

import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.atomic.AtomicLong;

import org.apache.lucene.analysis.Analyzer;
import org.apache.lucene.document.Document;
import org.apache.lucene.document.Fieldable;
import org.apache.lucene.index.CorruptIndexException;

import ru.yandex.msearch.config.DatabaseConfig;
//import org.apache.lucene.queryParser.ParseException;


public class CachingPrimaryKeyPart extends PrimaryKeyPartBase {
    private final ConcurrentHashMap<PrimaryKey.CacheKey, Document> primaryKeys =
        new ConcurrentHashMap<>();
    private final AtomicLong docsStorageSize = new AtomicLong(0);

    public CachingPrimaryKeyPart(
        final AnalyzerProvider analyzerProvider,
        final IndexManager indexManager,
        final IndexAccessor indexAccessor,
        final Map<QueueShard, QueueId> queueIds,
        final long version,
        final DatabaseConfig config)
        throws IOException
    {
        super(
            analyzerProvider,
            indexManager,
            indexAccessor,
            queueIds,
            version,
            config);
    }

    @Override
    protected void cleanup() {
        primaryKeys.clear();
    }

    @Override
    public int indexDocument(final HTMLDocument doc,
        final Analyzer analyzer)
        throws IOException
    {
        int docId = super.indexDocument(doc, analyzer);
        Document cached = doc.getDoc();
        docsStorageSize.addAndGet(getDocumentSize(cached, doc.primaryKey()));
        if (primaryKeys.putIfAbsent(doc.primaryKey().cacheKey(),
            cached) != null)
        {
            throw new CorruptIndexException(
                "Document for key: " + doc.primaryKey()
                + " is already in the part");
        }
        return docId;
    }

    @Override
    public boolean deleteDocument(final PrimaryKey primaryKey)
        throws IOException
    {
        Document doc = primaryKeys.remove(primaryKey.cacheKey());
        if (doc == null) {
            return false;
        } else {
            deleteDocuments(primaryKey.queryProducer());
            return true;
        }
    }

    public boolean hasPrimaryKey(final PrimaryKey primaryKey,
        final PrimaryKeySearcher primaryKeySearcher) {
        Document doc = primaryKeys.get(primaryKey.cacheKey());
        primaryKeySearcher.doc(doc);
        return doc != null;
//        return primaryKeys.containsKey(primaryKey.cacheKey());
    }

    public Document getDocumentByPrimaryKey(final PrimaryKey primaryKey) {
        return primaryKeys.get(primaryKey.cacheKey());
    }

    public void removePrimaryKey(final PrimaryKey primaryKey) {
        Document doc = primaryKeys.remove(primaryKey.cacheKey());
        if (doc != null) {
            docsStorageSize.addAndGet(-getDocumentSize(doc, primaryKey));
        }
    }

    private long getDocumentSize(final Document doc, final PrimaryKey key) {
        long size = 0;
        //doc size
        for (Fieldable f : doc.getFields()) {
            size += f.stringValue().length() * 2 + 100;
            size += f.name().length() * 2 + 100;
        }
        //key size
        size += key.cacheKey().weight();
        //HashNode size
        size += 48;
//        System.err.println("gds: " + size);
        return size;
    }

}
