package ru.yandex.msearch;

/**
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

import org.apache.lucene.document.Document;
import org.apache.lucene.index.IndexReader;
import org.apache.lucene.index.MultiFields;
import org.apache.lucene.index.codecs.CodecProvider;
import org.apache.lucene.index.codecs.fast_commit.FastCommitCodec;
import org.apache.lucene.index.codecs.yandex.YandexCodec;
import org.apache.lucene.index.codecs.standard.StandardCodec;
import org.apache.lucene.index.Fields;
import org.apache.lucene.index.FieldsEnum;
import org.apache.lucene.index.MultiFields;
import org.apache.lucene.index.Term;
import org.apache.lucene.index.Terms;
import org.apache.lucene.index.TermsEnum;
import org.apache.lucene.index.DocsEnum;
import org.apache.lucene.index.DocsAndPositionsEnum;
import org.apache.lucene.search.IndexSearcher;
import org.apache.lucene.search.TermQuery;
import org.apache.lucene.document.MapFieldSelector;
import org.apache.lucene.document.FieldSelector;
import org.apache.lucene.store.NIOFSDirectory;
import org.apache.lucene.util.BytesRef;
import org.apache.lucene.util.Bits;
import org.apache.lucene.search.DocIdSetIterator;

import java.io.File;
import java.io.IOException;
import java.util.Arrays;
import java.util.Collections;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Set;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.Callable;
import java.util.concurrent.Future;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;

public class CheckIndex {

  private CheckIndex() {}

  private static boolean skipErrors = false;

  public static void main(String[] args) throws Exception {
    String usage =
      "Usage: java ru.yandex.msearch.CheckIndex dir shards_count [threads (default 1)]";
    if (args.length > 0 && ("-h".equals(args[0]) || "-help".equals(args[0])) || args.length < 2) {
      System.out.println(usage);
      System.exit(0);
    }

    String index = args[0];
    int shardsCount = Integer.parseInt( args[1] );
    int threads = 1;
    if( args.length > 2 )
    {
        threads = Integer.parseInt( args[2] );
    }

    ThreadPoolExecutor pool = null;

    CodecProvider cp = CodecProvider.getDefault();
    cp.register( new YandexCodec() );
    Set<String> blooms = Collections.emptySet();
    cp.register( new FastCommitCodec(blooms) );
//    cp.register( new StandardCodec() );
    cp.setDefaultFieldCodec( "Standard" );

    boolean hasErrors = false;

    if( shardsCount == 0 )
    {
	skipErrors = true;
	try
	{
	    checkShard( index );
	    System.out.println( "Ok" );
	}
	catch( Exception e )
	{
	    System.out.println("Fail: " + e.getMessage() );
	    e.printStackTrace();
	    System.exit(-1);
	}
    }
    else
    {
        pool = new ThreadPoolExecutor( threads, threads, 1, TimeUnit.SECONDS, new ArrayBlockingQueue<Runnable>(shardsCount) );
        HashMap<Future,Integer> shardsMap = new HashMap<Future,Integer>();
        LinkedList<Future> futures = new LinkedList<Future>();
        for( int i = 0; i < shardsCount; i++ )
        {
            Future future = pool.submit(new CheckShardTask( index + "/" + i ));
            futures.add( future );
            shardsMap.put( future, i );
        }
        while( futures.size() > 0 )
        {
            Iterator<Future> iter = futures.iterator();
            while( iter.hasNext() )
            {
                Future future = iter.next();
	        try
	        {
                    try
                    {
                        future.get( 10, TimeUnit.MILLISECONDS );
                        iter.remove();
                    }
                    catch( TimeoutException ign )
                    {
                        continue;
                    }
		    System.out.println( shardsMap.get(future) + " Ok" );
		    System.out.flush();
	        }
	        catch( Exception e )
	        {
	            iter.remove();
	            hasErrors = true;
		    System.out.println( shardsMap.get(future) + " Fail: " + e.getMessage() );
		    System.out.flush();
//		        System.out.println( i + " Fail: " + e.getMessage() );
//	    	        e.printStackTrace();
	        }
	    }
	}
    }
    if( hasErrors ) System.exit(-1);
    System.exit( 0 );
  }
  
  private static void checkShard( String path ) throws Exception
  {
    IndexReader reader = IndexReader.open(NIOFSDirectory.get(new File(path)));
    try
    {
	IndexReader[] subs = reader.getSequentialSubReaders();
	if( subs == null || (subs != null && subs.length == 0) )
	{
	    subs = new IndexReader[1];
	    subs[0] = reader;
	}
	for( int i = 0; i < subs.length; i++ )
	{
	    checkSubReader( subs[i] );
	}
    }
    finally
    {
//	try
//	{
	    if( reader != null ) reader.close();
//	}
//	catch( IOException ign )
//	{
//	    throw ign;
//	}
    }
  }

  private static void checkSubReader( IndexReader reader ) throws Exception
  {
    int maxDoc = reader.maxDoc();
    if( maxDoc > 0 )
    {
	Bits deleted = reader.getDeletedDocs();
	for( int i = 0; i < maxDoc; i+=100 )
	{
	    if( deleted == null || deleted != null  && !deleted.get(i) )
	    {
		reader.document(i); //Test every 100 doc
	    }
	}
    }

//    IndexSearcher searcher = new IndexSearcher(reader);
//    searcher.search( new TermQuery(new Term("mid","999999")), 1 );
    listKeys( reader );
  }

  private static int LIMIT = 100000;
  private static void listKeys( IndexReader reader ) throws Exception
  {
    Terms terms;
    TermsEnum te = null;
    BytesRef term = null;
    String fname;
    FieldsEnum fields = null;
    if( reader == null ) return;
//    System.err.println( "Redare class: " + reader.getClass().getName() );
//    if( reader instanceof org.apache.lucene.index.DirectoryReader )
    if( reader.getClass().getName().equals("org.apache.lucene.index.DirectoryReader") )
    {
        Fields f = MultiFields.getFields(reader);
        if( fields == null ) return;
        fields = f.iterator();
    }
    else
    {
        fields = reader.fields().iterator();
    }

    fname = fields.next();
    for( ; fname != null; fname = fields.next() )
    {
	te = fields.terms();
//	System.out.println( "Checking field: " + fname );
//	te = terms.iterator();
//	te.seek( new BytesRef("QWEQWE"), true );
//	te.seek( new BytesRef(""), true );
	try
	{
	    DocsEnum td = null;
    	    DocsAndPositionsEnum pd = null;
	    int  lines = 0;
	    Bits skipDocs = new Bits.MatchNoBits( 0 );
//	    if( te.term() == null ) term = te.next();
	    while( te.next() != null )
	    {
//	    System.err.println( term.utf8ToString() );
		pd = te.docsAndPositions( skipDocs, pd );
		if( pd != null )
		{
		    int docId;
		    while( (docId = pd.nextDoc()) != DocIdSetIterator.NO_MORE_DOCS )
		    {
			int freq = pd.freq();
			int pos;
			for( int i = 0; i < freq; i++ )
			{
			    pd.nextPosition();
//			    System.err.println( i );
			}
		    }
		}
		else
		{
		    td = te.docs( skipDocs, td );
		    if( td != null )
		    {
			int docId;
			while( (docId = td.nextDoc()) != DocIdSetIterator.NO_MORE_DOCS )
			{
			}
		    }
		}
		lines++;
		if( lines == LIMIT ) break;
//		term = te.next();
	    }
	}
	catch( Exception e )
	{
	    if( skipErrors )
	    {
		e.printStackTrace();
		throw e;
	    }
	    else
	    {
		throw e;
	    }
	}
    }
  }

  private static class CheckShardTask implements Callable<Void>
  {
  private String path;
    public CheckShardTask( String path )
    {
        this.path = path;
    }

    public Void call() throws Exception
    {
        checkShard( path );
        return null;
    }
  }

}
