package ru.yandex.msearch;

import java.util.Arrays;
import java.util.Collections;
import java.util.HashSet;
import java.util.LinkedHashSet;
import java.util.Set;

import org.apache.lucene.util.BytesRef;
import org.apache.lucene.util.StringHelper;

import ru.yandex.http.util.BadRequestException;
import ru.yandex.msearch.collector.CollectingFieldToIndex;
import ru.yandex.msearch.collector.FieldToIndex;
import ru.yandex.msearch.collector.MergeFunc;
import ru.yandex.msearch.collector.SearchParams;
import ru.yandex.msearch.collector.docprocessor.DocProcessor;
import ru.yandex.msearch.collector.docprocessor.ModuleFieldsAggregator;
import ru.yandex.msearch.collector.group.NullGroupFunc;
import ru.yandex.msearch.collector.group.GroupFunc;
import ru.yandex.msearch.collector.group.GroupFuncFactory;
import ru.yandex.msearch.collector.postfilter.PostFilter;
import ru.yandex.msearch.collector.sort.SortFunc;
import ru.yandex.msearch.config.DatabaseConfig;
import ru.yandex.parser.uri.CgiParams;
import ru.yandex.parser.string.CollectionParser;
import ru.yandex.parser.string.NonEmptyValidator;
import ru.yandex.parser.string.NonNegativeIntegerValidator;
import ru.yandex.parser.string.NonNegativeLongValidator;
import ru.yandex.parser.string.PositiveIntegerValidator;

// TODO: most of parameters intersects with search parameters, this code
// duplication must be eliminated
public class ClusteringConfig implements SearchParams {
    private static final double EARTH_RADIUS = 6371000;

    private final DatabaseConfig config;
    private final GroupFunc group;
    private final boolean printGroup;
    private final PostFilter postFilter;
    private final GetFieldsConfig getFieldsConfig;
    private final Set<String> readFields;
    private final FieldConfig[] loadFields;
    private final DocProcessor docProcessor;
    private final int minClusterSize;
    private final long interval;
    private final long localInterval;
    private final double distance;
    private final boolean asc;
    private final int offset;
    private final int length;
    private final int mergedLength;
    private final int dateField;
    private final int latitudeField;
    private final int longitudeField;
    private final FieldToIndex fieldToIndex;
    private final Set<String> countDistinct;
    private final int[] fieldIndexes;
    private final String[] fieldList;
    private final BytesRef[] utf8FieldList;
    private final RequestContext ctx;

    public ClusteringConfig(
        final DatabaseConfig config,
        final CgiParams params,
        final ProcessorRequestContext processorRequestContext)
        throws BadRequestException
    {
        this.config = config;
        ctx = processorRequestContext.ctx();
        CollectingFieldToIndex fieldToIndex =
            processorRequestContext.fieldToIndex();
        group = params.get(
            "group",
            NullGroupFunc.INSTANCE,
            x -> GroupFuncFactory.INSTANCE.apply(x, fieldToIndex));
        printGroup = !params.getBoolean(
            "skip-group",
            group == NullGroupFunc.INSTANCE);
        minClusterSize =
            params.get("min-cluster", 8, PositiveIntegerValidator.INSTANCE);
        interval =
            params.get("interval", 1800L, NonNegativeLongValidator.INSTANCE);
        Set<String> fields = new HashSet<>();
        long distanceInMeters = params.get(
            "distance",
            1000L,
            NonNegativeLongValidator.INSTANCE);
        String latitudeField;
        String longitudeField;
        if (distanceInMeters == 0L) {
            localInterval = interval;
            latitudeField = null;
            longitudeField = null;
        } else {
            // XXX: it is implied that
            // ∀P,Q ∈ Cluster P.date() ≤ X.date() ≤ Q.date() ⇒ X ∈ Cluster
            // So, if local-interval is much bigger that date-interval, then
            // there are possible points which will belong to cluster, but
            // distance from other points can be any and time difference will
            // be bigger that interval
            localInterval = params.get(
                "local-interval",
                interval << 1L,
                NonNegativeLongValidator.INSTANCE);
            latitudeField = StringHelper.intern(
                params.getString("latitude-field", "latitude"));
            longitudeField = StringHelper.intern(
                params.getString("longitude-field", "longitude"));
            fields.add(latitudeField);
            fields.add(longitudeField);
        }
        distance = distanceInMeters / EARTH_RADIUS;
        asc = params.getBoolean("asc", false);
        offset = params.get("offset", 0, NonNegativeIntegerValidator.INSTANCE);
        length = params.get(
            "length",
            Integer.MAX_VALUE - offset,
            NonNegativeIntegerValidator.INSTANCE);
        mergedLength = params.get(
            "merged-length",
            Integer.MAX_VALUE,
            NonNegativeIntegerValidator.INSTANCE);

        Set<String> generatedFields = new HashSet<>();
        docProcessor = NewSearchRequest.extractDocProcessor(
            config,
            generatedFields,
            params,
            processorRequestContext);

        getFieldsConfig = NewSearchRequest.extractGetFieldsConfig(
            config,
            generatedFields,
            params);

        postFilter = NewSearchRequest.extractPostFilter(
            config,
            generatedFields,
            params,
            fieldToIndex);

        String dateField =
            StringHelper.intern(params.getString("date-field", "created"));

        Set<String> countDistinct = params.getAll(
            "count-distinct",
            Collections.emptySet(),
            new CollectionParser<>(
                NonEmptyValidator.INSTANCE,
                LinkedHashSet::new));
        this.countDistinct = new LinkedHashSet<>(countDistinct.size() << 1);
        for (String field: countDistinct) {
            this.countDistinct.add(StringHelper.intern(field));
        }

        fields.add(dateField);
        readFields = new HashSet<>(getFieldsConfig.fields());
        for (String field: readFields) {
            fieldToIndex.indexFor(field);
        }
        for (Set<String> set
            : Arrays.asList(
                fields,
                group.loadFields(),
                postFilter.loadFields(),
                this.countDistinct))
        {
            for (String field: set) {
                if (!generatedFields.contains(field)) {
                    NewSearchRequest.checkStoredField(config, field);
                    readFields.add(field);
                    fieldToIndex.indexFor(field);
                }
            }
        }

        ModuleFieldsAggregator fieldsAggregator = new ModuleFieldsAggregator();
        docProcessor.apply(fieldsAggregator);
        for (String field: fieldsAggregator.loadFields()) {
            FieldConfig fieldConfig = config.fieldConfig(field);
            if (fieldConfig != null && fieldConfig.store()) {
                readFields.add(field);
            }
        }
        for (String field: fieldsAggregator.outFields()) {
            FieldConfig fieldConfig = config.fieldConfig(field);
            if (fieldConfig == null || !fieldConfig.store()) {
                readFields.remove(field);
            }
        }

        this.fieldToIndex = fieldToIndex.compactFieldToIndex();
        loadFields = new FieldConfig[this.fieldToIndex.fieldsCount()];
        for (String field: readFields) {
            loadFields[this.fieldToIndex.indexFor(field)] =
                config.fieldConfigFast(field);
        }
        fieldIndexes = new int[getFieldsConfig.fields().size()];
        int i = 0;
        for (String field: getFieldsConfig.fields()) {
            fieldIndexes[i++] = this.fieldToIndex.indexFor(field);
        }
        this.dateField = this.fieldToIndex.indexFor(dateField);
        if (latitudeField == null) {
            this.latitudeField = -1;
        } else {
            this.latitudeField = this.fieldToIndex.indexFor(latitudeField);
        }
        if (longitudeField == null) {
            this.longitudeField = -1;
        } else {
            this.longitudeField = this.fieldToIndex.indexFor(longitudeField);
        }
        fieldList = getFieldsConfig.fields().toArray(
            new String[getFieldsConfig.fields().size()]);
        utf8FieldList = new BytesRef[fieldList.length];
        i = 0;
        for (String field: fieldList) {
            utf8FieldList[i++] = new BytesRef(field);
        }
    }

    public RequestContext ctx() {
        return ctx;
    }

    public PostFilter postFilter() {
        return postFilter;
    }

    public GetFieldsConfig getFieldsConfig() {
        return getFieldsConfig;
    }

    public int[] fieldIndexes() {
        return fieldIndexes;
    }

    public String[] fieldList() {
        return fieldList;
    }

    public BytesRef[] utf8FieldList() {
        return utf8FieldList;
    }

    public Set<String> readFields() {
        return readFields;
    }

    public DocProcessor docProcessor() {
        return docProcessor;
    }

    public int minClusterSize() {
        return minClusterSize;
    }

    public long interval() {
        return interval;
    }

    public long localInterval() {
        return localInterval;
    }

    public double distance() {
        return distance;
    }

    public boolean asc() {
        return asc;
    }

    public int offset() {
        return offset;
    }

    public int length() {
        return length;
    }

    public int mergedLength() {
        return mergedLength;
    }

    public int dateField() {
        return dateField;
    }

    public int latitudeField() {
        return latitudeField;
    }

    public int longitudeField() {
        return longitudeField;
    }

    public Set<String> countDistinct() {
        return countDistinct;
    }

    public boolean printGroup() {
        return printGroup;
    }

    @Override
    public GroupFunc groupFunc() {
        return group;
    }

    @Override
    public SortFunc sortFunc() {
        return null;
    }

    @Override
    public FieldToIndex fieldToIndex() {
        return fieldToIndex;
    }

    public FieldConfig[] loadFields() {
        return loadFields;
    }

    @Override
    public Set<String> mergeFields() {
        return Collections.emptySet();
    }

    @Override
    public MergeFunc mergeFunc() {
        return null;
    }
}

