package ru.yandex.msearch;

import java.io.IOException;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.logging.Level;

import org.apache.http.HttpException;
import org.apache.http.HttpRequest;
import org.apache.http.HttpResponse;
import org.apache.http.HttpStatus;
import org.apache.http.entity.ContentType;
import org.apache.http.entity.EntityTemplate;
import org.apache.http.protocol.HttpContext;
import org.apache.http.protocol.HttpRequestHandler;
import org.apache.lucene.queryParser.QueryParser;
import org.apache.lucene.search.Query;
import org.apache.lucene.util.BytesRef;

import ru.yandex.collection.PatternMap;
import ru.yandex.http.server.sync.JsonContentProducerWriter;
import ru.yandex.http.server.sync.Utf8JsonContentProducer;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.CharsetUtils;
import ru.yandex.http.util.request.RequestInfo;
import ru.yandex.http.util.server.LoggingServerConnection;
import ru.yandex.json.writer.JsonTypeExtractor;
import ru.yandex.json.writer.JsonValue;
import ru.yandex.json.writer.JsonWriterBase;
import ru.yandex.json.writer.Utf8JsonValue;
import ru.yandex.json.writer.Utf8JsonWriter;
import ru.yandex.msearch.collector.ClusteringCollector;
import ru.yandex.msearch.collector.CollectingFieldToIndex;
import ru.yandex.msearch.collector.cluster.Cluster;
import ru.yandex.msearch.collector.cluster.ClusterDoc;
import ru.yandex.msearch.collector.cluster.ClusterWithGroup;
import ru.yandex.msearch.collector.cluster.TruncatedCluster;
import ru.yandex.msearch.util.Compress;
import ru.yandex.msearch.util.IOStater;
import ru.yandex.parser.string.NonEmptyValidator;
import ru.yandex.parser.uri.CgiParams;
import ru.yandex.search.prefix.Prefix;

public class ClusterizeHandler
    extends SearchHandlerBase
    implements HttpRequestHandler
{
    private static final int MAX_MERGED_LENGTH = 1024;
    private static final ClusteringCollector.ClusterFactory<TruncatedCluster>
        TRUNCATED_CLUSTER_FACTORY =
            new ClusteringCollector.ClusterFactory<TruncatedCluster>() {
                @Override
                public TruncatedCluster create(
                    final ClusteringConfig config,
                    final Map<String, Map<String, Cluster.Counter>> counters,
                    final ClusterDoc doc)
                {
                    return new TruncatedCluster(config, counters, doc);
                }
            };
    private static final ClusteringCollector.ClusterFactory<Cluster>
        CLUSTER_FACTORY =
            new ClusteringCollector.ClusterFactory<Cluster>() {
                @Override
                public Cluster create(
                    final ClusteringConfig config,
                    final Map<String, Map<String, Cluster.Counter>> counters,
                    final ClusterDoc doc)
                {
                    return new Cluster(config, counters, doc);
                }
            };

    public ClusterizeHandler(
        final DatabaseManager dbManager,
        final Config config,
        final PatternMap<RequestInfo, IOStater> ioStaters)
    {
        super(dbManager, config, ioStaters);
    }

    @Override
    public void handle(final HttpRequest request,
        final HttpResponse response,
        final HttpContext context)
        throws HttpException, IOException
    {
        HttpRequestContext ctx = new HttpRequestContext(context);
        CgiParams params = new CgiParams(request);
        String text = params.get("text", NonEmptyValidator.INSTANCE);
        Index index = dbManager.indexOrException(params, SearchHandler.BRE_GEN);
        Prefix prefix = params.get("prefix", prefixParser(index.config()));

        ProcessorRequestContext requestContext =
            new ProcessorRequestContext(
                Collections.singleton(prefix),
                index,
                ctx,
                new CollectingFieldToIndex(),
                params);

        ClusteringConfig config =
            new ClusteringConfig(index.config(), params, requestContext);

        ClusteringCollector<?> col;
        if (config.latitudeField() == -1
            && config.mergedLength() <= MAX_MERGED_LENGTH)
        {
            col = new ClusteringCollector<TruncatedCluster>(
                config,
                TRUNCATED_CLUSTER_FACTORY);
        } else {
            col = new ClusteringCollector<Cluster>(config, CLUSTER_FACTORY);
        }
        PrefixingAnalyzerWrapper analyzer = index.searchAnalyzer(prefix);
        QueryParser parser = createParser(index.config(), params, analyzer);

        //FIXME context should be final
        requestContext.queryParser(parser);

        Query query = null;

        Compress.resetStats();
        IOStater ioStater = ioStaterFor(context);

        try {
            try {
                query = parser.parse(text);
                // XXX: Do not remove this log record!
                // It exposes the bug in WildcardQuery
                if (ctx.logger().isLoggable(Level.INFO)) {
                    ctx.logger().info("Request parsed: " + query);
                }
            } catch (Exception e) {
                throw new BadRequestException(
                    "Failed to parse query '" + text + '\'', e);
            }

            index.search(query, col, prefix);
            col.close();
            List<ClusterWithGroup> clusters = col.clusters();
            if (ctx.logger().isLoggable(Level.INFO)) {
                ctx.logger().info("Clusters count: " + clusters.size());
            }
            ((LoggingServerConnection) ctx.connection())
                .setHitsCount(Integer.toString(clusters.size()));
            Charset charset = CharsetUtils.acceptedCharset(request);
            EntityTemplate entity;
            if (charset.equals(StandardCharsets.UTF_8)) {
                entity =
                    new EntityTemplate(
                        new Utf8JsonContentProducer(
                            new Utf8ClustersProducer(clusters, config),
                            JsonTypeExtractor.NORMAL.extract(params)));
            } else {
                entity =
                    new EntityTemplate(
                        new JsonContentProducerWriter(
                            new ClustersProducer(clusters, config),
                            JsonTypeExtractor.NORMAL.extract(params),
                            charset));
            }
            entity.setChunked(true);
            entity.setContentType(
                ContentType.APPLICATION_JSON.withCharset(charset).toString());
            response.setEntity(entity);
            response.setStatusCode(HttpStatus.SC_OK);
        } finally {
            accountStats(ioStater, ctx.logger());
        }
    }

    @Override
    public String toString() {
        return "https://wiki.yandex-team.ru/ps/Documentation/Lucene/"
            + "SearchHandlers/clusterize";
    }

    private static boolean hasCounters(
        final Map<String, Map<String, Cluster.Counter>> counters)
    {
        if (counters != null) {
            for (Map<?, ?> fieldCounters: counters.values()) {
                if (!fieldCounters.isEmpty()) {
                    return true;
                }
            }
        }
        return false;
    }

    private static int printCount(
        final int clustersSize,
        final int offset,
        final int length)
    {
        int printCount;
        if (clustersSize <= offset) {
            printCount = 0;
        } else if (clustersSize >= offset + length) {
            printCount = length;
        } else {
            printCount = clustersSize - offset;
        }
        return printCount;
    }

    private static class ClustersProducer implements JsonValue {
        private final List<ClusterWithGroup> clusters;
        private final int offset;
        private final int length;
        private final int mergedLength;
        private final boolean printGroup;
        private final boolean skipNulls;

        ClustersProducer(
            final List<ClusterWithGroup> clusters,
            final ClusteringConfig config)
        {
            this.clusters = clusters;
            offset = config.offset();
            length = config.length();
            mergedLength = config.mergedLength();
            printGroup = config.printGroup();
            skipNulls = config.getFieldsConfig().skipNulls();
        }

        private void writeCluster(
            final JsonWriterBase writer,
            final ClusterWithGroup cluster)
            throws IOException
        {
            writer.startObject();
            writer.key("size");
            writer.value(cluster.cluster().size());
            if (printGroup && !(skipNulls && cluster.group() == null)) {
                writer.key("group");
                writer.value(cluster.group());
            }
            writer.key("max");
            writer.value(cluster.cluster().maxDate());
            writer.key("min");
            writer.value(cluster.cluster().minDate());
            Map<String, Map<String, Cluster.Counter>> counters =
                cluster.cluster().counters();
            if (hasCounters(counters)) {
                writer.key("counters");
                writer.startObject();
                for (Map.Entry<String, Map<String, Cluster.Counter>> entry
                    : counters.entrySet())
                {
                    Map<String, Cluster.Counter> value = entry.getValue();
                    if (!value.isEmpty()) {
                        writer.key(entry.getKey());
                        writer.startObject();
                        for (Map.Entry<String, Cluster.Counter> subentry
                            : value.entrySet())
                        {
                            writer.key(subentry.getKey());
                            writer.value(subentry.getValue().get());
                        }
                        writer.endObject();
                    }
                }
                writer.endObject();
            }
            if (mergedLength > 0) {
                writer.key("merged_docs");
                cluster.cluster().writeTo(writer);
            }
            writer.endObject();
        }

        @Override
        public void writeValue(final JsonWriterBase writer)
            throws IOException
        {
            writer.startObject();
            writer.key("hitsCount");
            writer.value(clusters.size());
            writer.key("hitsArray");
            writer.startArray();
            int printCount = printCount(clusters.size(), offset, length);
            for (int i = 0; i < printCount; ++i) {
                writeCluster(writer, clusters.get(i + offset));
                clusters.set(i + offset, null);
            }
            writer.endArray();
            writer.endObject();
        }
    }

    private static class Utf8ClustersProducer implements Utf8JsonValue {
        private static final BytesRef SIZE = new BytesRef("size");
        private static final BytesRef GROUP = new BytesRef("group");
        private static final BytesRef MAX = new BytesRef("max");
        private static final BytesRef MIN = new BytesRef("min");
        private static final BytesRef COUNTERS = new BytesRef("counters");
        private static final BytesRef MERGED_DOCS =
            new BytesRef("merged_docs");
        private static final BytesRef HITS_COUNT =
            new BytesRef("hitsCount");
        private static final BytesRef HITS_ARRAY =
            new BytesRef("hitsArray");

        private final List<ClusterWithGroup> clusters;
        private final int offset;
        private final int length;
        private final int mergedLength;
        private final boolean printGroup;
        private final boolean skipNulls;

        Utf8ClustersProducer(
            final List<ClusterWithGroup> clusters,
            final ClusteringConfig config)
        {
            this.clusters = clusters;
            offset = config.offset();
            length = config.length();
            mergedLength = config.mergedLength();
            printGroup = config.printGroup();
            skipNulls = config.getFieldsConfig().skipNulls();
        }

        private void writeCluster(
            final Utf8JsonWriter writer,
            final ClusterWithGroup cluster)
            throws IOException
        {
            writer.startObject();
            writer.key(SIZE);
            writer.value(cluster.cluster().size());
            if (printGroup && !(skipNulls && cluster.group() == null)) {
                writer.key(GROUP);
                writer.value((Utf8JsonValue) cluster.group());
            }
            writer.key(MAX);
            writer.value(cluster.cluster().maxDate());
            writer.key(MIN);
            writer.value(cluster.cluster().minDate());
            Map<String, Map<String, Cluster.Counter>> counters =
                cluster.cluster().counters();
            if (hasCounters(counters)) {
                writer.key(COUNTERS);
                writer.startObject();
                for(Map.Entry<String, Map<String, Cluster.Counter>> entry
                    : counters.entrySet())
                {
                    Map<String, Cluster.Counter> value = entry.getValue();
                    if (!value.isEmpty()) {
                        writer.key(entry.getKey());
                        writer.startObject();
                        for (Map.Entry<String, Cluster.Counter> subentry
                            : value.entrySet())
                        {
                            writer.key(subentry.getKey());
                            writer.value(subentry.getValue().get());
                        }
                        writer.endObject();
                    }
                }
                writer.endObject();
            }
            if (mergedLength > 0) {
                writer.key(MERGED_DOCS);
                cluster.cluster().writeTo(writer);
            }
            writer.endObject();
        }

        @Override
        public void writeValue(final Utf8JsonWriter writer)
            throws IOException
        {
            writer.startObject();
            writer.key(HITS_COUNT);
            writer.value(clusters.size());
            writer.key(HITS_ARRAY);
            writer.startArray();
            int printCount = printCount(clusters.size(), offset, length);
            for (int i = 0; i < printCount; ++i) {
                writeCluster(writer, clusters.get(i + offset));
                clusters.set(i + offset, null);
            }
            writer.endArray();
            writer.endObject();
        }
    }
}

