package ru.yandex.msearch;

import java.io.File;
import java.io.IOException;
import java.lang.reflect.InvocationTargetException;
import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.lucene.index.codecs.yandex.YandexCodec;
import org.apache.lucene.queryParser.QueryParser;
import org.apache.lucene.util.StringHelper;

import ru.yandex.analyzer.TokenizerType;
import ru.yandex.client.tvm2.ImmutableTvm2ServiceConfig;
import ru.yandex.client.tvm2.Tvm2ServiceConfigBuilder;
import ru.yandex.collection.PatternMap;
import ru.yandex.http.util.server.AuthsConfigBuilder;
import ru.yandex.http.util.server.ImmutableAuthsConfig;
import ru.yandex.http.util.server.ImmutableHttpServerConfig;
import ru.yandex.http.util.server.HttpServerConfigBuilder;
import ru.yandex.http.util.server.LimiterConfigBuilder;
import ru.yandex.http.util.server.LimitersConfigBuilder;
import ru.yandex.http.util.request.RequestInfo;
import ru.yandex.logger.BackendAccessLoggerConfigDefaults;
import ru.yandex.logger.ImmutableLoggerConfig;
import ru.yandex.logger.LoggerConfigBuilder;
import ru.yandex.logger.LoggerConfigDefaults;
import ru.yandex.logger.LoggersConfigBuilder;
import ru.yandex.logger.MinimalLoggerConfigDefaults;
import ru.yandex.logger.SearchProxyAccessLoggerConfigDefaults;
import ru.yandex.logger.StdoutLoggerConfigDefaults;
import ru.yandex.msearch.collector.docprocessor.DefaultDocProcessorFactory;
import ru.yandex.msearch.collector.docprocessor.DocProcessorFactory;
import ru.yandex.msearch.config.DatabaseConfig;
import ru.yandex.msearch.config.DatabaseConfigBuilder;
import ru.yandex.msearch.util.SleepingIOScheduler;
import ru.yandex.parser.config.ConfigException;
import ru.yandex.parser.config.IniConfig;
import ru.yandex.parser.string.CollectionParser;
import ru.yandex.parser.string.IntegerMemorySizeParser;
import ru.yandex.parser.string.LongMemorySizeParser;
import ru.yandex.parser.string.NonEmptyValidator;
import ru.yandex.parser.string.NonNegativeIntegerValidator;
import ru.yandex.parser.string.NotLessDoubleValidator;
import ru.yandex.parser.string.NotLessIntegerValidator;
import ru.yandex.parser.string.PositiveIntegerValidator;
import ru.yandex.parser.string.PositiveLongValidator;
import ru.yandex.search.prefix.PrefixParser;
import ru.yandex.search.prefix.PrefixType;
import ru.yandex.stater.GolovanPanelConfigBuilder;
import ru.yandex.stater.ImmutableGolovanPanelConfig;
import ru.yandex.stater.RequestsStater;
import ru.yandex.stater.SimpleHistogramMetric;
import ru.yandex.stater.StaterConfigBuilder;
import ru.yandex.stater.StatersConfigBuilder;

public class Config {
    private final int indexThreads;

    private final File indexPath;
    private final File xurlsRegexFile;

    private final PatternMap<RequestInfo, LoggerConfigBuilder> accessLog;
    private final PatternMap<RequestInfo, LoggerConfigBuilder> indexAccessLog;
    private final PatternMap<RequestInfo, LoggerConfigBuilder> fullLog;
    private final PatternMap<RequestInfo, LoggerConfigBuilder> indexFullLog;
    private final ImmutableLoggerConfig indexLog;
    private final ImmutableLoggerConfig errorLog;
    private final ImmutableTvm2ServiceConfig tvm2ServiceConfig;
    private final ImmutableGolovanPanelConfig indexerGolovanPanelConfig;
    private final ImmutableGolovanPanelConfig searcherGolovanPanelConfig;
    private final ImmutableAuthsConfig auths;
    private final ImmutableHttpServerConfig indexer;
    private final ImmutableHttpServerConfig http;
    private final ImmutableHttpServerConfig dump;
    private final ImmutableHttpServerConfig search;
    private final PatternMap<RequestInfo, StaterConfigBuilder>
        indexRequestsStaters;
    private final PatternMap<RequestInfo, StaterConfigBuilder>
        searchRequestsStaters;
    private final PatternMap<RequestInfo, LimiterConfigBuilder> indexLimiters;
    private final PatternMap<RequestInfo, LimiterConfigBuilder> searchLimiters;
    private final RequestsStater oldRequestsStater;
    private final long parallelDocsSize;

    private final long cacheCapacity;
    private final long compressedCacheCapacity;
    private final boolean compressedCacheDynamicShrink;
    private final long dynamicCacheMinMemoryFree;
    private final boolean returnMaxQueueId;
    private final boolean logDeleteDocument;
    private final boolean logIndexDocument;
    private final int ioThreads;
    private final int ioPriorityDivisor;
    private final boolean ioSleepingScheduler;

    private final File ssdCachePath;
    private final long ssdCacheSize;
    private final File statusFile;
    private final String dropPassword;
    private final boolean useFadvise;
    private final boolean infoStream;
    private final SimpleHistogramMetric indexLagHistogram;
    private final Map<String, DatabaseConfig> databasesConfigs;

    public static final String PREFIX_FIELD_KEY =
        StringHelper.intern("__prefix");
    public static final String PREFIX_FIELD_CONFIG =
        "tokenizer = keyword\n"
        + "prefixed = false\n"
        + "analyze = false\n"
        + "attribute = true\n"
        + "store = true";

    public static final String QUEUE_ID_FIELD_KEY =
        StringHelper.intern("__queue_id");
    public static final String QUEUE_ID_FIELD_CONFIG =
        "index = false\n"
        + "store = true\n"
        + "type = long";

    public static final String QUEUE_NAME_FIELD_KEY =
        StringHelper.intern("__queue_name");
    public static final String QUEUE_NAME_FIELD_CONFIG =
        "index = false\n"
        + "store = true";

    public Config(final IniConfig config) throws ConfigException {
        useFadvise = config.getBoolean(
            "use-fadvise",
            true);

        ioThreads = config.get(
            "io-threads",
            0,
            new NotLessIntegerValidator(0));
        ioSleepingScheduler = config.getBoolean(
            "io-sleeping-scheduler",
            true);
        ioPriorityDivisor = config.get(
            "io-priority-divisor",
            SleepingIOScheduler.DEFAULT_PRIORITY_DIVISOR,
            new NotLessIntegerValidator(0));
        dropPassword =
            config.getString(
                "drop-password",
                System.getenv("LUCENE_DROP_PASSWORD"));
        returnMaxQueueId = config.getBoolean("return-max-queue-id", false);
        indexThreads = config.get("index_threads", 10,
            PositiveIntegerValidator.INSTANCE);
        cacheCapacity = config.get("block-cache-size",
            1_000_000_000L,
            LongMemorySizeParser.INSTANCE);
        ssdCacheSize = config.get("ssd-cache-size",
            15_000_000_000L,
            LongMemorySizeParser.INSTANCE);
        compressedCacheCapacity = config.get("compressed-cache-size",
            1_000_000_000L,
            LongMemorySizeParser.INSTANCE);
        compressedCacheDynamicShrink = config.getBoolean(
            "compressed-cache-dynamic-shrink",
            false);
        dynamicCacheMinMemoryFree = config.get(
            "dynamic-cache-min-free-memory",
            1_000_000_000L,
            LongMemorySizeParser.INSTANCE);
        // tmp return for tests
        logDeleteDocument = config.getBoolean("log_delete_document", true);
        logIndexDocument = config.getBoolean("log_index_document", true);

        // TODO Now needed only by ssd cache
        indexPath = config.getDir("index_path", new File("/u0/index/"));
        ssdCachePath =
            config.getOutputFile("ssd_cache_file", null);
        statusFile =
            config.getOutputFile("status_file", new File("lucene.status"));
        xurlsRegexFile = config.getInputFile("xurls_regex_file", null);
        LoggerConfigBuilder accessLogConfig = new LoggerConfigBuilder(
            BackendAccessLoggerConfigDefaults.INSTANCE);
        accessLogConfig.single()
            .logFormat(
                BackendAccessLoggerConfigDefaults.INSTANCE.logFormat()
                    + " %{"
                    + SearchProxyAccessLoggerConfigDefaults.HITS_COUNT
                    + '}');
        accessLog = LoggersConfigBuilder.loadLoggers(
            config.section("access_log"),
            new PatternMap<>(accessLogConfig));
        indexAccessLog = LoggersConfigBuilder.loadLoggers(
            config.section("index_access_log"),
            new PatternMap<>(BackendAccessLoggerConfigDefaults.INSTANCE));
        fullLog = LoggersConfigBuilder.loadLoggers(
            config.section("full_log"),
            new PatternMap<>(LoggerConfigDefaults.INSTANCE));
        IniConfig indexFullLogSection = config.sectionOrNull("index_full_log");
        if (indexFullLogSection != null) {
            indexFullLog =
                LoggersConfigBuilder.loadLoggers(
                    indexFullLogSection,
                    new PatternMap<>(LoggerConfigDefaults.INSTANCE));
        } else {
            indexFullLog = fullLog;
        }

        indexLog =
            new LoggerConfigBuilder(
                config.section("index_log"),
                MinimalLoggerConfigDefaults.INSTANCE)
                .build();
        errorLog =
            new LoggerConfigBuilder(
                config.section("error_log"),
                StdoutLoggerConfigDefaults.INSTANCE)
                .build();
        indexRequestsStaters = StatersConfigBuilder.loadStaters(
            config.section("indexer-stat"),
            new PatternMap<>(new StaterConfigBuilder().prefix("indexer")));
        searchRequestsStaters = StatersConfigBuilder.loadStaters(
            config.section("searcher-stat"),
            new PatternMap<>(new StaterConfigBuilder().prefix("searcher")));
        oldRequestsStater =
            new StatersConfigBuilder().staters(
                StatersConfigBuilder.loadStaters(
                    config.section("old-stat"),
                    new PatternMap<>(
                        new StaterConfigBuilder().prefix("oldport"))))
                .build()
                .preparedStaters()
                .asterisk();
        IniConfig tvm2ServiceSection = config.sectionOrNull("tvm2");
        if (tvm2ServiceSection == null) {
            tvm2ServiceConfig = null;
        } else {
            tvm2ServiceConfig =
                new ImmutableTvm2ServiceConfig(
                    new Tvm2ServiceConfigBuilder(tvm2ServiceSection));
        }
        IniConfig indexerGolovanPanelSection =
            config.sectionOrNull("indexer-golovan-panel");
        if (indexerGolovanPanelSection == null) {
            indexerGolovanPanelConfig = null;
        } else {
            indexerGolovanPanelConfig =
                new ImmutableGolovanPanelConfig(
                    new GolovanPanelConfigBuilder(indexerGolovanPanelSection));
        }
        IniConfig searcherGolovanPanelSection =
            config.sectionOrNull("searcher-golovan-panel");
        if (searcherGolovanPanelSection == null) {
            searcherGolovanPanelConfig = null;
        } else {
            searcherGolovanPanelConfig =
                new ImmutableGolovanPanelConfig(
                    new GolovanPanelConfigBuilder(searcherGolovanPanelSection));
        }
        auths =
            new ImmutableAuthsConfig(
                new AuthsConfigBuilder(config.section("auth")));
        searchLimiters = LimitersConfigBuilder.loadLimiters(
            config.section("searcher-limiter"),
            new PatternMap<>(new LimiterConfigBuilder()));
        indexLimiters = LimitersConfigBuilder.loadLimiters(
            config.section("indexer-limiter"),
            new PatternMap<>(new LimiterConfigBuilder()));
        indexer = new HttpServerConfigBuilder(config.section("indexer"))
            .name("IndexerServer")
            .build();
        http = new HttpServerConfigBuilder(config.section("http"))
            .name("HttpServer")
            .build();
        final HttpServerConfigBuilder dumpDefaults =
            new HttpServerConfigBuilder(config.section("http"));
        dumpDefaults.name("DeumpServer");
        if (dumpDefaults.port() != 0) {
            dumpDefaults.port(dumpDefaults.port() + 10);
        }
        dump = new HttpServerConfigBuilder(config.section("dump"), dumpDefaults)
            .name("DumpServer")
            .build();
        search = new HttpServerConfigBuilder(config.section("search"))
            .name("SearchServer")
            .build();

        parallelDocsSize = config.getLong("parallel_docs_size", 1L << 30L);
        infoStream = config.getBoolean("info_stream", false);

        IniConfig statLagConfig = config.sectionOrNull("index-stat-lag");
        if (statLagConfig != null) {
            indexLagHistogram = new SimpleHistogramMetric(statLagConfig);
        } else {
            indexLagHistogram = null;
        }

        this.databasesConfigs = Collections.unmodifiableMap(parseDatabases(config));
        config.checkUnusedKeys();
    }

    public Map<String, DatabaseConfig> databasesConfigs() {
        return databasesConfigs;
    }


    public SimpleHistogramMetric indexLagHistogram() {
        return indexLagHistogram;
    }

    public boolean useFadvise() {
        return useFadvise;
    }

    public File statusFile() {
        return statusFile;
    }

    public File ssdCachePath() {
        return ssdCachePath;
    }

    public long ssdCacheSize() {
        return ssdCacheSize;
    }

    public int ioThreads() {
        return ioThreads;
    }

    public int ioPriorityDivisor() {
        return ioPriorityDivisor;
    }

    public boolean ioSleepingScheduler() {
        return ioSleepingScheduler;
    }

    public boolean returnMaxQueueId() {
        return returnMaxQueueId;
    }

    public int indexThreads() {
        return indexThreads;
    }

    public long cacheCapacity() {
        return cacheCapacity;
    }

    public long compressedCacheCapacity() {
        return compressedCacheCapacity;
    }

    public boolean compressedCacheDynamicShrink() {
        return compressedCacheDynamicShrink;
    }

    public long dynamicCacheMinMemoryFree() {
        return dynamicCacheMinMemoryFree;
    }

    // TODO required by ssd cache keep for a while
    public File indexPath() {
        return indexPath;
    }

    public File xurlsRegexFile() {
        return xurlsRegexFile;
    }


    public PatternMap<RequestInfo, LoggerConfigBuilder> accessLog() {
        return accessLog;
    }

    public PatternMap<RequestInfo, LoggerConfigBuilder> indexAccessLog() {
        return indexAccessLog;
    }

    public PatternMap<RequestInfo, LoggerConfigBuilder> fullLog() {
        return fullLog;
    }

    public PatternMap<RequestInfo, LoggerConfigBuilder> indexFullLog() {
        return indexFullLog;
    }

    public ImmutableLoggerConfig indexLog() {
        return indexLog;
    }

    public ImmutableLoggerConfig errorLog() {
        return errorLog;
    }

    public PatternMap<RequestInfo, StaterConfigBuilder>
    indexRequestsStaters() {
        return indexRequestsStaters;
    }

    public PatternMap<RequestInfo, StaterConfigBuilder>
    searchRequestsStaters() {
        return searchRequestsStaters;
    }

    public RequestsStater oldRequestsStater() {
        return oldRequestsStater;
    }

    public PatternMap<RequestInfo, LimiterConfigBuilder> searchLimiters() {
        return searchLimiters;
    }

    public PatternMap<RequestInfo, LimiterConfigBuilder> indexLimiters() {
        return indexLimiters;
    }

    public ImmutableTvm2ServiceConfig tvm2ServiceConfig() {
        return tvm2ServiceConfig;
    }

    public ImmutableGolovanPanelConfig indexerGolovanPanelConfig() {
        return indexerGolovanPanelConfig;
    }

    public ImmutableGolovanPanelConfig searcherGolovanPanelConfig() {
        return searcherGolovanPanelConfig;
    }

    public ImmutableAuthsConfig auths() {
        return auths;
    }

    public ImmutableHttpServerConfig indexer() {
        return indexer;
    }

    public ImmutableHttpServerConfig http() {
        return http;
    }

    public ImmutableHttpServerConfig dump() {
        return dump;
    }

    public ImmutableHttpServerConfig search() {
        return search;
    }

    public long parallelDocsSize() {
        return parallelDocsSize;
    }

    public boolean infoStream() {
        return infoStream;
    }

    public String dropPassword() {
        return dropPassword;
    }

    public static Map<String, DatabaseConfig> parseDatabases(
        final IniConfig config)
        throws ConfigException
    {
        Map<String, DatabaseConfig> databasesConfigs;
        IniConfig databasesSection = config.sectionOrNull("database");
        if (databasesSection != null) {
            databasesConfigs = new LinkedHashMap<>();
            for (Map.Entry<String, IniConfig> dbEntry : databasesSection.sections().entrySet()) {
                databasesConfigs.put(
                    dbEntry.getKey(),
                    new DatabaseConfigBuilder(dbEntry.getValue(), dbEntry.getKey()));
            }

            if (config.getString("shards", null) != null) {
                if (databasesConfigs.containsKey(DatabaseManager.DEFAULT_DATABASE)) {
                    throw new ConfigException(
                        "Default database specified, but shards field in main config found");
                }
                databasesConfigs.putIfAbsent(
                    DatabaseManager.DEFAULT_DATABASE,
                    new DatabaseConfigBuilder(config, DatabaseManager.DEFAULT_DATABASE));
            }
        } else {
            DatabaseConfig defaultDbConfig = new DatabaseConfigBuilder(config, DatabaseManager.DEFAULT_DATABASE);
            databasesConfigs = Collections.singletonMap(DatabaseManager.DEFAULT_DATABASE, defaultDbConfig);
        }

        return databasesConfigs;
    }
}

