package ru.yandex.msearch;

/* An example of a very simple, multi-threaded HTTP server.
 * Implementation notes are in WebServer.html, and also
 * as comments in the source code.
 */

import java.io.*;
import java.net.*;
import java.nio.file.FileVisitResult;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.attribute.BasicFileAttributes;
import java.util.*;

import java.util.logging.Logger;

import org.apache.http.HttpHost;

import org.apache.lucene.index.IndexWriter;
import org.apache.lucene.store.BlockCompressedInputStreamBase;
import org.apache.lucene.store.IndexOutput;
import org.apache.lucene.store.IndexInput;
import org.apache.lucene.store.FSDirectory;
import org.apache.lucene.store.Directory;
import org.apache.lucene.store.RAMDirectory;

import ru.yandex.cache.CopyEntry;
import ru.yandex.cache.DBCache;
import ru.yandex.cache.lmdb.LMDBCache;
import ru.yandex.cache.sqlite.SqliteCache;
import ru.yandex.cache.sqlite.SqliteCache2;

import ru.yandex.http.config.DnsConfigBuilder;

import ru.yandex.http.util.server.AutoRegisterRequestStaterConfigBuilder;
import ru.yandex.http.util.server.AutoRegisterStatus;
import ru.yandex.http.util.server.HttpServer;
import ru.yandex.http.util.server.ImmutableBaseServerConfig;
import ru.yandex.http.util.server.ImmutableLimitersConfig;
import ru.yandex.http.util.server.LimitersConfigBuilder;

import ru.yandex.logger.HandlersManager;
import ru.yandex.logger.ImmutableLoggersConfig;
import ru.yandex.logger.LoggersConfigBuilder;
import ru.yandex.logger.PrefixedLogger;
import ru.yandex.logger.SystemErrManager;

import ru.yandex.msearch.collector.outergroup.OuterGroupFunctionFactory;
import ru.yandex.msearch.util.IOScheduler;
import ru.yandex.msearch.util.JavaAllocator;

import ru.yandex.parser.config.IniConfig;

import ru.yandex.stater.ImmutableStatersConfig;
import ru.yandex.stater.StatersConfigBuilder;

import ru.yandex.unsafe.NativeMemory2;
import ru.yandex.unsafe.NativeMemory2.NativeMemoryAllocator;

import ru.yandex.util.filesystem.DeletingFileVisitor;

public class Daemon implements Closeable, ShardInitListener {
    private static final NativeMemoryAllocator ALLOCATOR =
        NativeMemoryAllocator.get("ConvertTemp");

    private final Indexer i;
    private final HTTPServer h;
    private final DumpServer d;
    private final HttpSearchServer httpSearchServer;
    private final DatabaseManager databaseManager;
    private final PrefixedLogger logger;
    private final DBCache ssdCache;
    private final RandomAccessFile statusFile;

    public Daemon(final Config config) throws Exception {
        statusFile = new RandomAccessFile(config.statusFile(), "rw");
        writeStatus("STARTING CLEANUP");
        HandlersManager handlersManager = new HandlersManager();
        logger =
            config.fullLog().asterisk().build().buildPrefixed(handlersManager);
        logger.severe("Starting lucene");
        Logger accessLogger =
            config.accessLog().asterisk().build().build(handlersManager);
        Logger indexLogger = config.indexLog().build(handlersManager);
        PrefixedLogger indexFullLogger;
        if (config.indexFullLog() == config.fullLog()) {
            indexFullLogger = logger;
        } else {
            indexFullLogger = config.indexFullLog().asterisk().build().buildPrefixed(handlersManager);
        }

        Logger errorLogger = config.errorLog().build(handlersManager);
        registerErrorLogger(errorLogger);
        if (config.infoStream()) {
            IndexWriter.setDefaultInfoStream(System.err);
        }

        OuterGroupFunctionFactory outerGroupFunctionFactory =
            new OuterGroupFunctionFactory(
                config.xurlsRegexFile(),
                logger);

        if (config.ioThreads() > 0) {
            IOScheduler.init(
                config.ioThreads(),
                config.ioPriorityDivisor(),
                config.ioSleepingScheduler());
        }
        // TODO ssdCache should support multidatabases
        if (config.ssdCachePath() != null) {
            writeStatus("STARTING SSD_CACHE_INIT");
            String fileName = config.ssdCachePath().getCanonicalPath();
            File parent = config.ssdCachePath().getParentFile();
            if (parent != null) {
                parent.mkdirs();
            }
            System.err.println("SSD cache filename: " + fileName);
/*
            boolean sqlite = false;
            try (FileInputStream is = new FileInputStream(fileName)) {
                if ((char) is.read() == 'S') {
                    sqlite = true;
                }
            } catch (java.io.FileNotFoundException ign) {
            }
            if (sqlite) {
                convertCache(
                    indexPath.getCanonicalPath(),
                    fileName,
                    config.ssdCacheSize());
            }
*/
            if (new File(fileName + ".v2").exists()) {
                try {
                    // TODO this is not allowed to use config.indexPath()
                    convertSqliteCache2To1(
                        config.indexPath().getCanonicalPath(),
                        fileName,
                        config.ssdCacheSize());
                } catch (Exception e) {
                    //drop old cache
                    new File(fileName + ".v2").delete();
                    new File(fileName + ".v2-shm").delete();
                    new File(fileName + ".v2-wal").delete();
                }
            }
//            ssdCache = new SqliteCache2(
//                fileName,
//                config.ssdCacheSize());
            ssdCache = new SqliteCache(fileName, config.ssdCacheSize());
//            ssdCache = new LMDBCache(fileName, config.ssdCacheSize());
            dropRemovedFiles();
        } else {
            ssdCache = null;
        }
        ru.yandex.msearch.util.Compress.fadviseEnabled(config.useFadvise());
        BlockCompressedInputStreamBase.setSsdCache(ssdCache);
        BlockCompressedInputStreamBase.setCacheCapacity(config.cacheCapacity());
        BlockCompressedInputStreamBase.setCompressedCacheCapacity(
            config.compressedCacheCapacity());
        BlockCompressedInputStreamBase.dynamicShrink(
            config.compressedCacheDynamicShrink(),
            config.compressedCacheCapacity(),
            config.dynamicCacheMinMemoryFree());
        writeStatus("STARTING INDEX_INIT");
        databaseManager = new DatabaseManager(this, config, logger, indexLogger);
        writeStatus("STARTING SERVER_INIT");
        i = new Indexer(
            databaseManager,
            config,
            new ImmutableBaseServerConfig(
                config.indexer(),
                new DnsConfigBuilder().build(),
                config.tvm2ServiceConfig(),
                new ImmutableLoggersConfig(
                    new LoggersConfigBuilder()
                        .loggers(config.indexFullLog())
                        .accessLoggers(config.indexAccessLog()),
                    handlersManager),
                new StatersConfigBuilder()
                    .staters(config.indexRequestsStaters())
                    .build(),
                config.auths(),
                new ImmutableLimitersConfig(
                    new LimitersConfigBuilder().
                        limiters(config.indexLimiters())),
                config.indexerGolovanPanelConfig(),
                new AutoRegisterRequestStaterConfigBuilder()
                    .status(AutoRegisterStatus.DISABLED)
                    .build()));
        h = new HTTPServer(
            databaseManager,
            config,
            outerGroupFunctionFactory,
            logger, accessLogger);
        d = new DumpServer(databaseManager, config, logger, accessLogger);
        httpSearchServer = new HttpSearchServer(
            databaseManager,
            config,
            outerGroupFunctionFactory,
            new ImmutableBaseServerConfig(
                config.search(),
                new DnsConfigBuilder().build(),
                config.tvm2ServiceConfig(),
                new ImmutableLoggersConfig(
                    new LoggersConfigBuilder()
                        .loggers(config.fullLog())
                        .accessLoggers(config.accessLog()),
                    handlersManager),
                new ImmutableStatersConfig(
                    new StatersConfigBuilder().
                        staters(config.searchRequestsStaters())),
                config.auths(),
                new ImmutableLimitersConfig(
                    new LimitersConfigBuilder().
                        limiters(config.searchLimiters())),
                config.searcherGolovanPanelConfig(),
                new AutoRegisterRequestStaterConfigBuilder()
                    .status(AutoRegisterStatus.DISABLED)
                    .build()));

        //cross register loggers
        httpSearchServer.registerLoggerForLogrotate(errorLogger);
        httpSearchServer.registerLoggerForLogrotate(indexLogger);
        httpSearchServer.registerLoggerForLogrotate(i.server());

        i.server().registerLoggerForLogrotate(errorLogger);
        i.server().registerLoggerForLogrotate(indexLogger);
        i.server().registerLoggerForLogrotate(httpSearchServer);


        httpSearchServer.start();
        System.err.println(handlersManager.handlers());
        writeStatus("STARTED");
    }

    private void dropRemovedFiles() throws IOException {
        String[] cachedFiles = ssdCache.uniqFiles();
        for (String file: cachedFiles) {
            System.err.println("Iterating cache files: " + file);
            if (!(new File(file).exists())) {
                System.err.println("Dropping cache for file: "
                    + file);
                ssdCache.removePrefix(file + ':', true);
            }
        }
    }

    public void writeStatus(final String status) throws IOException {
        byte[] data = (status + "\n").getBytes("UTF-8");
        statusFile.seek(0);
        statusFile.write(data);
        statusFile.setLength(data.length);
    }

    @Override
    public void shardInit() throws IOException {
        statusFile.setLength(statusFile.length());
    }

    private void convertCache(
        final String indexPath,
        final String cachePath,
        final long size)
        throws Exception
    {
        System.err.println("Converting cache");
        String tempCachePath = indexPath + "/cache-convert-temp/tmp.db";
        new File(tempCachePath).getParentFile().mkdirs();
        try (
            LMDBCache tempCache =
                new LMDBCache(tempCachePath, size);
            SqliteCache oldCache = new SqliteCache(cachePath, size))
        {
            tempCache.copyFrom(oldCache);
        }
        File old = new File(cachePath);
        old.delete();
        new File(cachePath + "-shm").delete();
        new File(cachePath + "-wal").delete();
        for (String suf: new String[]{"", "-lock"}) {
            Path to = new File(cachePath + suf).toPath();
            Path from = new File(tempCachePath + suf).toPath();
            java.nio.file.Files.move(
                from,
                to,
                java.nio.file.StandardCopyOption.REPLACE_EXISTING);
        }
        System.err.println("Converting cache done");
    }

    private void convertSqliteCache2(
        final String indexPath,
        final String cachePath,
        final long size)
        throws Exception
    {
        System.err.println("Converting cache");
        String tempCachePath = indexPath + "/cache-convert-temp/tmp.db";
        new File(tempCachePath).getParentFile().mkdirs();
        byte[] data = new byte[0];
        try (
            DataOutputStream os = new DataOutputStream(
                new BufferedOutputStream(
                    new FileOutputStream(tempCachePath)));
            SqliteCache2 oldCache = new SqliteCache2(cachePath, size))
        {
            for (
                CopyEntry one = oldCache.getOne();
                one != null;
                one = oldCache.getOne())
            {
                os.writeUTF(one.key());
                os.writeInt(one.compressedSize());
                os.writeInt(one.decompressedSize());
                os.writeInt(one.accessFreq());
                if (data.length < one.compressedSize()) {
                    data = new byte[one.compressedSize() << 1];
                }
                NativeMemory2.unboxedRead(
                    one.address(),
                    data,
                    0,
                    one.compressedSize());
                os.write(data, 0, one.compressedSize());
                one.close();
            }
            os.writeUTF("\"");
        }
        File old = new File(cachePath + ".v2");
        old.delete();
        new File(cachePath + ".v2-shm").delete();
        new File(cachePath + ".v2-wal").delete();
        NativeMemory2 tempBuf = ALLOCATOR.alloc(1024 * 1024);
        try (
            DataInputStream is = new DataInputStream(
                new BufferedInputStream(
                    new FileInputStream(tempCachePath)));
            LMDBCache tempCache =
                new LMDBCache(cachePath, size))
        {
            while (true) {
                String key = is.readUTF();
                if (key.equals("\"")) {
                    break;
                }
                int compressedSize = is.readInt();
                int decompressedSize = is.readInt();
                int accessFreq = is.readInt();
                if (data.length < compressedSize) {
                    data = new byte[compressedSize << 1];
                }
                is.read(data, 0, compressedSize);
                tempBuf.write(0, data, 0, compressedSize);
                tempCache.put(
                    key,
                    tempBuf.address(),
                    compressedSize,
                    decompressedSize,
                    accessFreq,
                    true);
            }
        } finally {
            tempBuf.free();
        }
        new File(tempCachePath).delete();
        System.err.println("Converting cache done");
    }

    private void convertSqliteCache2To1(
        final String indexPath,
        final String cachePath,
        final long size)
        throws Exception
    {
        System.err.println("Converting cache");
        String tempCachePath = indexPath + "/cache-convert-temp/tmp.db";
        new File(tempCachePath).getParentFile().mkdirs();
        byte[] data = new byte[0];
        try (
            DataOutputStream os = new DataOutputStream(
                new BufferedOutputStream(
                    new FileOutputStream(tempCachePath)));
            SqliteCache2 oldCache = new SqliteCache2(cachePath, size))
        {
            for (
                CopyEntry one = oldCache.getOne();
                one != null;
                one = oldCache.getOne())
            {
                os.writeUTF(one.key());
                os.writeInt(one.compressedSize());
                os.writeInt(one.decompressedSize());
                os.writeInt(one.accessFreq());
                if (data.length < one.compressedSize()) {
                    data = new byte[one.compressedSize() << 1];
                }
                NativeMemory2.unboxedRead(
                    one.address(),
                    data,
                    0,
                    one.compressedSize());
                os.write(data, 0, one.compressedSize());
                one.close();
                oldCache.remove(one.key());
            }
            os.writeUTF("\"");
        }
        File old = new File(cachePath + ".v2");
        old.delete();
        new File(cachePath + ".v2-shm").delete();
        new File(cachePath + ".v2-wal").delete();
        NativeMemory2 tempBuf = ALLOCATOR.alloc(1024 * 1024);
        try (
            DataInputStream is = new DataInputStream(
                new BufferedInputStream(
                    new FileInputStream(tempCachePath)));
            SqliteCache tempCache =
                new SqliteCache(cachePath, size))
        {
            while (true) {
                String key = is.readUTF();
                if (key.equals("\"")) {
                    break;
                }
                int compressedSize = is.readInt();
                int decompressedSize = is.readInt();
                int accessFreq = is.readInt();
                if (data.length < compressedSize) {
                    data = new byte[compressedSize << 1];
                }
                is.read(data, 0, compressedSize);
                tempBuf.write(0, data, 0, compressedSize);
                tempCache.put(
                    key,
                    tempBuf.address(),
                    compressedSize,
                    decompressedSize,
                    accessFreq,
                    true);
            }
        } finally {
            tempBuf.free();
        }
        new File(tempCachePath).delete();
        System.err.println("Converting cache done");
    }


    private void registerErrorLogger(final Logger logger) {
        SystemErrManager.setErr(logger);
    }

    private void deregisterErrorLogger() {
        SystemErrManager.unsetErr();
    }

    @Override
    public void close() {
        logger.severe("Shutting down");
        try {
            databaseManager.stopJobs();
            httpSearchServer.close();
            h.close();
            d.close();
            i.close();
            databaseManager.close();
            NativeMemory2.printStats();
            logger.severe("Freeing cache");
            org.apache.lucene.store.BlockCompressedInputStreamBase.waitFree();
            logger.severe("Cache freed");
            NativeMemory2.printStats();
            JavaAllocator.printStats();
            org.apache.lucene.store.BlockCompressedInputStreamBase
                .printCacheInputs();
            if (ssdCache != null) {
                ssdCache.close();
            }
        } catch (Exception e){};
        IndexWriter.setDefaultInfoStream(null);
        deregisterErrorLogger();
    }

    public int searchPort() {
        return h.port();
    }

    public int dumpPort() {
        return d.port();
    }

    public int searchServerPort() throws IOException {
        return httpSearchServer.port();
    }

    public HttpHost searchServerHost() throws IOException {
        return httpSearchServer.host();
    }

    public HttpServer<?, ?> httpSearchServer() {
        return httpSearchServer;
    }

    public int jsonServerPort() throws IOException {
        return i.jsonServerPort();
    }

    public HttpHost jsonServerHost() throws IOException {
        return i.server().host();
    }

    public HttpServer<?, ?> jsonIndexerServer() {
        return i.server();
    }

    public Index index() {
        if (databaseManager.databases().size() == 1) {
            return databaseManager.databases().entrySet().iterator().next().getValue();
        }

        throw new RuntimeException("Deprecated method");
    }

    public Index defaultDatabase() {
        return databaseManager.databases().get(DatabaseManager.DEFAULT_DATABASE);
    }

    private static void forceJit() throws IOException {
        RAMDirectory ramDir = new RAMDirectory();
        IndexOutput out = ramDir.createOutput( "fake" );
        for( int i = 0; i < 10000000; i++ )
        {
            out.writeVInt( i );
            out.writeVLong( i );
        }
        out.close();
        IndexInput in = ramDir.openInput( "fake" );
        for( int i = 0; i < 10000000; i++ )
        {
            in.readVInt();
            in.readVLong();
        }
        in.close();
    }

    public static Daemon createDaemon(final String config)
        throws Exception
    {
        forceJit();
        final Daemon daemon =
            new Daemon(new Config(new IniConfig(Paths.get(config))));

        Runtime.getRuntime().addShutdownHook(new Thread() {
            @Override
            public void run() {
                daemon.close();
            }
        });
        return daemon;
    }

    public static void main(final String... argv) throws Exception {
        String usage = "Daemon -config <file>";

        try
        {
            String config = null;
            if( argv.length == 0 )
            {
                System.err.println("Usage: " + usage);
                System.out.println( "###failed_to_start###" );
                return;
            }

            for( int i = 0; i < argv.length; i++ )
            {
                if( argv[i].equals("-config") )
                {                 // parse -index option
                config = argv[++i];
                }
                else if( i != argv.length-1 )
                {
                System.err.println("Usage: " + usage);
                System.out.println( "###failed_to_start###" );
                return;
                }
            }

            if (config == null) {
                System.err.println("Usage: " + usage);
                System.out.println( "###failed_to_start###" );
                return;
            }

            createDaemon(config);
            System.out.println("###started###");
        } catch (Exception e) {
            System.out.println("###failed_to_start###");
            System.out.println(e.toString());
            e.printStackTrace();
            Runtime.getRuntime().exit( -1 );
        }
    }
}

