package ru.yandex.msearch;

import java.io.File;
import java.io.IOException;
import java.nio.file.FileVisitResult;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.attribute.BasicFileAttributes;
import java.util.ArrayList;
import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.function.Function;
import java.util.logging.Level;
import java.util.logging.Logger;

import ru.yandex.http.util.server.AbstractHttpServer;
import ru.yandex.logger.PrefixedLogger;
import ru.yandex.msearch.config.DatabaseConfig;
import ru.yandex.parser.uri.CgiParams;
import ru.yandex.parser.uri.CgiParamsBase;
import ru.yandex.search.prefix.Prefix;
import ru.yandex.util.filesystem.DeletingFileVisitor;

public class DatabaseManager {
    public static final String DEFAULT_DATABASE = "default";
    public static final String DB_CGI_PARAM = "db";
    private static final String DROP_INDEX = "drop-index";
    private final Map<String, Index> databases;
    private final Map<String, List<Index>> databasesByService;
    private final Config daemonConfig;
    private final PrefixedLogger logger;

    public DatabaseManager(
        final Daemon daemon,
        final Config config,
        final PrefixedLogger logger,
        final Logger indexLogger)
        throws Exception
    {
        this.daemonConfig = config;
        this.logger = logger;
        Map<String, Index> databases = new LinkedHashMap<>();
        Map<String, List<Index>> databasesByService = new LinkedHashMap<>();
        for (Map.Entry<String, DatabaseConfig> dbConfigEntry: config.databasesConfigs().entrySet()) {
            Index index = initIndex(
                daemon,
                dbConfigEntry.getKey(),
                config,
                dbConfigEntry.getValue(),
                logger,
                indexLogger);

            databases.put(dbConfigEntry.getKey(), index);
            if (dbConfigEntry.getValue().services().size() > 0) {
                for (String service: dbConfigEntry.getValue().services()) {
                    databasesByService.computeIfAbsent(service, (x) -> new ArrayList<>()).add(index);
                }
            } else {
                databasesByService.computeIfAbsent(DEFAULT_DATABASE, (x) -> new ArrayList<>()).add(index);
            }

        }

        this.databases = Collections.unmodifiableMap(databases);
        this.databasesByService = Collections.unmodifiableMap(databasesByService);
        logger.info("Databases " + databases.keySet());
        indexLogger.info("Databases " + databases.keySet());
    }

    private Index initIndex(
        final Daemon daemon,
        final String name,
        final Config daemonConfig,
        final DatabaseConfig dbConfig,
        final PrefixedLogger logger,
        final Logger indexLogger)
        throws Exception
    {
        final File dropIndex = new File(dbConfig.indexPath(), DROP_INDEX);
        if (dropIndex.exists() && dropIndex.isFile()) {
            dropIndex(name, dbConfig, dropIndex);
        }

        logger.info("Starting database " + dbConfig.name());
        indexLogger.info("Starting database " + dbConfig.name());
        return new Index(dbConfig.indexPath(), daemonConfig, dbConfig, logger, indexLogger, daemon);
    }

    public Map<String, Index> databases() {
        return databases;
    }

    private void dropIndex(final String name, final DatabaseConfig dbConfig, final File dropIndex) throws IOException {
        System.err.println("WARNING: drop-index file found. Deleting index. " + name);
        Files.walkFileTree(
            dbConfig.indexPath().toPath(),
            new DeletingFileVisitor() {
                @Override
                public FileVisitResult visitFile(
                    final Path file,
                    final BasicFileAttributes attrs)
                    throws IOException
                {
                    if (file.endsWith(DROP_INDEX)) {
                        return FileVisitResult.CONTINUE;
                    } else {
                        return super.visitFile(file, attrs);
                    }
                }
                @Override
                public FileVisitResult postVisitDirectory(
                    final Path dir,
                    final IOException e)
                    throws IOException
                {
                    if (dir.equals(dbConfig.indexPath().toPath())) {
                        dropIndex.delete();
                    }
                    return super.postVisitDirectory(dir, e);
                }
            });
    }

    public Index index(final String name) {
        return databases.get(name);
    }

    public Index database(final Map<String, String> params) {
        String dbName = params.getOrDefault(DB_CGI_PARAM, DEFAULT_DATABASE);
        if (dbName != null) {
            return databases.get(dbName);
        }

        return null;
    }

    public Index database(final CgiParamsBase params, final String service) {
        String dbName = params.getOrNull(DB_CGI_PARAM);
        if (dbName != null) {
            return databases.get(dbName);
        }

        if (service == QueueShard.DEFAULT_SERVICE || QueueShard.DEFAULT_SERVICE.equalsIgnoreCase(service)) {
            return databases.get(DEFAULT_DATABASE);
        }

        List<Index> indexes = databasesByService.get(service);
        if (indexes == null) {
            return databases.get(DEFAULT_DATABASE);
        }

        if (indexes.size() != 1) {
            return null;
        }

        return indexes.get(0);
    }

    public Index index() {
        return databases.get(DEFAULT_DATABASE);
    }

    public <E extends Throwable> Index indexOrException(
        final CgiParams params,
        final Function<String, E> func)
        throws E
    {
        String name = params.getString("db", DEFAULT_DATABASE);
        Index database = databases.get(name);
        if (database == null) {
            throw func.apply(name);
        }

        return database;
    }

    public Index index(final CgiParams params) {
        return databases.get(params.getString(DB_CGI_PARAM, DEFAULT_DATABASE));
    }

    public Config daemonConfig() {
        return daemonConfig;
    }

    public boolean isDirty(final QueueShard shard) {
        List<Index> indexes = indexByService(shard.service());
        boolean dirty = false;
        for (Index index: indexes) {
            dirty |= index.isDirty(shard);
        }

        return dirty;
    }

    // TODO add interning ?
    public List<Index> indexByService(final String service) {
        List<Index> result = databasesByService.get(service);
        if (result == null) {
            result =  databasesByService.get(DEFAULT_DATABASE);
        }

        return result;
    }

    public long queueId(
        final QueueShard shard,
        final boolean checkCopyness)
        throws IOException
    {
        return queueId(shard, null, checkCopyness);
    }

    public long queueId(
        final QueueShard shard,
        final Prefix prefix,
        final boolean checkCopyness)
        throws IOException
    {
        List<Index> indexes = indexByService(shard.service());
        if (checkCopyness) {
            for (Index index: indexes) {
                index.checkShardCopied(shard);
            }
        }

        if (indexes.size() == 0) {
            throw new IOException("Service is not served by this backend " + shard.service());
        }

        if (indexes.size() == 1) {
            if (prefix == null) {
                return indexes.get(0).queueId(shard, checkCopyness);
            } else {
                return indexes.get(0).queueId(shard, prefix, checkCopyness);
            }
        }

        if (daemonConfig.returnMaxQueueId()) {
            return maxQueueId(indexes, prefix, shard, checkCopyness);
        } else {
            return minQueueId(indexes, prefix, shard, checkCopyness);
        }
    }

    public long minQueueId(
        final List<Index> indexList,
        final Prefix prefix,
        final QueueShard shard,
        final boolean checkCopyness)
        throws IOException
    {
        long minQueueId = Long.MAX_VALUE;
        for (Index index: indexList) {
            long queueId;
            if (prefix == null) {
                queueId = index.queueId(shard, checkCopyness);
            }  else {
                queueId = index.queueId(shard, prefix, checkCopyness);
            }
            if (queueId != -1 && queueId < minQueueId) {
                minQueueId = queueId;
            }
        }
        if (minQueueId == Long.MAX_VALUE) {
            minQueueId = -1;
        }
        return minQueueId;
    }


    public long maxQueueId(
        final List<Index> indexList,
        final Prefix prefix,
        final QueueShard shard,
        final boolean checkCopyness) throws IOException
    {
        long maxQueueId = -1;
        for (Index index: indexList) {
            long queueId;
            if (prefix == null) {
                queueId = index.queueId(shard, checkCopyness);
            } else {
                queueId = index.queueId(shard, prefix, checkCopyness);
            }

            if (queueId > maxQueueId) {
                maxQueueId = queueId;
            }
        }
        return maxQueueId;
    }

    public void registerIndexStaters(final AbstractHttpServer<?, ?> server) {
        for (Index index: databases.values()) {
            server.registerStater(index.getJobsManager().stater());
            server.registerStater(index.stater());
            server.registerStater(index.fieldsCache());
        }
    }

    public void stopJobs() throws IOException {
        IOException ioException = null;
        for (Map.Entry<String, Index> entry: databases.entrySet()) {
            try {
                entry.getValue().getJobsManager().stopJobs();
            } catch (IOException ioe) {
                if (ioException != null) {
                    logger.log(
                        Level.WARNING,
                        "Failed to stop jobs for database " + entry.getKey(),
                        ioe);
                }
                ioException = new IOException("Failed to stop jobs for database ", ioe);
            }
        }

        if (ioException != null) {
            throw ioException;
        }
    }

    public void close() {
        for (Map.Entry<String, Index> entry: databases.entrySet()) {
            entry.getValue().close();
        }
    }
}
