package ru.yandex.msearch;

import java.util.function.Function;

import org.apache.lucene.analysis.KeywordAnalyzer;
import org.apache.lucene.analysis.PerFieldAnalyzerWrapper;
import org.apache.lucene.analysis.WhitespaceAnalyzer;

import ru.yandex.analyzer.CustomAnalyzer;
import ru.yandex.analyzer.FilterChainFactory;
import ru.yandex.msearch.config.DatabaseConfig;

public enum DefaultAnalyzerFactory implements Function<DatabaseConfig, PrefixingAnalyzerWrapper> {
    SEARCH {
        @Override
        public PrefixingAnalyzerWrapper apply(final DatabaseConfig config) {
            return DefaultAnalyzerFactory.createAnalyzer(config, FieldConfig::searchFilters);
        }
    },
    INDEX {
        @Override
        public PrefixingAnalyzerWrapper apply(final DatabaseConfig config) {
            return DefaultAnalyzerFactory.createAnalyzer(config, FieldConfig::indexFilters);
        }
    };

    private static PrefixingAnalyzerWrapper createAnalyzer(
            final DatabaseConfig daemonConfig,
            final Function<FieldConfig, FilterChainFactory> factory) {
        PerFieldAnalyzerWrapper pAnalyzer = new PerFieldAnalyzerWrapper(new WhitespaceAnalyzer());
        for (String field : daemonConfig.knownFields()) {
            FieldConfig config = daemonConfig.fieldConfig(field);
            if (config.analyze()) {
                pAnalyzer.addAnalyzer(field, new CustomAnalyzer(config, factory.apply(config)));
            } else if (config.index()) {
                pAnalyzer.addAnalyzer(field, new KeywordAnalyzer());
            }
        }

        PrefixingAnalyzerWrapper analyzer = new PrefixingAnalyzerWrapper(pAnalyzer);
        for (String field : daemonConfig.knownFields()) {
            FieldConfig config = daemonConfig.fieldConfig(field);
            if ((config.analyze() || config.index()) && !config.prefixed()) {
                analyzer.addSkipField(field);
            }
        }
        analyzer.setSeparator('#');
        return analyzer;
    }
}
