//package org.apache.lucene.index;
package ru.yandex.msearch;

/**
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

import org.apache.lucene.document.Document;
import org.apache.lucene.document.FieldSelector;
import org.apache.lucene.index.FilterIndexReader;
import org.apache.lucene.index.IndexReader;
import org.apache.lucene.index.IndexReader.AtomicReaderContext;
import org.apache.lucene.index.IndexReader.CompositeReaderContext;
import org.apache.lucene.index.IndexReader.ReaderContext;
import org.apache.lucene.index.SegmentReader;
import org.apache.lucene.search.IndexSearcher;
import org.apache.lucene.search.Query;
import org.apache.lucene.search.QueryProducer;
import org.apache.lucene.search.Scorer;
import org.apache.lucene.search.Weight;
import org.apache.lucene.store.Directory;
import org.apache.lucene.util.BitVector;
import org.apache.lucene.util.Bits;
import org.apache.lucene.util.BytesRef;
import org.apache.lucene.util.MapBackedSet;
import org.apache.lucene.util.ReaderUtil;

import java.io.IOException;
import java.io.PrintWriter;
import java.io.StringWriter;
import java.io.Writer;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.LinkedList;
import java.util.Map;
import java.util.Comparator;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.Callable;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.Future;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;

import ru.yandex.concurrent.NamedThreadFactory;

import ru.yandex.util.unicode.ByteSequence;

public class DeleteHandlingIndexReader extends FilterIndexReader
{
private static final boolean DEBUG = false;
ReaderContext topReaderContext;
AtomicReaderContext[] origLeaves;
AtomicReaderContext[] filterLeaves;
DeleteHandlingReaderLeave[] subReaders;
private final boolean cloned;

private final static int threads = Runtime.getRuntime().availableProcessors();
private final static ThreadPoolExecutor parallelExecutor =
        new ThreadPoolExecutor(threads,
            threads,
            1,
            TimeUnit.HOURS,
            new ArrayBlockingQueue<Runnable>(threads),
            new NamedThreadFactory("Deletor-"),
            new ThreadPoolExecutor.CallerRunsPolicy()
        );


    public DeleteHandlingIndexReader(IndexReader in) throws IOException
    {
        super(in);
        this.cloned = false;
        //Ensure we have the first reference to this reader.
        while( in.getRefCount() < 1 ) in.incRef();
        while( in.getRefCount() > 1 ) in.decRef();
        origLeaves = ReaderUtil.leaves(in.getTopReaderContext());
        filterLeaves = new AtomicReaderContext[origLeaves.length];
        subReaders = new DeleteHandlingReaderLeave[origLeaves.length];
        topReaderContext = new CompositeReaderContext(this, filterLeaves, filterLeaves);
        for( int i = 0; i < origLeaves.length; i++ )
        {
//            origLeaves[i].reader.incRef();
            AtomicReaderContext origCtx = origLeaves[i];
            subReaders[i] = new DeleteHandlingReaderLeave(origCtx.reader);
            filterLeaves[i] = new AtomicReaderContext(topReaderContext, subReaders[i], i, origCtx.docBase, i, origCtx.docBase);
        }
    }

    private DeleteHandlingIndexReader(IndexReader in, boolean clone/*ignored*/)
    {
        super(in);
        this.cloned = true;
        in.incRef();
    }

    public IndexReader origReader()
    {
        return in;
    }

    @Override
    public int numDocs() {
        int count = 0;
        for (DeleteHandlingReaderLeave subReader : subReaders) {
            count += subReader.numDocs();
        }
        return count;
    }

    @Override
    public long numDocsLong() {
        long count = 0;
        for (DeleteHandlingReaderLeave subReader : subReaders) {
            count += subReader.numDocsLong();
        }
        return count;
    }

    public DeleteHandlingIndexReader reopen()
    {
//        for( DeleteHandlingReaderLeave reader : subReaders )
//        {
//            if( reader.hasDeletes() )
//            {
                return clone();
//            }
//        }
//        return this;
    }

    @Override
    public void addReaderFinishedListener(ReaderFinishedListener listener) {
        readerFinishedListeners.add(listener);
//        in.addReaderFinishedListener(listener);
    }

    @Override
    public void removeReaderFinishedListener(ReaderFinishedListener listener) {
        readerFinishedListeners.remove(listener);
//        in.removeReaderFinishedListener(listener);
    }

    @Override
    public void incRef() {
//        System.err.println(exception(new Exception("IncRef: " + this.toString())));
        super.incRef();
//        in.incRef();
    }

//    @Override
//    public int getRefCount() {
//        return in.getRefCount();
//    }

    @Override
    public void decRef() throws IOException {
//        System.err.println(exception(new Exception("DecRef: " + this.toString())));
        super.decRef();
//        in.decRef();
    }

    public DeleteHandlingIndexReader clone()
    {
        DeleteHandlingIndexReader clone = new DeleteHandlingIndexReader(in,true);
        clone.origLeaves = origLeaves;
        clone.filterLeaves = new AtomicReaderContext[filterLeaves.length];
        clone.subReaders = new DeleteHandlingReaderLeave[subReaders.length];
        clone.topReaderContext =  new CompositeReaderContext(clone, clone.filterLeaves, clone.filterLeaves);
        for( int i = 0; i < subReaders.length; i++ )
        {
            clone.subReaders[i] = subReaders[i].clone();
            AtomicReaderContext ctx = filterLeaves[i];
            clone.filterLeaves[i] = new AtomicReaderContext(clone.topReaderContext, clone.subReaders[i], i, ctx.docBase, i, ctx.docBase);
            if( clone.subReaders[i] == subReaders[i] )
            {
//                clone.filterLeaves[i].reader.incRef();
            }
        }
        return clone;
    }

  @Override
  protected synchronized void doClose() throws IOException {
//    System.err.println( "DeleteHandlingIndexreader.doClose.in.getRefCount() = " + in.getRefCount() );
    in.decRef();
/*
    IOException ioe = null;
    for (int i = 0; i < subReaders.length; i++) {
      // try to close each reader, even if an exception is thrown
      try {
        subReaders[i].decRef();
      } catch (IOException e) {
        if (ioe == null) ioe = e;
      }
    }
    // throw the first exception
    if (ioe != null) throw ioe;
*/
  }


  public IndexReader[] getSequentialSubReaders() {
    return subReaders;
  }

  public ReaderContext getTopReaderContext() {
    return topReaderContext;
  }

    public void deleteDocuments(QueryProducer query) throws IOException {
        BytesRef queryKey = new BytesRef(4);
        queryKey.writeInt(System.identityHashCode(query));
        for (int i = 0; i < filterLeaves.length; i++) {
            deleteDocuments(query, queryKey, filterLeaves[i]);
        }
    }

    public void deleteDocumentsBatch(
        final Collection<QueryProducer> queries)
        throws IOException
    {
        for (int i = 0; i < filterLeaves.length; i++)
        {
            deleteDocuments(queries, filterLeaves[i]);
        }
    }

    public void deleteDocuments(
        final Collection<QueryProducer> queries)
        throws IOException
    {
        if (DEBUG) {
            System.err.println("DeleteHandlingIndexReader<"+this
                +">.transfering deletes: size=" + queries.size());
        }
        LinkedList<Future> futures = new LinkedList<Future>();
        try {
            ArrayList<QueryProducer> batch = new ArrayList<>();
            for (QueryProducer query : queries) {
                if (DEBUG) {
                    System.err.println("DeleteHandlingIndexReader<"+this
                        +">. transfer query: " + query);
                }
                batch.add(query);
                if (batch.size() > 100) {
//                System.err.println("eleteHandlingIndexReader<"+this+">.query: " + query);
//                final Query q = query;
                    final ArrayList<QueryProducer> b = new ArrayList<>(batch);
                    futures.add(
                        parallelExecutor.submit(
                            new Callable<Void>() {
                                @Override
                                public Void call() throws IOException {
                                    deleteDocumentsBatch(b);
                                    return null;
                                }
                            }));
                    batch.clear();
                }
            }
            if (batch.size() > 0) {
                deleteDocumentsBatch(batch);
            }
            for (Future fut : futures) {
                fut.get();
            }
        } catch (Exception e) {
            throw new IOException(e);
        }
    }

    public void deleteDocuments(
        final QueryProducer query,
        final BytesRef queryKey,
        final AtomicReaderContext readerContext)
        throws IOException
    {
        final DeleteHandlingReaderLeave reader =
            (DeleteHandlingReaderLeave)readerContext.reader;
        reader.deleteDocuments(query, queryKey);
    }

    public void deleteDocuments(final Collection<QueryProducer> queries,
        final AtomicReaderContext readerContext)
        throws IOException
    {
        final DeleteHandlingReaderLeave reader =
            (DeleteHandlingReaderLeave)readerContext.reader;
        reader.deleteDocuments(queries);
    }


/*
  public void deleteDocuments(Query query, AtomicReaderContext readerContext) throws IOException
  {
    final IndexReader reader = readerContext.reader;
    IndexSearcher searcher = new IndexSearcher(reader);
    try {
        Weight weight = query.weight(searcher);
        Scorer scorer = weight.scorer((AtomicReaderContext)searcher.getTopReaderContext(), Weight.ScorerContext.def());
        if (scorer != null) {
            while(true)  {
                int doc = scorer.nextDoc();
                if( doc == Scorer.NO_MORE_DOCS ) break;
//                System.err.println("DeleteHandlingIndexLeave<"+reader+">.deleteDocuments: " + query + ", docId=" + doc);
                reader.deleteDocument(doc);
            }
        }
    } finally {
      searcher.close();
    }
  }
*/
    public static String exception(Throwable aThrowable)
    {
        final Writer result = new StringWriter();
        final PrintWriter printWriter = new PrintWriter(result);
        aThrowable.printStackTrace(printWriter);
        return result.toString();
    }

  @Override
  public String toString() {
    final StringBuilder buffer = new StringBuilder("DeletesHandlingIndexReader(");
    buffer.append(in);
    buffer.append(",cloned=");
    buffer.append(cloned);
    buffer.append(')');
    buffer.append('<');
    buffer.append(this.hashCode());
    buffer.append('>');
    return buffer.toString();
  }

    private static class DeleteHandlingReaderLeave extends FilterIndexReader
    {
        private int deletesCount;
        private Bits deletes;
        private BitsOverlay deletesOverlay;
        private final IndexReader in;
        private boolean cloned;
        private DeleteHandlingReaderLeave[] clones =
            new DeleteHandlingReaderLeave[0];

        public DeleteHandlingReaderLeave(final IndexReader in)
        {
            this(in, in.getDeletedDocs());
            this.cloned = false;
        }

        public DeleteHandlingReaderLeave(final IndexReader in, Bits deletes)
        {
            super(in);
            this.in = in;
            this.cloned = true;
            deletesCount = 0;
            this.deletes = deletes;
            if( this.deletes == null )
            {
                this.deletes = new BitVector(in.maxDoc());
            }
        }

        @Override
        public int numDocs() {
            return maxDoc() - deletes.count();
/*
            int count = 0;
            //this is slow
            for (int i = 0; i < maxDoc(); i++) {
                if (!deletes.get(i)) {
                    count++;
                }
            }
            throw new RuntimeException("SLOW");
*/
//            return count;
        }

        @Override
        public Object getCoreCacheKey() {
            return in.getCoreCacheKey();
        }

        @Override
        public synchronized Bits getDeletedDocs() {
//            System.err.println(this + " getDeletesDocs called: " + deletes);
            return deletes;
        }

        public boolean hasDeletes()
        {
//            System.err.println(this + " hasDeletes called");
            return true;
//            return deletesCount > 0;
        }

        public synchronized void deleteDocument(int docNum) {
//            if( docNum > in.maxDoc() ) throw new RuntimeException("DocNum out of range: " + docNum + ", " + in.maxDoc());
            if(deletesCount == 0)
            {
//                deletesOverlay = cloneBits(deletes);
                deletesOverlay = new BitsOverlay(deletes);
            }
            deletesCount++;
            deletesOverlay.set(docNum);
            for (DeleteHandlingReaderLeave clone : clones) {
                clone.deleteDocument(docNum);
            }
        }

        public synchronized void deleteDocuments(int[] docNums) {
//            if( docNum > in.maxDoc() ) throw new RuntimeException("DocNum out of range: " + docNum + ", " + in.maxDoc());
            if(deletesCount == 0)
            {
//                deletesOverlay = cloneBits(deletes);
                deletesOverlay = new BitsOverlay(deletes);
            }
            deletesCount += docNums.length;
            for (int d : docNums) {
                deletesOverlay.set(d);
            }
            for (DeleteHandlingReaderLeave clone : clones) {
                clone.deleteDocuments(docNums);
            }
        }

        public synchronized void deleteDocuments(int[] docs, int offset, int length) {
//            if( docNum > in.maxDoc() ) throw new RuntimeException("DocNum out of range: " + docNum + ", " + in.maxDoc());
            if(deletesCount == 0)
            {
//                deletesOverlay = cloneBits(deletes);
                deletesOverlay = new BitsOverlay(deletes);
            }
            deletesCount += length;
            for (int i = 0; i < length; i++) {
                deletesOverlay.set(docs[offset + i]);
            }
            for (DeleteHandlingReaderLeave clone : clones) {
                clone.deleteDocuments(docs, offset, length);
            }
        }

        public synchronized void deleteDocuments(final ByteSequence seq) {
            if (deletesCount == 0) {
                deletesOverlay = new BitsOverlay(deletes);
            }
            IntReader intReader = new IntReader(seq);
            for (
                int doc = intReader.readInt();
                doc != -1;
                doc = intReader.readInt())
            {
                deletesOverlay.set(doc);
                deletesCount ++;
            }
            for (DeleteHandlingReaderLeave clone : clones) {
                clone.deleteDocuments(seq);
            }
        }

        public boolean deleteDocuments(Collection<? extends QueryProducer> queries) throws IOException
        {
            IndexSearcher searcher = new IndexSearcher(in);
            final boolean cacheDeletes = in instanceof SegmentReader;
            int[] allDeletes = new int[queries.size() * 2];
            int totalDeletes = 0;
            IntWriter deletes = null;
            try {
                BytesRef queryKey = new BytesRef(4);
                for (QueryProducer query : queries) {
                    int deleted = 0;
                    if (cacheDeletes) {
                        SegmentReader sr = (SegmentReader)in;
                        queryKey.length = 0;
                        queryKey.offset = 0;
                        queryKey.writeInt(System.identityHashCode(query));
                        ByteSequence deletesRef = sr.getCachedDelete(queryKey);
                        if (deletesRef != null) {
                            deleteDocuments(deletesRef);
                            continue;
                        }
                    }
                    Weight weight = query.produceQuery().weight(searcher);
                    Scorer scorer = weight.scorer((AtomicReaderContext)searcher.getTopReaderContext(), Weight.ScorerContext.def());
                    if (scorer != null) {
                        while(true)  {
                            int doc = scorer.nextDoc();
                            if( doc == Scorer.NO_MORE_DOCS ) break;
//                              System.err.println("DeleteHandlingIndexLeave<"+reader+">.deleteDocuments: " + query + ", docId=" + doc);
//                            deleteDocument(doc);
                            allDeletes[totalDeletes++] = doc;
                            if (totalDeletes == allDeletes.length) {
                                allDeletes = Arrays.copyOf(allDeletes, allDeletes.length << 1);
                            }
                            if (cacheDeletes) {
                                if (deletes == null) {
                                    deletes = new IntWriter(2);
                                }
                                deletes.writeInt(doc);
                            }
                        }
                    }
                    if (cacheDeletes) {
                        SegmentReader sr = (SegmentReader)in;
                        if (deletes == null) {
                            sr.cacheDeleteQuery(queryKey, IntWriter.EMPTY.ref);
                        } else {
                            sr.cacheDeleteQuery(queryKey, deletes.ref);
                            deletes.reset();
                        }
//                    System.err.println("Caching: " + query + ": " + deletes);
                    }
                }
                deleteDocuments(allDeletes, 0, totalDeletes);
            } finally {
                searcher.close();
            }
            return false;
        }


        public boolean deleteDocuments(
            final QueryProducer query,
            final BytesRef queryKey)
            throws IOException
        {
            IndexSearcher searcher = new IndexSearcher(in);
            final boolean cacheDeletes = in instanceof SegmentReader;
            IntWriter deletes = null;
            int deleted = 0;
            if (cacheDeletes) {
                SegmentReader sr = (SegmentReader)in;
                ByteSequence deletesRef = sr.getCachedDelete(queryKey);
                if (deletesRef != null) {
                    deleteDocuments(deletesRef);
                    return true;
                }
            }
            try {
                Weight weight = query.produceQuery().weight(searcher);
                Scorer scorer = weight.scorer((AtomicReaderContext)searcher.getTopReaderContext(), Weight.ScorerContext.def());
                if (scorer != null) {
                    while(true)  {
                        int doc = scorer.nextDoc();
                        if (doc == Scorer.NO_MORE_DOCS) {
                            break;
                        }
//                      System.err.println("DeleteHandlingIndexLeave<"+reader+">.deleteDocuments: " + query + ", docId=" + doc);
                        deleteDocument(doc);
                        if (cacheDeletes) {
                            if (deletes == null) {
                                deletes = new IntWriter(8);
                            }
                            deletes.writeInt(doc);
                        }
                    }
                    scorer.close();
                }
                if (cacheDeletes) {
                    SegmentReader sr = (SegmentReader)in;
                    if (deletes == null) {
                        sr.cacheDeleteQuery(queryKey, IntWriter.EMPTY.ref);
                    } else {
                        sr.cacheDeleteQuery(queryKey, deletes.ref);
                    }
//                    System.err.println("Caching: " + query + ": " + deletes);
                }
            } finally {
                searcher.close();
            }
            return false;
        }

        public synchronized DeleteHandlingReaderLeave clone()
        {
            if(deletesCount == 0) return this;
            DeleteHandlingReaderLeave clone = new DeleteHandlingReaderLeave(in, deletesOverlay);
            clones = Arrays.copyOf(clones, clones.length + 1);
            clones[clones.length - 1] = clone;
            return clone;
        }

        @Override
        public String toString() {
            final StringBuilder buffer = new StringBuilder("DeletesHandlingReaderLeave(");
            buffer.append(in);
            buffer.append(",cloned=");
            buffer.append(cloned);
            buffer.append(')');
            buffer.append(", deletes=");
            buffer.append(deletes);
            buffer.append(", deletesOverlay=");
            buffer.append(deletesOverlay);
            return buffer.toString();
        }
    }

    private static final int MAX_OVERLAYED_DELETES = 100;
    private static final int MAX_COMPACT_LENGTH = 100000;
    private static class BitsOverlay implements Bits {
        private Bits in = null;
        private int[][] ids;
        private int[] overlay;
        private int count = 0;
        private final int length;
        private int totalOverlayedDeletes;
        private boolean compacted = false;

        public BitsOverlay(final Bits in) {
            this.length = in.length();
            if (in instanceof BitsOverlay) {
                BitsOverlay other = (BitsOverlay)in;

                totalOverlayedDeletes = 0;
                for (int i = 0; i < other.ids.length; i++) {
                    totalOverlayedDeletes += other.ids[i].length;
                }

                if (totalOverlayedDeletes > MAX_OVERLAYED_DELETES || this.length <= MAX_COMPACT_LENGTH) { //merge old overlays
                    this.in = cloneBits(other.in);
                    for (int i = 0; i < other.ids.length; i++) {
                        for (int j = 0; other.ids[i][j] != -1; j++) {
                            ((BitVector)this.in).set(other.ids[i][j]);
                        }
                    }
                    for (int i = 0; i < other.overlay.length && other.overlay[i] != -1; i++) {
                        ((BitVector)this.in).set(other.overlay[i]);
                    }
                    this.ids = new int[0][];
                    this.overlay = new int[1];
                    compacted = true;
                } else {
                    this.in = other.in;
                    this.ids = new int[other.ids.length + 1][];
                    int i = 0;
                    for (; i < other.ids.length; i++) {
                        this.ids[i] = other.ids[i];
                    }
                    this.ids[i] = other.overlay;
                    this.overlay = new int[16];
                }
            } else {
                this.in = in;
                ids = new int[0][];
                overlay = new int[16];
            }
            overlay[0] = -1;
        }

        public void compact() {
//            System.err.println("DeleteHandlingIndexreader.bitsoverlay<"+this+"> compact: totalDeletes=" + totalOverlayedDeletes);
            in = cloneBits(in);
            for (int i = 0; i < ids.length; i++) {
                for (int j = 0; ids[i][j] != -1; j++) {
                    ((BitVector)this.in).set(ids[i][j]);
                }
            }
            for (int i = 0; overlay[i] != -1; i++) {
                ((BitVector)this.in).set(overlay[i]);
            }
            ids = new int[0][];
            overlay = new int[1];
            overlay[0] = -1;
            count = 0;
            totalOverlayedDeletes = 0;
            compacted = true;
        }

        public void set(int id) {
            if (compacted) {
                ((BitVector)in).set(id);
                return;
            }
            overlay[count++] = id;
            overlay[count] = -1; //end of array marker
            if (count + 1 == overlay.length) {
                overlay = Arrays.copyOf(overlay, overlay.length * 2);
            }
            totalOverlayedDeletes++;
            if (totalOverlayedDeletes >= MAX_OVERLAYED_DELETES) {
                compact();
            }
        }

        @Override
        public boolean get(int id) {
            boolean ret = getInt(id);
//            System.err.println("deletesHandlingIndexLeave<"+this+">.deletes.get("+id+"): " + ret);
//            new Exception("DeletesHandlio.get trace").printStackTrace();
            return ret;
        }
        public boolean getInt(int id) {
            if (in != null && in.get(id)) {
                return true;
            }
            for (int i = 0; i < ids.length; i++) {
                if (checkArray(ids[i], id)) {
                    return true;
                }
            }
            return checkArray(overlay, id);
        }

        @Override
        public int length() {
            return length;
        }

        @Override
        public int count() {
            int count = 0;
            if (in != null) {
                count += in.count();
            }
            for (int i = 0; i < ids.length; i++) {
                int[] arr = ids[i];
                int j = 0;
                while (arr[j] != -1) j++;
                count += j;
            }
            int[] arr = overlay;
            int j = 0;
            while (arr[j] != -1) j++;
            count += j;
            return count;
        }

        private final boolean checkArray(int[] array, int id) {
            for (int i = 0; array[i] != -1; i++) {
                if (array[i] == id) return true;
            }
            return false;
        }

        private final BitVector cloneBits(Bits bits)
        {
            if(bits instanceof BitVector)
            {
                return (BitVector)((BitVector)bits).clone();
            }
            else
            {
                BitVector newBits = new BitVector(bits.length());
                for( int i = 0; i < bits.length(); i++ )
                {
                    if( bits.get(i) ) newBits.set(i);
                }
                return newBits;
            }
        }

    }

    private static class IntWriter {
        private static final IntWriter EMPTY = new IntWriter(0);
        private final BytesRef ref;

        IntWriter(int size) {
            ref = new BytesRef(size);
            ref.length = 0;
            ref.offset = 0;
        }

        public void reset() {
            ref.length = 0;
            ref.offset = 0;
        }

        public void writeInt(int value) {
            writeByte((byte) (value >> 24));
            writeByte((byte) (value >> 16));
            writeByte((byte) (value >> 8));
            writeByte((byte) value);
        }

        public void writeByte(byte b) {
            if (ref.length >= ref.bytes.length) {
                ref.bytes = Arrays.copyOf(ref.bytes, ref.bytes.length << 1);
            }
            ref.bytes[ref.length++] = b;
        }
    }

    private static class IntReader {
        private ByteSequence ref;
        int pos;

        public IntReader(final ByteSequence ref) {
            this.ref = ref;
        }

        public void wrap(final ByteSequence ref) {
            this.ref = ref;
            pos = 0;
        }

        public int readInt() {
            try {
            if (pos >= ref.length()) {
                return -1;
            }
            return ((readByte() & 0xFF) << 24)
                | ((readByte() & 0xFF) << 16)
                | ((readByte() & 0xFF) <<  8)
                | (readByte() & 0xFF);
            } catch (Throwable e) {
                System.err.println("ir.ref.l: " + ref.length()
                    + ", ir.ref.off: " + ref.offset()
                    + ", pos: " + pos);
                throw e;
            }
        }

        public int readByte() {
            return ref.byteAt(pos++);
        }
    }
}
