package ru.yandex.msearch;

import java.text.ParseException;
import java.util.ArrayList;
import java.util.List;

import org.apache.lucene.util.StringHelper;

import ru.yandex.analyzer.FilterChainFactory;
import ru.yandex.analyzer.FilterFactory;
import ru.yandex.analyzer.TokenizerType;
import ru.yandex.analyzer.preprocessor.IdentityFieldPreprocessor;
import ru.yandex.analyzer.preprocessor.FieldPreprocessor;
import ru.yandex.analyzer.preprocessor.FieldPreprocessorType;
import ru.yandex.msearch.collector.YaField.FieldType;
import ru.yandex.msearch.config.FilterParser;
import ru.yandex.msearch.fieldscache.FieldCache;
import ru.yandex.msearch.warmfunction.WarmFunction;
import ru.yandex.msearch.warmfunction.WarmFunctionFactory;
import ru.yandex.parser.config.ConfigException;
import ru.yandex.parser.config.IniConfig;
import ru.yandex.parser.string.CollectionParser;
import ru.yandex.parser.string.PositiveIntegerValidator;

public class FieldConfig {
    private final String field;
    private final TokenizerType tokenizer;
    private final boolean store;
    private final boolean index;
    private final boolean prefixed;
    private final boolean analyze;
    private final boolean attribute;
    private final boolean norm;
    private final FieldCache.Type cache;
    private final boolean bloom;
    private final boolean normalizeUtf;
    private final boolean ignoreBinaryData;
    private final int indexDivisor;
    private final FieldType type;
    private final String indexAlias;
    private final String cacheAlias;
    private final List<WarmFunction> warmers;
    private final boolean cacheOnlyActive;

    private final FilterChainFactory indexFilters;
    private final FilterChainFactory searchFilters;

    public FieldConfig(final String name) {
        this.field = name;
        this.type = FieldType.SIZE_ONLY;

        this.tokenizer = null;
        this.store = false;
        this.index = false;
        this.prefixed = false;
        this.analyze = false;
        this.attribute = false;
        this.norm = false;
        this.cache = FieldCache.Type.NO_CACHE;
        this.bloom = false;
        this.normalizeUtf = false;
        this.indexDivisor = 0;
        this.indexAlias = null;
        this.cacheAlias = null;
        this.warmers = null;
        this.cacheOnlyActive = false;
        this.ignoreBinaryData = false;
        this.indexFilters =
            new FilterChainFactory(IdentityFieldPreprocessor.INSTANCE);
        this.searchFilters = indexFilters;
    }

    public FieldConfig(
        final String field,
        final IniConfig config,
        final int defaultIndexDivisor)
        throws ConfigException
    {
        this.field = field;
        try {
            warmers = WarmFunctionFactory.parseFunctions(
                config.getString("warm", ""));
        } catch (ParseException e) {
            throw new ConfigException(
                "Error parsing field <" + field + "> warmers",
                e);
        }
        cacheOnlyActive = config.getBoolean("cache-only-active", false);
        tokenizer = config.getEnum(
            TokenizerType.class,
            "tokenizer",
            TokenizerType.WHITESPACE);
        store = config.getBoolean("store", false);
        index = config.getBoolean("index", true);
        if (index) {
            prefixed = config.getBoolean("prefixed", false);
            analyze = config.getBoolean("analyze", false);
            attribute = config.getBoolean("attribute", false);
            normalizeUtf = config.getBoolean("normalize-utf", false);
            ignoreBinaryData = config.getBoolean("ignore-binary-data", false);
        } else {
            prefixed = false;
            analyze = false;
            attribute = false;
            normalizeUtf = false;
            ignoreBinaryData = false;
            //just read config entries to prevent unused keys exception
            config.getBoolean("prefixed", false);
            config.getBoolean("analyze", false);
            config.getBoolean("attribute", false);
            config.getBoolean("normalize-utf", false);
            config.getBoolean("ignore-binary-data", false);
        }
        norm = config.getBoolean("norm", false);
        cache = config.getEnum(
            FieldCache.Type.class,
            "cache",
            FieldCache.Type.NO_CACHE);
        bloom = config.getBoolean("bloom", false);
        indexDivisor = config.get(
            "index-divisor",
            defaultIndexDivisor,
            PositiveIntegerValidator.INSTANCE);
        type = config.getEnum(FieldType.class, "type", FieldType.STRING);
        String indexAlias = config.getString("index_alias", null);
        if (indexAlias == null) {
            this.indexAlias = null;
        } else {
            this.indexAlias = StringHelper.intern(indexAlias);
        }

        String cacheAlias = config.getString("cache_alias", null);
        if (cacheAlias == null) {
            this.cacheAlias = null;
        } else {
            this.cacheAlias = StringHelper.intern(cacheAlias);
        }

        FieldPreprocessor indexPreprocessor;
        IniConfig preprocessorConfig =
            config.sectionOrNull("index_preprocessor");
        if (preprocessorConfig == null) {
            indexPreprocessor = IdentityFieldPreprocessor.INSTANCE;
        } else {
            FieldPreprocessorType preprocessorType =
                preprocessorConfig.getEnum(FieldPreprocessorType.class, "type");
            indexPreprocessor =
                preprocessorType.createPreprocessor(preprocessorConfig);
        }

        String searchFiltersKey;
        if (config.getString("search-filters", null) == null) {
            searchFiltersKey = "filters";
        } else {
            searchFiltersKey = "search-filters";
        }

        String indexFiltersKey;
        if (config.getString("index-filters", null) == null) {
            indexFiltersKey = "filters";
        } else {
            indexFiltersKey = "index-filters";
        }

        CollectionParser<FilterFactory, FilterChainFactory, Exception>
            searchFiltersParser
                = new CollectionParser<>(
                    new FilterParser(field, ':'),
                    () -> new FilterChainFactory(
                        IdentityFieldPreprocessor.INSTANCE),
                    '|');
        searchFilters = config.get(
            searchFiltersKey,
            new FilterChainFactory(IdentityFieldPreprocessor.INSTANCE),
            searchFiltersParser);

        CollectionParser<FilterFactory, FilterChainFactory, Exception>
            indexFiltersParser
                = new CollectionParser<>(
                    new FilterParser(field, ':'),
                    () -> new FilterChainFactory(indexPreprocessor),
                    '|');
        indexFilters = config.get(
            indexFiltersKey,
            new FilterChainFactory(indexPreprocessor),
            indexFiltersParser);
    }

    public String field() {
        return field;
    }

    public TokenizerType tokenizer() {
        return tokenizer;
    }

    public FilterChainFactory indexFilters() {
        return indexFilters;
    }

    public FilterChainFactory searchFilters() {
        return searchFilters;
    }

    public boolean index() {
        return index;
    }

    public boolean prefixed() {
        return prefixed;
    }

    public boolean store() {
        return store;
    }

    public boolean analyze() {
        return analyze;
    }

    public boolean norm() {
        return norm;
    }

    public boolean attribute() {
        return attribute;
    }

    public FieldCache.Type cache() {
        return cache;
    }

    public boolean cacheOnlyActive() {
        return cacheOnlyActive;
    }

    public boolean bloom() {
        return bloom;
    }

    public boolean normalizeUtf() {
        return normalizeUtf;
    }

    public boolean ignoreBinaryData() {
        return ignoreBinaryData;
    }

    public int indexDivisor() {
        return indexDivisor;
    }

    public FieldType type() {
        return type;
    }

    public String indexAlias() {
        return indexAlias;
    }

    public String cacheAlias() {
        return cacheAlias;
    }

    public boolean warm() {
        return warmers != null;
    }

    public List<WarmFunction> warmers() {
        return warmers;
    }
}


