package ru.yandex.msearch;

import java.io.File;
import java.io.IOException;
import java.io.StringReader;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.StandardOpenOption;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.IdentityHashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import org.apache.lucene.util.StringHelper;

import ru.yandex.analyzer.FilterFactory;
import ru.yandex.analyzer.PrefixSubstringsFilterFactory;
import ru.yandex.collection.IdentityHashSet;
import ru.yandex.parser.config.ConfigException;
import ru.yandex.parser.config.IniConfig;
import ru.yandex.parser.string.CollectionParser;
import ru.yandex.parser.string.FileParser;
import ru.yandex.parser.string.NonEmptyValidator;

public class FieldsConfig {
    private final Map<String, FieldConfig> fields;
    private final Map<String, FieldConfig> dynamicFields;
    private final Map<String, FieldConfig> staticFields;
    private final Map<String, FieldConfig> fieldsFast;
    private final Map<String, FieldConfig> dynamicFieldsFast;
    private final Map<String, FieldConfig> staticFieldsFast;
    private final Set<String> bloomSet;
    private final Set<String> staticBloomSet;
    private final Map<String, Integer> prefixSubstringFilterFields;
    private final Set<String> ignoredFields;
    private final Set<String> rootAliasedFields;
    // map alias name -> root field config
    private final Map<String, List<FieldConfig>> aliasReverseChain;
    private final Set<String> storedFields;
    private final Set<String> indexedFields;
    private final Map<String, File> dynamicFieldsFiles;
    private final File dynamicFieldsDirectory;
    private final Map<String, Integer> indexDivisors;
    private final int indexDivisor;

    public FieldsConfig(
        final IniConfig config,
        final int indexDivisor)
        throws IOException, ConfigException
    {
        this.indexDivisor = indexDivisor;
        Map<String, FieldConfig> fields = new LinkedHashMap<>();
        Map<String, FieldConfig> dynamicFields = new LinkedHashMap<>();
        Map<String, FieldConfig> fieldsFast = new IdentityHashMap<>();
        Map<String, FieldConfig> dynamicFieldsFast = new IdentityHashMap<>();
        Set<String> bloomSet = new HashSet<>();
        Set<String> dynamicBloomSet = new HashSet<>();

        FieldConfig prefixFieldConfig = createFieldConfig(
            Config.PREFIX_FIELD_KEY,
            Config.PREFIX_FIELD_CONFIG,
            indexDivisor);
        fields.put(Config.PREFIX_FIELD_KEY, prefixFieldConfig);
        fieldsFast.put(Config.PREFIX_FIELD_KEY, prefixFieldConfig);

        FieldConfig queueIdFieldConfig = createFieldConfig(
            Config.QUEUE_ID_FIELD_KEY,
            Config.QUEUE_ID_FIELD_CONFIG,
            indexDivisor);
        fields.put(Config.QUEUE_ID_FIELD_KEY, queueIdFieldConfig);
        fieldsFast.put(Config.QUEUE_ID_FIELD_KEY, queueIdFieldConfig);

        FieldConfig queueNameFieldConfig = createFieldConfig(
            Config.QUEUE_NAME_FIELD_KEY,
            Config.QUEUE_NAME_FIELD_CONFIG,
            indexDivisor);
        fields.put(Config.QUEUE_NAME_FIELD_KEY, queueNameFieldConfig);
        fieldsFast.put(Config.QUEUE_NAME_FIELD_KEY, queueNameFieldConfig);

        Map<String, Integer> prefixFiltersFields = new LinkedHashMap<>();

        IniConfig fieldConfigs = config.section("field");
        loadFields(fieldConfigs, fields, fieldsFast, prefixFiltersFields, bloomSet, indexDivisor);

        this.staticFieldsFast = Collections.unmodifiableMap(fieldsFast);
        this.staticBloomSet = Collections.unmodifiableSet(bloomSet);
        this.staticFields = Collections.unmodifiableMap(fields);

        this.dynamicFieldsDirectory = config.get("dynamic_fields_dir", null, FileParser.DIRECTORY);
        if (dynamicFieldsDirectory != null) {
            if (!dynamicFieldsDirectory.exists()) {
                System.err.println("Dynamic fields dir not exists, creating " + dynamicFieldsDirectory.toString());
                Files.createDirectory(dynamicFieldsDirectory.toPath());
            }

            Map<String, File> files = new LinkedHashMap<>();
            List<Path> paths = Files.list(dynamicFieldsDirectory.toPath()).collect(Collectors.toList());
            for (Path path: paths) {
                File dynamicFieldsFile = path.toAbsolutePath().toFile();
                String name = dynamicFieldsFile.getName();
                int index = name.lastIndexOf('.');
                if (index > 0) {
                    name = name.substring(0, index);
                }

                System.err.println("Loading Dynamic File " + dynamicFieldsFile.toString());
                files.put(name, dynamicFieldsFile);
                loadFields(
                    new IniConfig(dynamicFieldsFile).section("field"),
                    dynamicFields,
                    dynamicFieldsFast,
                    Collections.emptyMap(),
                    dynamicBloomSet,
                    indexDivisor);
            }
            dynamicFieldsFiles = Collections.unmodifiableMap(files);
            System.err.println("Loaded dynamic fields " + dynamicFields);
        } else {
            dynamicFieldsFiles = Collections.emptyMap();
        }

        this.dynamicFieldsFast = Collections.unmodifiableMap(dynamicFieldsFast);
        //this.dynamicBloomSet = Collections.unmodifiableSet(dynamicBloomSet);
        this.dynamicFields = Collections.unmodifiableMap(dynamicFields);

        fields = new LinkedHashMap<>();
        fields.putAll(this.dynamicFields);
        fields.putAll(this.staticFields);
        this.fields = Collections.unmodifiableMap(fields);
        fieldsFast = new LinkedHashMap<>();
        fieldsFast.putAll(this.dynamicFieldsFast);
        fieldsFast.putAll(this.staticFieldsFast);
        this.fieldsFast = Collections.unmodifiableMap(fieldsFast);
        this.bloomSet = this.staticBloomSet;

        this.prefixSubstringFilterFields = Collections.unmodifiableMap(prefixFiltersFields);

        Set<String> rootAliasedFields = new HashSet<>();
        Map<String, List<FieldConfig>> aliasReverseChain = new LinkedHashMap<>();
        rootAliasedFields(rootAliasedFields, aliasReverseChain, this.fields);
        this.rootAliasedFields =
            Collections.unmodifiableSet(rootAliasedFields);
        this.aliasReverseChain = Collections.unmodifiableMap(aliasReverseChain);

        Set<String> ignoredFields = config.get(
            "ignored_fields",
            Collections.emptySet(),
            new CollectionParser<>(NonEmptyValidator.TRIMMED, HashSet::new));
        for (String field: ignoredFields) {
            if (fields.containsKey(field)) {
                throw new ConfigException("Field " + field
                    + " marked as ignored, but has config");
            }
        }

        this.ignoredFields = Collections.unmodifiableSet(ignoredFields);

        storedFields = new IdentityHashSet<>(this.fields.size() << 2);
        indexedFields = new IdentityHashSet<>(this.fields.size() << 2);
        for (Map.Entry<String, FieldConfig> entry: this.fields.entrySet()) {
            if (entry.getValue().store()) {
                storedFields.add(StringHelper.intern(entry.getKey()));
            }
            if (entry.getValue().index()) {
                indexedFields.add(StringHelper.intern(entry.getKey()));
            }
        }

        Map<String, Integer> indexDivisors = new HashMap<>();
        for(Map.Entry<String, FieldConfig> fieldEntry: this.fields.entrySet()) {
            final FieldConfig fieldConfig = fieldEntry.getValue();
            final int fieldIndexDivisor = fieldConfig.indexDivisor();
            indexDivisors.put(fieldEntry.getKey(), fieldIndexDivisor);
        }
        this.indexDivisors = Collections.unmodifiableMap(indexDivisors);
        System.err.println("Loaded dynamic fields " + dynamicFields.keySet());
    }

    public FieldsConfig(
        final FieldsConfig config,
        final String name,
        final String dynamicConfigStr)
        throws IOException, ConfigException
    {
        if (config.dynamicFieldsFiles == null) {
            throw new ConfigException("Dynamic fields are not supported");
        }

        this.staticBloomSet = config.staticBloomSet;
        this.staticFieldsFast = config.staticFieldsFast;
        this.staticFields = config.staticFields;

        this.ignoredFields = config.ignoredFields;
        this.prefixSubstringFilterFields = config.prefixSubstringFilterFields;
        this.indexDivisor = config.indexDivisor;
        this.dynamicFieldsDirectory = config.dynamicFieldsDirectory;
        Map<String, FieldConfig> fields = new LinkedHashMap<>();
        Map<String, FieldConfig> fieldsFast = new IdentityHashMap<>();
        Map<String, Integer> prefixFilterFields = new LinkedHashMap<>();
        Set<String> dynamicBloomSet = new HashSet<>();
        IniConfig dynamicConfig = new IniConfig(new StringReader(dynamicConfigStr));
        if (config.dynamicFieldsFiles.get(name) != null) {
            Files.write(
                Path.of(config.dynamicFieldsFiles.get(name).toURI()),
                dynamicConfigStr.getBytes(StandardCharsets.UTF_8),
                StandardOpenOption.TRUNCATE_EXISTING);
            this.dynamicFieldsFiles = config.dynamicFieldsFiles;
        } else {
            File file = new File(dynamicFieldsDirectory,  name + ".conf");
            Files.write(Path.of(file.toURI()),
                dynamicConfigStr.getBytes(StandardCharsets.UTF_8),
                StandardOpenOption.CREATE_NEW);
            Map<String, File> dynamicFieldsFiles = new LinkedHashMap<>(config.dynamicFieldsFiles);
            dynamicFieldsFiles.put(name, file);
            this.dynamicFieldsFiles = Collections.unmodifiableMap(dynamicFieldsFiles);
        }

        for (Map.Entry<String, File> entry: dynamicFieldsFiles.entrySet()) {
            loadFields(
                new IniConfig(entry.getValue()).section("field"),
                fields,
                fieldsFast,
                prefixFilterFields,
                dynamicBloomSet,
                config.indexDivisor);
        }

//        loadFields(
//            dynamicConfig.section("field"),
//            fields,
//            fieldsFast,
//            prefixFilterFields,
//            dynamicBloomSet,
//            config.indexDivisor);

        this.dynamicFields =
            Collections.unmodifiableMap(new LinkedHashMap<>(fields));
        System.err.println("Loaded dynamic fields " + dynamicFields);

        this.dynamicFieldsFast =
            Collections.unmodifiableMap(new LinkedHashMap<>(fieldsFast));

        fields.putAll(staticFields);
        fieldsFast.putAll(staticFieldsFast);
        //prefixFilterFields.putAll(prefixFilterFields);


        this.fields = Collections.unmodifiableMap(fields);
        this.fieldsFast = Collections.unmodifiableMap(fieldsFast);

        Set<String> rootAliasedFields = new HashSet<>();
        Map<String, List<FieldConfig>> aliasReverseChain = new LinkedHashMap<>();
        rootAliasedFields(rootAliasedFields, aliasReverseChain, this.fields);
        this.rootAliasedFields =
            Collections.unmodifiableSet(rootAliasedFields);
        this.aliasReverseChain = Collections.unmodifiableMap(aliasReverseChain);
        this.bloomSet = staticBloomSet;
        storedFields = new IdentityHashSet<>(this.fields.size() << 2);
        indexedFields = new IdentityHashSet<>(this.fields.size() << 2);
        for (Map.Entry<String, FieldConfig> entry: this.fields.entrySet()) {
            if (entry.getValue().store()) {
                storedFields.add(StringHelper.intern(entry.getKey()));
            }
            if (entry.getValue().index()) {
                indexedFields.add(StringHelper.intern(entry.getKey()));
            }
        }

        Map<String, Integer> indexDivisors = new HashMap<>();
        for(Map.Entry<String, FieldConfig> fieldEntry: this.fields.entrySet()) {
            final FieldConfig fieldConfig = fieldEntry.getValue();
            final int fieldIndexDivisor = fieldConfig.indexDivisor();
            indexDivisors.put(fieldEntry.getKey(), fieldIndexDivisor);
        }
        this.indexDivisors = Collections.unmodifiableMap(indexDivisors);

        System.err.println("Loaded dynamic fields " + dynamicFields.keySet() + " all fields " + fields.keySet());
    }

    private static FieldConfig createFieldConfig(
        final String fieldName,
        final String fieldConfig,
        final int indexDivisor)
        throws ConfigException
    {
        try {
            return new FieldConfig(
                fieldName,
                new IniConfig(new StringReader(fieldConfig)),
                indexDivisor);
        } catch (IOException e) {
            throw new ConfigException(
                "Failed to parse " + fieldName + " field config",
                e);
        }
    }

    private void rootAliasedFields(
        final Set<String> rootAliasedFields,
        final Map<String, List<FieldConfig>> aliasReverseChain,
        final Map<String, FieldConfig> fields)
        throws ConfigException
    {
        Set<String> aliases = new HashSet<>(fields.size());
        // alias -> parent map
        Map<String, FieldConfig> aliasRelations = new LinkedHashMap<>(fields.size());
        for (Map.Entry<String, FieldConfig> field: fields.entrySet()) {
            String alias = field.getValue().indexAlias();
            if (alias != null) {
                if (!fields.containsKey(alias)) {
                    throw new ConfigException("Don't know how to index "
                        + alias + ", which is an index alias for "
                        + field.getKey());
                }
                if (aliases.contains(alias)) {
                    throw new ConfigException("Field " + alias
                        + " is an index alias for two or more fields, which"
                        + " may lead to unpredictable behaviour");
                }
                aliases.add(alias);
                aliasRelations.put(alias, field.getValue());
            }
        }

        for (Map.Entry<String, FieldConfig> entry: aliasRelations.entrySet()) {
            List<FieldConfig> chain = new ArrayList<>();
            FieldConfig nextConfig = entry.getValue();
            while (nextConfig != null) {
                chain.add(nextConfig);
                nextConfig = aliasRelations.get(nextConfig.field());
            }

            aliasReverseChain.put(entry.getKey(), Collections.unmodifiableList(chain));
        }

        for (Map.Entry<String, FieldConfig> field: fields.entrySet()) {
            if (aliases.contains(field.getKey())) {
                if (field.getValue().store()) {
                    throw new ConfigException("Field " + field.getKey()
                        + " is stored and is an alias for another field. In"
                        + " order to avoid yet another stupid bug one should"
                        + " store value only in the root aliased field");
                }
            } else if (field.getValue().indexAlias() != null) {
                rootAliasedFields.add(field.getKey());
            }
        }
    }

    private void loadFields(
        final IniConfig fieldConfigs,
        final Map<String, FieldConfig> fields,
        final Map<String, FieldConfig> fieldsFast,
        final Map<String, Integer> prefixFiltersFields,
        final Set<String> bloomSet,
        final int indexDivisor)
        throws ConfigException
    {
        for (Map.Entry<String, IniConfig> entry
            : fieldConfigs.sections().entrySet())
        {
            String field = StringHelper.intern(entry.getKey());
            FieldConfig fieldConfig =
                new FieldConfig(field, entry.getValue(), indexDivisor);
            if (!fieldConfig.analyze()) {
                if (fieldConfig.indexFilters().size() > 0 || fieldConfig.searchFilters().size() > 0) {
                    throw new ConfigException("Field " + field
                        + " is disabled for analyzing,"
                        + " but have filters defined,"
                        + " which is meaningless");
                }
                if (fieldConfig.prefixed()) {
                    throw new ConfigException("Field " + field
                        + " is disabled for analyzing,"
                        + " but marked as prefixed,"
                        + " which is a pretty good way"
                        + " to shoot yourself in the foot");
                }
            }
            for (FilterFactory factory: fieldConfig.indexFilters()) {
                if (factory instanceof PrefixSubstringsFilterFactory) {
                    prefixFiltersFields.put(
                        fieldConfig.field(),
                        ((PrefixSubstringsFilterFactory) factory).maxSubstringSize());
                }
            }

            fields.put(field, fieldConfig);
            fieldsFast.put(field, fieldConfig);
            if (fieldConfig.bloom()) {
                bloomSet.add(field);
            }
        }
    }

    public Map<String, FieldConfig> fields() {
        return fields;
    }

    public Map<String, Integer> indexDivisors() {
        return indexDivisors;
    }

    public Map<String, FieldConfig> dynamicFields() {
        return dynamicFields;
    }

    public Map<String, FieldConfig> staticFields() {
        return staticFields;
    }

    public Map<String, FieldConfig> fieldsFast() {
        return fieldsFast;
    }

    public Map<String, FieldConfig> dynamicFieldsFast() {
        return dynamicFieldsFast;
    }

    public Map<String, FieldConfig> staticFieldsFast() {
        return staticFieldsFast;
    }

    public Set<String> bloomSet() {
        return bloomSet;
    }

    public Set<String> staticBloomSet() {
        return staticBloomSet;
    }

    public Map<String, Integer> prefixSubstringFilterFields() {
        return prefixSubstringFilterFields;
    }

    public Set<String> ignoredFields() {
        return ignoredFields;
    }

    public Set<String> rootAliasedFields() {
        return rootAliasedFields;
    }

    public Map<String, List<FieldConfig>> aliasReverseChain() {
        return aliasReverseChain;
    }

    public Set<String> storedFields() {
        return storedFields;
    }

    public Set<String> indexedFields() {
        return indexedFields;
    }

    public Map<String, File> dynamicFieldsFiles() {
        return dynamicFieldsFiles;
    }

    public int indexDivisor() {
        return indexDivisor;
    }
}
