//package org.apache.lucene.index;
package ru.yandex.msearch;

/**
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

import org.apache.lucene.document.Document;
import org.apache.lucene.document.FieldSelector;
import org.apache.lucene.index.FilterIndexReader;
import org.apache.lucene.index.IndexReader;
import org.apache.lucene.index.IndexReader.AtomicReaderContext;
import org.apache.lucene.index.IndexReader.CompositeReaderContext;
import org.apache.lucene.index.IndexReader.ReaderContext;
import org.apache.lucene.index.SegmentReader;
import org.apache.lucene.search.IndexSearcher;
import org.apache.lucene.search.Query;
import org.apache.lucene.search.QueryProducer;
import org.apache.lucene.search.Scorer;
import org.apache.lucene.search.Weight;
import org.apache.lucene.store.Directory;
import org.apache.lucene.util.BitVector;
import org.apache.lucene.util.Bits;
import org.apache.lucene.util.BytesRef;
import org.apache.lucene.util.MapBackedSet;
import org.apache.lucene.util.OpenBitSet;
import org.apache.lucene.util.ReaderUtil;

import java.io.IOException;
import java.io.PrintWriter;
import java.io.StringWriter;
import java.io.Writer;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.LinkedList;
import java.util.Map;
import java.util.Comparator;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.Callable;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.Future;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;

public class FilteringIndexReader extends FilterIndexReader {
    ReaderContext topReaderContext;
    AtomicReaderContext[] origLeaves;
    AtomicReaderContext[] filterLeaves;
    FilteringReaderLeave[] subReaders;

    public FilteringIndexReader(final IndexReader in) throws IOException {
        super(in);
        origLeaves = ReaderUtil.leaves(in.getTopReaderContext());
        filterLeaves = new AtomicReaderContext[origLeaves.length];
        subReaders = new FilteringReaderLeave[origLeaves.length];
        topReaderContext =
            new CompositeReaderContext(this, filterLeaves, filterLeaves);
        for (int i = 0; i < origLeaves.length; i++) {
            AtomicReaderContext origCtx = origLeaves[i];
            subReaders[i] = new FilteringReaderLeave(origCtx.reader);
            filterLeaves[i] =
                new AtomicReaderContext(
                    topReaderContext,
                    subReaders[i],
                    i,
                    origCtx.docBase,
                    i,
                    origCtx.docBase);
        }
    }

    @Override
    public int numDocs() {
        int count = 0;
        for (FilteringReaderLeave subReader : subReaders) {
            count += subReader.numDocs();
        }
        return count;
    }

    public IndexReader[] getSequentialSubReaders() {
        return subReaders;
    }

    public ReaderContext getTopReaderContext() {
        return topReaderContext;
    }

    @Override
    public String toString() {
        final StringBuilder buffer = new StringBuilder("FilteringIndexReader(");
        buffer.append(in);
        buffer.append('<');
        buffer.append(this.hashCode());
        buffer.append('>');
        return buffer.toString();
    }

     @Override
    protected void doClose() throws IOException {
    }

    static class FilteringReaderLeave extends FilterIndexReader {
        private FilteringBits deletes;

        public FilteringReaderLeave(final IndexReader in) {
            super(in);
            deletes = new FilteringBits(in.getDeletedDocs());
        }

        @Override
        public int numDocs() {
            return in.numDocs();
        }

        @Override
        public Object getCoreCacheKey() {
            return in.getCoreCacheKey();
        }

        @Override
        public synchronized Bits getDeletedDocs() {
            return deletes;
        }

        public boolean hasDeletes() {
            return true;
        }

        public void deleteDocument(final int docNum) {
            deletes.set(docNum);
        }

        @Override
        public String toString() {
            final StringBuilder buffer = new StringBuilder("FilteringReaderLeave(");
            buffer.append(in);
            return buffer.toString();
        }

         @Override
        protected void doClose() throws IOException {
        }
    }

    private static class FilteringBits implements Bits {
        private final Bits other;
        private final OpenBitSet overlay;

        FilteringBits(final Bits other) {
            this.other = other;
            overlay = new OpenBitSet(other.length());
        }

        public void set(final int index) {
            if (!other.get(index)) {
                overlay.set(index);
            }
        }

        @Override
        public boolean get(final int index) {
            return other.get(index) || overlay.get(index);
        }

        @Override
        public int length() {
            return other.length();
        }

        @Override
        public int count() {
            return other.count() + overlay.count();
        }
    }
}
