package ru.yandex.msearch;

import java.text.ParseException;

import java.io.Closeable;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.logging.Level;
import java.util.logging.Logger;

import org.apache.lucene.document.Document;
import org.apache.lucene.document.Field;
import org.apache.lucene.document.Fieldable;
import org.apache.lucene.util.StringHelper;

import ru.yandex.function.GenericAutoCloseable;
import ru.yandex.msearch.config.DatabaseConfig;
import ru.yandex.search.json.fieldfunction.ConditionsAccessor;
import ru.yandex.search.json.fieldfunction.FieldAccessor;
import ru.yandex.search.json.fieldfunction.FieldFunction;
import ru.yandex.search.json.fieldfunction.FieldFunctionException;
import ru.yandex.search.json.fieldfunction.InvalidStateFieldAccessor;
import ru.yandex.search.json.fieldfunction.value.DefaultValue;
import ru.yandex.search.json.fieldfunction.value.FunctionValue;

import ru.yandex.search.prefix.Prefix;

public class HTMLDocument implements FieldAccessor, Closeable {
    private static final String BINARY_ZEROES = new String(
        new char[] {0, 0});
    private static final String BINARY_EFBFBD = new String(
        new byte[] {
            (byte) 0xEF, (byte) 0xBF, (byte) 0xBD,
            (byte) 0xEF, (byte) 0xBF, (byte) 0xBD,
            (byte) 0xEF, (byte) 0xBF, (byte) 0xBD});
    private final Set<String> deletedFields = new HashSet<>();
    private final DatabaseConfig config;
    private final Map<String, FieldFunction> functionFields;
    private final Logger logger;
    private final Prefix prefix;
    private final long phantomQueueId;
    private final String queueName;
    private Document doc;
    private boolean failOnDuplicateKey;
    private final boolean orderIndependentUpdate;
    private long weight;
    protected PrimaryKey primaryKey;

    public Logger logger() {
        return logger;
    }

    public boolean fieldConfigModified(final Fieldable field) {
        final String name = field.name();
        final FieldConfig config = this.config.fieldConfigFast(name);
        if (config == null || field == null) {
            return false;
        }
        if (config.index() != field.isIndexed()
            || (config.analyze() != field.isTokenized())
            || (config.store() != field.isStored()))
        {
            return true;
        } else {
            return false;
        }
    }

    public Field getField(final String name) {
        return getField(name, config.fieldConfigFast(name));
    }

    public static Field getField(final String name, final FieldConfig config) {
        if (config == null) {
            return null;
        }
        Field.Index index;
        if (!config.index()) {
            index = Field.Index.NO;
        } else if (config.analyze()) {
            if (config.norm()) {
                index = Field.Index.ANALYZED;
            } else {
                index = Field.Index.ANALYZED_NO_NORMS;
            }
        } else {
            if (config.norm()) {
                index = Field.Index.NOT_ANALYZED;
            } else {
                index = Field.Index.NOT_ANALYZED_NO_NORMS;
            }
        }
        Field field = new Field(name, "",
            config.store() ? Field.Store.YES : Field.Store.NO, index);
        if (config.attribute() || !config.analyze()) {
            field.setOmitTermFreqAndPositions(true);
        }
        return field;
    }

    private static <T> T validatePrimaryKey(final String name, final T value)
        throws ParseException
    {
        if (value == null) {
            throw new ParseException("Field " + name
                + " is a part of primary key"
                + " but no value is set for it", 0);
        } else {
            return value;
        }
    }

    private String unshare(final String string) {
        if (config.primaryKeyPartCacheDocuments()) {
            return string;
        } else {
            StringBuilder sb = new StringBuilder(string.length());
            sb.append(string);
            return new String(sb);
        }
    }

    @Override
    public void close() {
        if (primaryKey != null) {
            primaryKey.close();
            primaryKey = null;
        }
    }

    private PrimaryKey extractPrimaryKey(final Document doc)
        throws ParseException
    {
        Set<String> primaryKeyFields = config.primaryKey();
        if (primaryKeyFields == null) {
            return null;
        }

        if (primaryKeyFields.size() == 1) {
            String name = primaryKeyFields.iterator().next();
            String value = unshare(validatePrimaryKey(name, doc.get(name)));
            return new PrimaryKey.BasicPrimaryKey(name, value, prefix, config);
        } else {
            Map<String, String> primaryKey = new HashMap<>();
            for (String key: primaryKeyFields) {
                String value = unshare(validatePrimaryKey(key, doc.get(key)));
                primaryKey.put(key, value);
            }
            return PrimaryKey.create(primaryKey, prefix, config);
        }
    }

    protected PrimaryKey extractPrimaryKey(
        final Map<String, FieldFunction> doc)
        throws ParseException
    {
        Set<String> primaryKeyFields = config.primaryKey();
        if (primaryKeyFields == null) {
            return null;
        }

        try {
            if (primaryKeyFields.size() == 1) {
                String name = primaryKeyFields.iterator().next();
                FieldFunction value = validatePrimaryKey(name, doc.get(name));
                return new PrimaryKey.BasicPrimaryKey(
                    name,
                    unshare(value.value(InvalidStateFieldAccessor.INSTANCE, null).stringValue()),
                    prefix,
                    config);
            } else {
                Map<String, String> primaryKey = new HashMap<>();
                for (String key: primaryKeyFields) {
                    FieldFunction value =
                        validatePrimaryKey(key, doc.get(key));
                    primaryKey.put(
                        key,
                        unshare(value.value(InvalidStateFieldAccessor.INSTANCE, null).stringValue()));
                }
                return PrimaryKey.create(primaryKey, prefix, config);
            }
        } catch (FieldFunctionException e) {
            ParseException pce = new ParseException(
                "Error caught trying to access primary key value",
                0);
            pce.initCause(e);
            throw pce;
        }
    }

    private final boolean containsBinaryData(final String value) {
        return value.indexOf(BINARY_ZEROES) != -1
            || value.indexOf(BINARY_EFBFBD) != -1;
    }

    public void setField(final String name, final String value) {
        FieldConfig fieldConfig = config.fieldConfigFast(name);
        if (fieldConfig == null) {
            if (!config.isIgnored(name)) {
                if (logger.isLoggable(Level.WARNING)) {
                    logger.warning("Unknown field: " + name);
                }
            }
            return;
        }
        if (value != null
            && fieldConfig.ignoreBinaryData()
            && containsBinaryData(value))
        {
            if (logger.isLoggable(Level.WARNING)) {
                logger.warning("Field<" + name + "> contains binary data "
                    + "and field.ignore-binary-data parameter is set, "
                    + "ignoring value");
            }
            return;
        }
        setField(name, value, fieldConfig);
    }

    public void setField(
        final String name,
        final String value,
        final FieldConfig config)
    {
        weight += name.length() * 2 + 100;
        if (value == null) {
            doc.removeField(name);
            deletedFields.add(name);
        } else {
            weight += value.length() * 2 + 100;
            Field field = doc.getField( name );
            if (field == null) {
                field = getField( name );
                doc.add( field );
            }
            field.setValue( value );
        }
        String alias = config.indexAlias();
        if (alias != null) {
            setField(alias, value);
        }
    }

    @Override
    public FunctionValue getFieldValue(String fieldName) {
        fieldName = StringHelper.intern(fieldName);
        Field field = doc.getField(fieldName);
        if (field == null) {
            return DefaultValue.INSTANCE;
        }
        return config.fieldConfigFast(fieldName).type()
            .createFieldFunctionValue(field);
    }

    public String getFieldValueFromFieldFunctionOrNull(final String fieldName,
                                      FieldFunction fieldFunction,
                                      FieldAccessor accessor,
                                      ConditionsAccessor condAccessor) throws FieldFunctionException {
        final FunctionValue funcValue = fieldFunction.value(accessor, condAccessor);
        final FieldConfig fieldConfig = config.fieldConfigFast(fieldName);
        if (funcValue != null && fieldConfig != null) {
            return fieldConfig.type().indexableString(
                            funcValue);
        } else {
            return null;
        }
    }

    public void updateWith(final HTMLDocument doc) {
        for (Fieldable field: doc.getDoc().getFields()) {
            setField(field.name(), field.stringValue());
        }
        for (String field: doc.deletedFields()) {
            setField(field, null);
        }
    }

    public void updateWith(final Map<String, FieldFunction> doc,
                           FieldAccessor accessor,
                           ConditionsAccessor condAccessor)
            throws FieldFunctionException
    {
        try {
            if (this.doc == null) {
                this.doc = new Document();
            }

            if (orderIndependentUpdate) {
                final String[] kvToUpdate = new String[doc.size() * 2];
                int kvToUpdateIt = 0;
                for (Map.Entry<String, FieldFunction> field : doc.entrySet()) {
                    final String fieldName = StringHelper.intern(field.getKey());
                    final String fieldValue = getFieldValueFromFieldFunctionOrNull(fieldName, field.getValue(),
                            accessor, condAccessor);

                    kvToUpdate[kvToUpdateIt++] = fieldName;
                    kvToUpdate[kvToUpdateIt++] = fieldValue;
                }

                for (int i = 0; i < kvToUpdateIt; i += 2) {
                    setField(kvToUpdate[i], kvToUpdate[i + 1]);
                }
            } else {
                for (Map.Entry<String, FieldFunction> field : doc.entrySet()) {
                    final String fieldName = StringHelper.intern(field.getKey());
                    final String fieldValue = getFieldValueFromFieldFunctionOrNull(fieldName, field.getValue(),
                            accessor, condAccessor);

                    setField(fieldName, fieldValue);
                }
            }
        } catch (Exception e) {
            throw new FieldFunctionException("Unhandled exception", e);
        }
    }

    public void updateWith(final Map<String, FieldFunction> doc, ConditionsAccessor condAccessor)
        throws FieldFunctionException
    {
        updateWith(doc, this, condAccessor);
    }

    public Set<String> deletedFields() {
        return Collections.unmodifiableSet(deletedFields);
    }

    public Document getDoc() {
        return doc;
    }

    public Prefix prefix() {
        return prefix;
    }

    public long phantomQueueId() {
        return phantomQueueId;
    }

    public String queueName() {
        return queueName;
    }

    public PrimaryKey primaryKey() {
        return primaryKey;
    }

    public void prepare(final ConditionsAccessor condAccessor)
        throws ParseException
    {
        if (doc == null && functionFields != null) {
            try {
                doc = createDocFromFunctions(condAccessor);
            } catch (RuntimeException e) {
                ParseException ex =
                    new ParseException("Failed to prepare HTMLDocument", 0);
                ex.initCause(e);
                throw ex;
            }
        }
    }

    public boolean failOnDuplicateKey() {
        return failOnDuplicateKey;
    }

    public void setFailOnDuplicateKey(final boolean failOnDup) {
        this.failOnDuplicateKey = failOnDup;
    }

    protected HTMLDocument(
        final Prefix prefix,
        final long phantomQueueId,
        final String queueName,
        final boolean orderIndependentUpdate,
        final DatabaseConfig config,
        final Logger logger)
    {
        this.config = config;
        this.logger = logger;
        functionFields = null;
        this.prefix = prefix;
        this.phantomQueueId = phantomQueueId;
        this.queueName = queueName;
        this.orderIndependentUpdate = orderIndependentUpdate;
    }

    public HTMLDocument(
        final Map<String, FieldFunction> document,
        final Prefix prefix,
        final long phantomQueueId,
        final String queueName,
        final boolean orderIndependentUpdate,
        final DatabaseConfig config,
        final Logger logger)
        throws ParseException
    {
        this.config = config;
        functionFields = document;
        this.prefix = prefix;
        this.phantomQueueId = phantomQueueId;
        this.queueName = queueName;
        this.logger = logger;
        this.orderIndependentUpdate = orderIndependentUpdate;
    }

    public HTMLDocument(
        final Document document,
        final Prefix prefix,
        long phantomQueueId,
        String queueName,
        final boolean orderIndependentUpdate,
        final DatabaseConfig config,
        final Logger logger)
        throws ParseException
    {
        this.config = config;
        functionFields = null;
        doc = document;
        this.prefix = prefix;
        this.logger = logger;
        primaryKey = extractPrimaryKey(doc);
        this.orderIndependentUpdate = orderIndependentUpdate;

        if (phantomQueueId == QueueShard.MAGIC_QUEUEID) {
            Fieldable queueIdField = doc.getField(Config.QUEUE_ID_FIELD_KEY);
            if (queueIdField != null) {
                try {
                    phantomQueueId = Long.parseLong(queueIdField.stringValue());
                } catch (RuntimeException e) {
                }
            }
        }
        this.phantomQueueId = phantomQueueId;

        if (queueName == null
            || queueName.equals(QueueShard.DEFAULT_SERVICE))
        {
            Fieldable queueNameField = doc.getField(Config.QUEUE_NAME_FIELD_KEY);
            if (queueNameField == null) {
                queueName = null;
            } else {
                queueName = queueNameField.stringValue();
            }
        }
        this.queueName = queueName;

        List<Fieldable> modifiedFields = null;
        for (Fieldable field: doc.fields()) {
            if (fieldConfigModified(field)) {
                if (modifiedFields == null) {
                    modifiedFields = new ArrayList<>(2);
                }
                modifiedFields.add(field);
            }
        }
        if (modifiedFields != null && modifiedFields.size() > 0) {
            for (Fieldable field: modifiedFields) {
                doc.removeField(field.name());
                setField(field.name(), field.stringValue());
            }
        }
        for (String rootAliasedField: config.rootAliasedFields()) {
            Fieldable field = doc.getField(rootAliasedField);
            if (field != null) {
                setField(field.name(), field.stringValue());
            }
        }
        setAuxFields();
    }

    protected void setAuxFields() {
        setField(Config.PREFIX_FIELD_KEY, prefix.toString());
        if (phantomQueueId != QueueShard.MAGIC_QUEUEID) {
            setField(Config.QUEUE_ID_FIELD_KEY, Long.toString(phantomQueueId));
        }
        if (queueName != null
            && !queueName.equals(QueueShard.DEFAULT_SERVICE))
        {
            setField(Config.QUEUE_NAME_FIELD_KEY, queueName);
        }
    }

    private Document createDocFromFunctions(ConditionsAccessor condAccessor)
        throws ParseException
    {
        try {
            doc = new Document();
            updateWith(functionFields, InvalidStateFieldAccessor.INSTANCE, condAccessor);
            setAuxFields();
            primaryKey = extractPrimaryKey(doc);
            return doc;
        } catch (FieldFunctionException e) {
            ParseException pce = new ParseException(
                "Error caught trying to access document field values",
                0);
            pce.initCause(e);
            throw pce;
        }
    }

    public long weight() {
        return this.weight;
    }

    public boolean orderIndependentUpdate() {
        return orderIndependentUpdate;
    }
}

