package ru.yandex.msearch;

/* An example of a very simple, multi-threaded HTTP server.
 * Implementation notes are in WebServer.html, and also
 * as comments in the source code.
 */

import java.io.*;
import java.net.*;
import java.util.*;

import java.util.logging.Logger;

import ru.yandex.http.util.ServerException;
import ru.yandex.http.util.YandexHttpStatus;

import ru.yandex.io.TaggedIOException;

import ru.yandex.logger.PrefixedLogger;

import ru.yandex.msearch.collector.outergroup.OuterGroupFunctionFactory;

import ru.yandex.search.prefix.PrefixParser;

public class HTTPServer implements HTTPConstants, Runnable {
    /* Where worker threads stand idle */
    private final Vector<HTTPWorker> threads = new Vector<HTTPWorker>();
    private final Vector<HTTPWorker> threadsAll = new Vector<HTTPWorker>();
    private final ServerSocket ss;
    private final PrefixedLogger logger;
    private final Logger accessLogger;
    private volatile boolean exit = false;

    public HTTPServer(
        final DatabaseManager databaseManager,
        final Config config,
        final OuterGroupFunctionFactory outerGroupFunctionFactory,
        final PrefixedLogger logger,
        final Logger accessLogger)
        throws IOException
    {
        this.logger = logger;
        this.accessLogger = accessLogger;
        /* start worker threads */
        for (int i = 0; i < config.http().workers(); ++i) {
            HTTPWorker w =
                new HTTPWorker(
                    this,
                    databaseManager,
                    config,
                    outerGroupFunctionFactory,
                    logger,
                    accessLogger);
            (new Thread(w, "HTTPWorker #"+i)).start();
            threads.addElement(w);
            threadsAll.addElement(w);
        }

        ss = new ServerSocket(config.http().port(), 10000);
        (new Thread(this, "HTTPAccepter")).start();

    }

    public int port() {
        return ss.getLocalPort();
    }

    public void close() throws IOException {
        exit = true;
        for (HTTPWorker w : threadsAll) {
            w.stop();
        }
        ss.close();
    }

    public void run()
    {
        try
	{
    	    while (!exit)
	    {
        	Socket s = ss.accept();
		long startTime = System.currentTimeMillis();
		if( exit ) break;

        	HTTPWorker w = null;
        	synchronized (threads)
		{
		    if (threads.isEmpty())
		    {
			logger.severe("HTTP_SERVER: Queue is full. Dropping connection");
			s.close();
            	    } else {
                	w = threads.elementAt(0);
                	threads.removeElementAt(0);
                	w.setSocket(s, startTime);
            	    }
        	}
    	    }
	    ss.close();
	} catch( IOException ioe )
	{
	    ioe.printStackTrace();
	    return;
	}
    }

    public void freeWorker(final HTTPWorker worker) {
        threads.addElement(worker);
    }

    public static class RequestContext implements ru.yandex.msearch.RequestContext {
        private Socket s;
        private String sessionId;
        private PrefixedLogger logger;

        public RequestContext(
            final Socket s,
            final String sessionId,
            final PrefixedLogger logger)
        {
            this.s = s;
            this.sessionId = sessionId;
            this.logger = logger;
        }

        public String sessionId() {
            return sessionId;
        }

        @Override
        public void checkAbort() throws IOException
        {
            if (!s.isConnected() || s.isClosed()) {
                logger.severe("Connection closed by client");
                throw new TaggedIOException(
                    TaggedIOException.createTag(),
                    new IOException(
                        "Connection closed by remote <requester> host. "
                            + "Aborting any processing."));
            }
            try
            {
                s.getInputStream().available();
//                s.sendUrgentData( 0 );
            }
            catch( IOException e )
            {
                logger.severe("Connection closed by client");
                throw new TaggedIOException(
                    TaggedIOException.createTag(),
                    new IOException(
                        "Connection closed by remote <requester> host. "
                            + "Aborting any processing."));
            }
        }

        @Override
        public PrefixedLogger logger() {
            return logger;
        }
    }
}


interface HTTPConstants {
    /** 2XX: generally "OK" */
    public static final int HTTP_OK = 200;
    public static final int HTTP_CREATED = 201;
    public static final int HTTP_ACCEPTED = 202;
    public static final int HTTP_NOT_AUTHORITATIVE = 203;
    public static final int HTTP_NO_CONTENT = 204;
    public static final int HTTP_RESET = 205;
    public static final int HTTP_PARTIAL = 206;

    /** 3XX: relocation/redirect */
    public static final int HTTP_MULT_CHOICE = 300;
    public static final int HTTP_MOVED_PERM = 301;
    public static final int HTTP_MOVED_TEMP = 302;
    public static final int HTTP_SEE_OTHER = 303;
    public static final int HTTP_NOT_MODIFIED = 304;
    public static final int HTTP_USE_PROXY = 305;

    /** 4XX: client error */
    public static final int HTTP_BAD_REQUEST = 400;
    public static final int HTTP_UNAUTHORIZED = 401;
    public static final int HTTP_PAYMENT_REQUIRED = 402;
    public static final int HTTP_FORBIDDEN = 403;
    public static final int HTTP_NOT_FOUND = 404;
    public static final int HTTP_BAD_METHOD = 405;
    public static final int HTTP_NOT_ACCEPTABLE = 406;
    public static final int HTTP_PROXY_AUTH = 407;
    public static final int HTTP_CLIENT_TIMEOUT = 408;
    public static final int HTTP_CONFLICT = 409;
    public static final int HTTP_GONE = 410;
    public static final int HTTP_LENGTH_REQUIRED = 411;
    public static final int HTTP_PRECON_FAILED = 412;
    public static final int HTTP_ENTITY_TOO_LARGE = 413;
    public static final int HTTP_REQ_TOO_LONG = 414;
    public static final int HTTP_UNSUPPORTED_TYPE = 415;

    /** 5XX: server error */
    public static final int HTTP_SERVER_ERROR = 500;
    public static final int HTTP_INTERNAL_ERROR = 501;
    public static final int HTTP_BAD_GATEWAY = 502;
    public static final int HTTP_UNAVAILABLE = 503;
    public static final int HTTP_GATEWAY_TIMEOUT = 504;
    public static final int HTTP_VERSION = 505;
}

