package ru.yandex.msearch;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.BufferedReader;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.io.PrintStream;
import java.lang.management.GarbageCollectorMXBean;
import java.lang.management.ManagementFactory;
import java.lang.management.MemoryMXBean;
import java.lang.management.MemoryUsage;
import java.net.InetAddress;
import java.net.Socket;
import java.net.URLDecoder;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Set;
import java.util.logging.Level;
import java.util.logging.Logger;

import com.tecnick.htmlutils.htmlentities.HTMLEntities;
import org.apache.http.HttpConnectionMetrics;
import org.apache.http.HttpInetConnection;
import org.apache.http.HttpStatus;
import org.apache.http.message.BasicHttpRequest;
import org.apache.http.protocol.HttpContext;
import org.apache.http.protocol.HttpCoreContext;
import org.apache.lucene.analysis.Analyzer;
import org.apache.lucene.document.Document;
import org.apache.lucene.index.DocsEnum;
import org.apache.lucene.index.Fields;
import org.apache.lucene.index.IndexReader;
import org.apache.lucene.index.MultiFields;
import org.apache.lucene.index.Term;
import org.apache.lucene.index.Terms;
import org.apache.lucene.index.TermsEnum;
import org.apache.lucene.queryParser.ParseException;
import org.apache.lucene.queryParser.QueryParser;
import org.apache.lucene.search.BooleanClause;
import org.apache.lucene.search.BooleanQuery;
import org.apache.lucene.search.DocIdSetIterator;
import org.apache.lucene.search.MatchAllDocsQuery;
import org.apache.lucene.search.PrefixQuery;
import org.apache.lucene.search.Query;
import org.apache.lucene.search.TermQuery;
import org.apache.lucene.search.TermRangeQuery;
import org.apache.lucene.util.Bits;
import org.apache.lucene.util.BytesRef;
import org.apache.lucene.util.StringHelper;
import org.apache.lucene.util.Version;

import ru.yandex.http.util.ServerException;
import ru.yandex.http.util.SynchronizedHttpContext;
import ru.yandex.http.util.YandexHttpStatus;
import ru.yandex.http.util.server.HttpServer;
import ru.yandex.http.util.server.HttpSessionInfo;
import ru.yandex.http.util.server.JsonStatsConsumer;
import ru.yandex.http.util.server.LoggingServerConnection;
import ru.yandex.io.TaggedIOException;
import ru.yandex.json.writer.JsonType;
import ru.yandex.json.writer.JsonTypeExtractor;
import ru.yandex.json.writer.JsonWriter;
import ru.yandex.logger.PrefixedLogger;
import ru.yandex.logger.SearchProxyAccessLoggerConfigDefaults;
import ru.yandex.msearch.collector.DocCollector;
import ru.yandex.msearch.collector.MergeFunc;
import ru.yandex.msearch.collector.MergeFuncFactory;
import ru.yandex.msearch.collector.ParametrizedDocCollector;
import ru.yandex.msearch.collector.PruningCollector;
import ru.yandex.msearch.collector.SortedCollector;
import ru.yandex.msearch.collector.YaDoc3;
import ru.yandex.msearch.collector.group.GroupFunc;
import ru.yandex.msearch.collector.group.GroupFuncFactory;
import ru.yandex.msearch.collector.outergroup.OuterGroupFunction;
import ru.yandex.msearch.collector.outergroup.OuterGroupFunctionFactory;
import ru.yandex.msearch.collector.outergroup.XURLSOuterDeduplicator;
import ru.yandex.msearch.collector.sort.SimpleSortFunc;
import ru.yandex.msearch.collector.sort.SortFunc;
import ru.yandex.msearch.collector.sort.SortFuncFactory;
import ru.yandex.msearch.config.DatabaseConfig;
import ru.yandex.msearch.jobs.IndexCopyJob;
import ru.yandex.msearch.jobs.Job;
import ru.yandex.msearch.util.Compress;
import ru.yandex.parser.string.BooleanParser;
import ru.yandex.queryParser.MultiFieldQueryParser;
import ru.yandex.queryParser.YandexQueryParser;
import ru.yandex.queryParser.YandexQueryParserFactory;
import ru.yandex.search.NullScorerFactory;
import ru.yandex.search.prefix.Prefix;
import ru.yandex.search.prefix.PrefixParser;
import ru.yandex.stater.RequestInfo;
import ru.yandex.stater.RequestsStater;
import ru.yandex.util.timesource.TimeSource;

public class HTTPWorker implements HTTPConstants, Runnable {

    private static final int BUF_SIZE = 2048;
    private static final byte[] EOL = {(byte) '\r', (byte) '\n'};

    /* buffer to use for requests */
    private final byte[] buf = new byte[BUF_SIZE];
    /* Socket to client we're handling */
    private final HTTPServer server;
    private final DatabaseManager databaseManager;
    private final Config config;
    private final OuterGroupFunctionFactory outerGroupFunctionFactory;
    private volatile boolean exit = false;
    private long startTime = 0L;
    private Socket s;
    private String requestType;
    private String requestString;
    private HTTPServer.RequestContext ctx;
    private final RequestsStater stater;
    private final PrefixedLogger logger;
    private final Logger accessLogger;

    private static final HashSet<String> dumpingShards = new HashSet<>();

    public HTTPWorker(
        final HTTPServer server,
        final DatabaseManager databaseManager,
        final Config config,
        final OuterGroupFunctionFactory outerGroupFunctionFactory,
        final PrefixedLogger logger,
        final Logger accessLogger) {
        this.server = server;
        this.databaseManager = databaseManager;
        this.config = config;
        this.outerGroupFunctionFactory = outerGroupFunctionFactory;
        this.logger = logger;
        this.accessLogger = accessLogger;
        stater = config.oldRequestsStater();
        s = null;
    }

    synchronized void setSocket(Socket s, long startTime) {
        this.startTime = startTime;
        this.s = s;
        notify();
    }

    public void run() {
        while (!exit) {
            synchronized (this) {
                if (s == null) {
                    /* nothing to do */
                    try {
                        wait();
                    } catch (InterruptedException e) {
                        Thread.currentThread().interrupt();
                    }
                    continue;
                }
            }
            try {
                try (Socket socket = s) {
                    handleClient();
                } catch (Exception e) {
                    e.printStackTrace();
                }
            } catch (java.lang.OutOfMemoryError error) {
            }
            /* go back in wait queue if there's fewer
             * than numHandler connections.
             */
            s = null;
            server.freeWorker(this);
        }
    }

    synchronized void stop() {
        exit = true;
        notify();
    }

    private static void parseRequestString(
        final String request,
        final Map<String, String> requestParams)
        throws IOException {
        String[] cols = request.split("&");
        for (int i = 0; i < cols.length; i++) {
            int colon;
            String name, value;
            colon = cols[i].indexOf('=');
            if (colon == -1) {
                name = cols[i];
                value = null;
            } else {
                name = cols[i].substring(0, colon);
                value = URLDecoder.decode(cols[i].substring(colon + 1), "utf8");
            }
            requestParams.put(name, value);
        }
    }

    private void parseHttpRequest(
        final InputStream is,
        final PrintStream os,
        final Map<String, String> httpHeaders,
        final Map<String, String> requestParams)
        throws IOException {
        BufferedReader reader = new BufferedReader(new InputStreamReader(is));
        String line = reader.readLine();
        if (line == null) {
            throw new IOException("Empty request");
        }
        int space1 = line.indexOf(' ');
        if (space1 == -1) {
            throw new IOException("invalid http request: " + line);
        }
        requestType = line.substring(0, space1);
        int space2 = line.lastIndexOf(' ');
        if (space2 == -1) {
            throw new IOException("invalid http request (no http version tag found): " + line);
        }
        requestString = line.substring(space1 + 1, space2);

        //now read headers until end
        while (true) {
            line = reader.readLine();
            if (line == null) {
                throw new IOException("Error reading http request");
            }
            if (line.trim().length() == 0) {
                break;
            }
            String name, value;
            int colon = line.indexOf(':');
            if (colon == -1) {
                name = line.toLowerCase();
                ;
                value = "";
            } else {
                name = line.substring(0, colon).toLowerCase();
                value = line.substring(colon + 1);
                for (int i = 0; i < value.length(); ++i) {
                    if (!Character.isWhitespace(value.charAt(i))) {
                        value = value.substring(i);
                        break;
                    }
                }
            }
            httpHeaders.put(name, value);
        }

        if (requestType.equalsIgnoreCase("GET")) {
            int colon = requestString.indexOf('?');
            if (colon != -1) {
                parseRequestString(
                    requestString.substring(colon + 1),
                    requestParams);
            }
        } else if (requestType.equalsIgnoreCase("POST")) {
            String contentType = httpHeaders.get("content-type");
            if (!contentType.equalsIgnoreCase("application/x-www-form-urlencoded")) {
                throw new IOException("Error parsing http request (only application/x-www-form-urlencoded content " +
                    "type is supported in POST, while recceived is: " + contentType + ")");
            }
            while (true) {
                line = reader.readLine();
                if (line == null) {
                    break;
                }
                parseRequestString(line, requestParams);
            }
        }
    }

    private double getDouble(final Map<String, String> params,
                             final String name, final double defaultValue)
        throws NumberFormatException {
        String value = params.get(name);
        if (value != null) {
            return Double.parseDouble(value);
        } else {
            return defaultValue;
        }
    }

    private int getInt(final Map<String, String> params,
                       final String name, final int defaultValue)
        throws NumberFormatException {
        String value = params.get(name);
        if (value != null) {
            return Integer.parseInt(value);
        } else {
            return defaultValue;
        }
    }

    private Integer
    getInt(final Map<String, String> params,
           final String name, final Integer defaultValue)
        throws NumberFormatException {
        String value = params.get(name);
        if (value != null) {
            return Integer.parseInt(value);
        } else {
            return defaultValue;
        }
    }

    private static boolean getBoolean(final Map<String, String> params,
                                      final String name, final boolean defaultValue)
        throws Exception {
        String value = params.get(name);
        if (value != null) {
            return BooleanParser.INSTANCE.apply(value);
        } else {
            return defaultValue;
        }
    }

    private String getString(final Map<String, String> params,
                             final String name, final String defaultValue)
        throws NumberFormatException {
        String value = params.get(name);
        if (value != null) {
            return value;
        } else {
            return defaultValue;
        }
    }

    void handleClient() throws Exception, IOException {
        String sessionId =
            Long.toHexString((long) (Math.random() * Long.MAX_VALUE));
        ctx = new HTTPServer.RequestContext(
            s,
            sessionId,
            logger.replacePrefix(sessionId));
        try (InputStream is = new BufferedInputStream(s.getInputStream());
             OutputStream os = new BufferedOutputStream(s.getOutputStream());
             PrintStream ps = new PrintStream(os, false, "UTF-8"))
        {
            int retval = 0;
            int statusCode = 200;
            Map<String, String> httpHeaders = new HashMap<>();
            Map<String, String> requestParams = new HashMap<>();

            parseHttpRequest(is, ps, httpHeaders, requestParams);
            try {
                s.setSoTimeout(config.http().timeout());
                s.setTcpNoDelay(true);

                if (ctx.logger().isLoggable(Level.INFO)) {
                    ctx.logger().info(
                        "FROM " + s.getInetAddress().getHostAddress() + ": "
                            + requestType + " " + requestString);
                }

                Compress.resetStats();
                retval = handleRequest(ctx, requestParams, ps, os, requestString, true, false);
            } catch (Exception e) {
                ctx.logger().log(Level.SEVERE, "", e);
                retval = 0;
                statusCode = 500;
                if (e instanceof ServerException) {
                    statusCode = ((ServerException) e).statusCode();
                } else if (e instanceof TaggedIOException) {
                    statusCode = YandexHttpStatus.SC_CLIENT_CLOSED_REQUEST;
                }
                printHeader(ps, requestString, retval);
            } finally {
                try {
                    RequestInfo info = new RequestInfo(
                        TimeSource.INSTANCE.currentTimeMillis(),
                        statusCode,
                        startTime,
                        startTime,
                        0L,
                        0L);
                    if (ctx.logger().isLoggable(Level.INFO)) {
                        ctx.logger().info("IOStats: " + Compress.stats());
                        ctx.logger().info("Total execution time from request: "
                            + requestString + " : " + info.requestTime()
                            + " : " + retval);
                    }
                    HttpContext context = new SynchronizedHttpContext();
                    HttpSessionInfo sessionInfo = new HttpSessionInfo(context);
                    for (Map.Entry<String, String> header : httpHeaders.entrySet()) {
                        sessionInfo.put(
                            "http_"
                                + header.getKey().toLowerCase(Locale.ENGLISH)
                                .replace('-', '_'),
                            header.getValue());
                    }
                    sessionInfo.put(
                        LoggingServerConnection.STATUS,
                        Integer.toString(statusCode));
                    sessionInfo.put(
                        SearchProxyAccessLoggerConfigDefaults.HITS_COUNT,
                        Integer.toString(retval));
                    sessionInfo.put(
                        LoggingServerConnection.REQUEST_TIME,
                        Long.toString(info.requestTime()));
                    sessionInfo.put(
                        LoggingServerConnection.PROCESSING_TIME,
                        Long.toString(info.processingTime()));

                    context.setAttribute(
                        HttpCoreContext.HTTP_CONNECTION,
                        new FakeInetConnection(s));
                    context.setAttribute(
                        HttpCoreContext.HTTP_REQUEST,
                        new BasicHttpRequest(requestType, requestString));
                    context.setAttribute(HttpServer.SESSION_ID, ctx.sessionId());

                    accessLogger.log(Level.INFO, "", sessionInfo);
                    if (!requestString.equals("/ping")
                        && !requestString.equals("/stat")) {
                        stater.accept(info);
                    }
                } catch (Exception ign) {
                    ctx.logger().log(Level.SEVERE, "", ign);
                }
            }
        } catch (Exception e) {
            ctx.logger().log(Level.SEVERE, "", e);
        } finally {
            ctx = null;
        }
    }

    Analyzer getAnalyzer(Prefix prefix, final DatabaseConfig config) throws Exception {
        PrefixingAnalyzerWrapper analyzer = DefaultAnalyzerFactory.SEARCH.apply(config);
        analyzer.setPrefix(prefix.toString());
        return analyzer;
    }

    private final static String[] WEB_SEARCH_FIELDS = {
        "mid",
        "suid",
        "body_text",
        "pure_body",
        "hdr_subject",
        "hdr_from",
        "hdr_to",
        "hdr_cc",
        "hdr_bcc",
        "hdr_reply_to",
        "attachname",
        "album",
        "artist",
        "author",
        "comment",
        "composer",
        "description",
        "genre",
        "keywords",
        "released",
        "subject",
        "title"
    };
    private final static String[] IMAP_SEARCH_FIELDS = {"mid", "suid", "body_text", "pure_body", "hdr_subject",
        "hdr_from", "hdr_to", "hdr_cc", "hdr_bcc", "hdr_reply_to", "attachname", "headers", "url"};

    Query createSimpleQuery(String query, Prefix user, DatabaseConfig config) throws Exception {
        String searchText = HTMLEntities.unhtmlentities(query);
        Analyzer analyzer = getAnalyzer(user, config);
        Query q;
        QueryParser parser =
            new MultiFieldQueryParser(
                new YandexQueryParserFactory(config),
                analyzer,
                IMAP_SEARCH_FIELDS);
        parser.setAllowLeadingWildcard(true);
        parser.setAnalyzeRangeTerms(true);
        parser.setAutoGeneratePhraseQueries(true);
        try {
            q = parser.parse(searchText);
        } catch (Exception e) {
            int sep = searchText.indexOf(':');
            if (sep != -1) {
                q = new TermQuery(new Term(searchText.substring(0, sep), searchText.substring(sep + 1)));
            } else {
                q = new TermQuery(new Term("body_text", searchText));
            }
        }
        if (ctx.logger().isLoggable(Level.FINE)) {
            ctx.logger().fine("PreQuery: " + q.toString());
        }
        return q;
    }

    Query createSuperSimpleQuery(String query, Prefix user, final DatabaseConfig config) throws Exception {
        String searchText = HTMLEntities.unhtmlentities(query);
        Analyzer analyzer = getAnalyzer(user, config);
        Query q;
        QueryParser parser = new QueryParser(Version.LUCENE_40, "", analyzer);
        parser.setAllowLeadingWildcard(true);
        parser.setAnalyzeRangeTerms(true);
        parser.setAutoGeneratePhraseQueries(true);
        try {
            q = parser.parse(searchText);
        } catch (Exception e) {
            int sep = searchText.indexOf(':');
            if (sep != -1) {
                q = new TermQuery(new Term(searchText.substring(0, sep), searchText.substring(sep + 1)));
            } else {
                q = new TermQuery(new Term("body_text", searchText));
            }
        }
        if (ctx.logger().isLoggable(Level.FINE)) {
            ctx.logger().fine("PreQuery: " + q.toString());
        }
        return q;
    }

    private boolean wildCardAllowed(Map<String, String> params) {
        if (params.containsKey("imap") && params.get("imap").equals("1")) {
            return false;
        }
        if (params.containsKey("only_atta") && params.get("only_atta").equals("yes")) {
            return false;
        }
        String searchText = params.get("text");
        if (searchText == null) {
            searchText = "";
        }
        if (searchText.equals("$$$") || searchText.length() == 0) {
            return false;
        }
        return true;
    }

    private Query prepareSuidQuery(
        final Prefix user,
        final boolean dated,
        final boolean datePrefixed,
        final DatabaseConfig config)
        throws ParseException {
        if (dated && datePrefixed) {
            return new MatchAllDocsQuery();
        } else {
            String suid = "suid";
            FieldConfig suidConfig = config.fieldConfig(suid);
            String prefix = suidConfig != null && suidConfig.prefixed() ? user.toString() + '#' : "";
            return new TermQuery(new Term(suid, prefix + user.toString()));
        }
    }

    Query createQuery(
        Map<String, String> params,
        Prefix user,
        boolean wildcard,
        ParametrizedDocCollector col,
        final DatabaseConfig config)
        throws Exception
    {
        String searchText = params.get("text");
        if (searchText == null) {
            searchText = "";
        }

        boolean imapSearch = false;

        if (params.containsKey("imap") && params.get("imap").equals("1")) {
            imapSearch = true;
        }

        searchText = HTMLEntities.unhtmlentities(searchText);

        Analyzer analyzer = getAnalyzer(user, config);

        Query query;
        long lemmerTime = System.currentTimeMillis();
        String date = "received_date";
        FieldConfig dateConfig = config.fieldConfig(date);
        boolean datePrefixed = dateConfig != null && dateConfig.prefixed();

        boolean dated;
        long fromTS;
        long toTS;
        if (params.containsKey("dated")) {
            int fromYear = 0;
            int fromMon = 0;
            int fromDay = 0;
            int toYear = 3000;
            int toMon = 11;
            int toDay = 30;

            try {
                if (params.containsKey("from_year")) {
                    fromYear = Integer.parseInt(params.get("from_year").toString());
                }
                if (params.containsKey("from_month")) {
                    fromMon = Integer.parseInt(params.get("from_month").toString());
                }
                if (params.containsKey("from_day")) {
                    fromDay = Integer.parseInt(params.get("from_day").toString());
                }

                if (params.containsKey("to_year")) {
                    toYear = Integer.parseInt(params.get("to_year").toString());
                }
                if (params.containsKey("to_month")) {
                    toMon = Integer.parseInt(params.get("to_month").toString());
                }
                if (params.containsKey("to_day")) {
                    toDay = Integer.parseInt(params.get("to_day").toString());
                }
            } catch (Exception e) {
                e.printStackTrace();
            }
            Calendar cal = Calendar.getInstance();
            cal.clear();
            cal.set(fromYear, fromMon, fromDay);
            fromTS = cal.getTimeInMillis() / 1000L;
            cal.clear();
            cal.set(toYear, toMon, toDay);
            toTS = cal.getTimeInMillis() / 1000L + 86400L;
            dated = toTS > fromTS && (fromTS != 0L || toTS != Long.MAX_VALUE);
        } else {
            dated = false;
            fromTS = 0L;
            toTS = 0L;
        }

        if (searchText.equals("$$$")) {
            query = prepareSuidQuery(user, dated, datePrefixed, config);
            wildcard = true;
        } else if (params.containsKey("only_atta") && params.get("only_atta").equals("yes")) {
            query = new PrefixQuery(new Term("attachname", user.toString() + "#"));
            wildcard = true;
        } else {
            String[] fields;
            if (imapSearch) {
                fields = IMAP_SEARCH_FIELDS;
            } else {
                fields = WEB_SEARCH_FIELDS;
            }

            if (params.containsKey("scope")) {
                String scope = params.get("scope");
                fields = new String[1];
                fields[0] = scope;
            }


            if (!imapSearch) {
                // TODO: escape this characters instead of removing
                searchText = searchText.replaceAll("[*\\\\(){}\\[\\]'?~\u00a0+:!^-]", " ");
                if (wildcard) {
                    searchText = searchText.replace('"', ' ');
                }
                searchText = searchText.toLowerCase(Locale.ENGLISH).trim();
                ctx.logger().fine("SEARCH TEXT: " + searchText);
            }


            if (searchText.length() == 0) {
                query = prepareSuidQuery(user, dated, datePrefixed, config);
                wildcard = true;
            } else {
                String origText = searchText.replace('"', ' ').trim();
                if (!imapSearch) {
                    if (!wildcard) {
                        String[] text = searchText.split("\"");
                        for (int i = 0; i < text.length; ++i) {
                            text[i] = text[i].trim();
                            if ((i & 1) == 0) {
                                text[i] = text[i].replaceAll(" +", " AND ");
                            } else {
                                text[i] = '"' + text[i] + '"';
                            }
                        }
                        StringBuilder sb = new StringBuilder();
                        for (String token : text) {
                            if (!token.isEmpty()) {
                                if (sb.length() > 0) {
                                    sb.append(" AND ");
                                }
                                sb.append(token);
                            }
                        }
                        searchText = sb.toString();
                        if (ctx.logger().isLoggable(Level.FINE)) {
                            ctx.logger().fine("Search query rewritten: "
                                + searchText);
                        }
                    } else {
                        String[] words = searchText.split(" ");
                        if (words.length > 0) {
                            searchText = "";
                            for (int i = 0; i < words.length; i++) {
                                if (words[i].trim().length() == 0) {
                                    continue;
                                }
                                if (searchText.length() > 0) {
                                    searchText += " AND ";
                                }
                                searchText += words[i].trim();
                            }
                            searchText += "*";
                        } else {
                            searchText += "*";
                        }
                    }
                }

                QueryParser parser = new MultiFieldQueryParser(
                    new YandexQueryParserFactory(config), analyzer, fields);
                parser.setAutoGeneratePhraseQueries(true);
                if (imapSearch) {
                    parser.setAllowLeadingWildcard(true);
                    parser.setAnalyzeRangeTerms(true);
                }
                try {
                    query = parser.parse(searchText);
                    if (!imapSearch) {
                        BooleanQuery attachtypesQuery = new BooleanQuery(true);
                        attachtypesQuery.add(query, BooleanClause.Occur.SHOULD);
                        attachtypesQuery.add(
                            new YandexQueryParser(
                                Version.LUCENE_40,
                                "attachtype",
                                analyzer,
                                config)
                                .parse(origText),
                            BooleanClause.Occur.SHOULD);
                        query = attachtypesQuery;
                    }
                } catch (Exception e) {
                    e.printStackTrace();
                    query = new TermQuery(new Term("body_text", searchText));
                }
            }
        }

        if (dated) {
            String prefix = datePrefixed ? user.toString() + '#' : "";
            Query dateRange = new TermRangeQuery(date, new BytesRef(prefix + Long.toString(fromTS)),
                new BytesRef(prefix + Long.toString(toTS)), true, false);
            if (query instanceof MatchAllDocsQuery) {
                query = dateRange;
            } else {
                BooleanQuery boolQuery = new BooleanQuery(true);
                boolQuery.add(dateRange, BooleanClause.Occur.MUST);
                boolQuery.add(query, BooleanClause.Occur.MUST);
                query = boolQuery;
            }
        }

        String queryToString = query.toString();
        lemmerTime = System.currentTimeMillis() - lemmerTime;
        if (ctx.logger().isLoggable(Level.FINE)) {
            ctx.logger().fine("Query(" + lemmerTime + "): " + queryToString);
        }
        return query;
    }

    public void singleSearch(final Index index, Prefix suid, Query query, DocCollector col) throws IOException {
        col.setPrefix(suid);
        index.search(query, col, suid);
    }

    private void search(final Index index, Map<String, String> params, String query, String request, ParametrizedDocCollector col,
                        boolean wildcard) throws Exception, IOException {
        final boolean checkCopyness =
            getBoolean(params, "check-copyness", index.config().checkCopyness());
        Query q;
        Prefix prefix = index.config().prefixParser().apply(params.get("user"));
        if (ctx.logger().isLoggable(Level.FINE)) {
            ctx.logger().fine("Prefix: " + prefix + ", hash: " + prefix.hash()
                + ", shard: " + (prefix.hash() % 1000));
        }
        if (checkCopyness) {
            try {
                index.checkShardCopied(prefix);
            } catch (IOException e) {
                throw new ServerException(
                    HttpStatus.SC_SERVICE_UNAVAILABLE, e);
            }
        }
        if (query == null) {
            q = createQuery(params, prefix, !wildcard, col, index.config());
        } else {
            q = createSimpleQuery(query, prefix, index.config());
        }
        singleSearch(index, prefix, q, col);
    }

    public static int parseDumpIndex(
        final Map<String, String> params,
        final PrintStream ps,
        final OutputStream os,
        final Index index,
        final RequestContext ctx)
        throws IOException, Exception
    {
        String inShardsStr = params.get("innershards");
        String outShardsStr = params.get("outshards");
        if (inShardsStr == null || outShardsStr == null) {
            ps.println("shards parameters are not specified");
            return 0;
        }
        String in[] = inShardsStr.split("-");
        String out[] = outShardsStr.split(":");
        if (in.length != 2 || out.length != 2) {
            ps.println("invalid shards parameters");
            return 0;
        }
        String shardingFieldsStr = params.get("sharding-fields");
        Set<String> shardingFields = new HashSet<String>();
        if (shardingFieldsStr != null) {
            for (String field : shardingFieldsStr.split(",")) {
                shardingFields.add(field);
            }
        } else {
            shardingFields.add("suid");
            shardingFields.add("__prefix");
        }
        int inStart;
        int inEnd;
        int outTotal;
        int outShard;

        try {
            inStart = Integer.parseInt(in[0]);
            inEnd = Integer.parseInt(in[1]);
            outTotal = Integer.parseInt(out[0]);
            outShard = Integer.parseInt(out[1]);
        } catch (Exception e) {
            ps.println("invalid shards parameters");
            return 0;
        }
        boolean lockIndex = getBoolean(params, "lock-index", false);
        //TODO: Remove this
        final String dumpLockKey = inShardsStr + '#' + outShardsStr;
        synchronized (dumpLockKey) {
            if (!dumpingShards.add(dumpLockKey)) {
                ps.println("Shards: " + lockIndex + " is dumping now."
                    + "Parallel dumps is prohibited");
                if (ctx.logger().isLoggable(Level.FINE)) {
                    ctx.logger().fine("Shards: " + lockIndex
                        + " is dumping now."
                        + "Parallel dumps is prohibited");
                }
                return 0;
            }
        }
        try {
            dumpIndex(
                os,
                inStart,
                inEnd,
                outTotal,
                outShard,
                shardingFields,
                lockIndex,
                index,
                ctx);
        } finally {
            synchronized (dumpLockKey) {
                dumpingShards.remove(dumpLockKey);
            }
        }
        return 1;
    }


    public static void dumpIndex(
        OutputStream ps,
        int innerStart,
        int innerEnd,
        int outTotal,
        int outShard,
        Set<String> shardingFields,
        boolean lockIndex,
        final Index index,
        final RequestContext ctx)
        throws IOException
    {
        if (!index.dumpAllowed()) {
            throw new IOException("Index is empty (was not copied) dump is forbidden");
        }
        IndexDumper dumper =
            new IndexDumper(
                index,
                outShard,
                outTotal,
                innerStart,
                innerEnd,
                shardingFields,
                lockIndex,
                ctx.logger());
        dumper.dump(ps);
        ps.flush();
    }

    private void copyIndex(
        Map<String, String> params,
        PrintStream ps,
        final Index index)
        throws Exception
    {
        String from = params.get("from");
        String shards = params.get("shards");
        String threadsStr = params.get("threads");
        if (from == null) {
            ps.println("<from> parameter is not specified");
            return;
        }
        if (shards == null) {
            ps.println("<shards> parameter is not specified");
            return;
        }
        int threads = 1;
        if (threadsStr != null) {
            threads = Integer.parseInt(threadsStr);
        }
        int timeout = getInt(params, "timeout", 60 * 60);
        int retryCount = getInt(params, "retry-count", 5);

        String[] shardsParts = shards.split("-");
        if (shardsParts.length != 2) {
            ps.println("invalid <shards> parameter");
            return;
        }
        int startShard = Integer.parseInt(shardsParts[0]);
        int endShard = Integer.parseInt(shardsParts[1]);

        String[] urls = from.split(";");
        boolean lockIndex = getBoolean(params, "lock-index", true);
        String shardingFieldsStr =
            getString(params, "sharding-fields", "__prefix,suid");
        Set<String> shardingFields = new HashSet<String>();
        for (String f : shardingFieldsStr.split(",")) {
            shardingFields.add(f);
        }
        boolean checkDups = getBoolean(params, "check-dups", true);
        boolean replace = getBoolean(params, "replace", false);
        boolean disableIndexing = getBoolean(params, "disable-indexing", true);

        Job job = Job.createIndexCopyJob(index, startShard, endShard, urls, threads, timeout, retryCount,
            shardingFields, lockIndex, checkDups, replace, disableIndexing);
        if (job == null) {
            ps.println("can't create copy job");
        }
        if (params.containsKey("abort") || params.containsKey("remove")) {
            job = index.getJobsManager().getJob(job);
            if (job != null) {
                job.stop();
                index.getJobsManager().removeJob(job);
            } else {
                ps.println("Job does not exists");
            }
        } else if (params.containsKey("stop")) {
            job = index.getJobsManager().getJob(job);
            if (job != null) {
                job.stop();
            } else {
                ps.println("Job does not exists");
            }
        } else if (params.containsKey("update")) {
            IndexCopyJob newJob = (IndexCopyJob) job;
            job = index.getJobsManager().addJob(job);
            if (job != newJob) {
                job.update(newJob);
            } else {
                ps.println("Job does not exists");
            }
        } else if (params.containsKey("status")) {
            job = index.getJobsManager().getJob(job);
            if (job == null) {
                ps.println("Job does not exists");
            }
        } else {
            IndexCopyJob newJob = (IndexCopyJob) job;
            job = index.getJobsManager().addJob(job);
            if (job != newJob) {
                job.update(newJob);
            }
            job.start();
        }
        if (job != null) {
            printHeaderHtml(ps, "copyIndex request");
            job.printStats(ps);
        }
    }

    private void removeShards(final Index index, int start, int stop, Map<String, String> params, PrintStream ps) throws IOException {
        int threads = 1;
        String threadsStr = params.get("threads");
        if (threadsStr != null) {
            threads = Integer.parseInt(threadsStr);
        }
        String shardingFieldsStr =
            getString(params, "sharding-fields", "__prefix,suid");
        Set<String> shardingFields = new HashSet<String>();
        for (String f : shardingFieldsStr.split(",")) {
            shardingFields.add(f);
        }

        Job job = Job.createShardsRemoverJob(index, start, stop, threads, shardingFields);
        if (job == null) {
            ps.println("can't create copy job");
        }
        if (params.containsKey("abort") || params.containsKey("remove")) {
            job = index.getJobsManager().getJob(job);
            if (job != null) {
                job.stop();
                index.getJobsManager().removeJob(job);
            } else {
                ps.println("Job does not exists");
            }
        } else if (params.containsKey("stop")) {
            job = index.getJobsManager().getJob(job);
            if (job != null) {
                job.stop();
            } else {
                ps.println("Job does not exists");
            }
        } else if (params.containsKey("status")) {
            job = index.getJobsManager().getJob(job);
            if (job == null) {
                ps.println("Job does not exists");
            }
        } else {
            job = index.getJobsManager().addJob(job);
            job.start();
        }
        if (job != null) {
            printHeaderHtml(ps, "removerShards request");
            job.printStats(ps);
        }
    }

    int handleRequest(
        HTTPServer.RequestContext ctx,
        Map<String, String> params,
        PrintStream ps,
        OutputStream os,
        String request,
        boolean digits,
        boolean wildcard)
        throws Exception, IOException
    {
        if (request.equals("/ping")) {
            printHeader(ps, request);
            ps.println("pong");
            return 0;
        }

        if (request.equals("/stat")) {
            JsonType jsonType =
                JsonTypeExtractor.HUMAN_READABLE
                    .extract(
                        getString(params, "json-type", null),
                        getBoolean(params, "hr", true));
            printHeader(ps, request);
            JsonWriter jsonWriter =
                jsonType.create(new OutputStreamWriter(ps));
            JsonStatsConsumer statsConsumer =
                new JsonStatsConsumer(jsonWriter);
            stater.stats(statsConsumer);
            statsConsumer.close();
            jsonWriter.close();
            return 0;
        }

        if (request.equals("/rotatelogs")) {
            ctx.logger().info("Rotating logs");
            ctx.logger().info("Nothing is rotated, use new search port");
            ps.println("ok");
            return 0;
        }

        String dbname = params.getOrDefault("db", DatabaseManager.DEFAULT_DATABASE);
        Index index = databaseManager.index(dbname);

        if (params.containsKey("getshardnum")) {
            Prefix user;
            printHeader(ps, request);
            if (params.containsKey("user")) {
                user = index.config().prefixParser().apply(params.get("user"));
            } else {
                return 0;
            }
            int shards = index.config().shards();
            if (ctx.logger().isLoggable(Level.INFO)) {
                ctx.logger().info("User: " + user + ", shard: "
                    + (user.hash() % shards));
            }
            ps.println(user.hash() % shards);
            return 0;
        }

        if (params.containsKey("getshardmem")) {
            int shard;
            printHeader(ps, request);
            if (params.containsKey("shard")) {
                shard = Integer.parseInt(params.get("shard"));
            } else {
                return 0;
            }
            AbstractPart.PartsStats stats = index.getShard(shard).getPartsStats();
            if (ctx.logger().isLoggable(Level.INFO)) {
                ctx.logger().info(
                    "Shard: " + shard + " memory usage: "
                        + stats.toString());
            }
            ps.println(stats.toString());
            return 0;
        }

        if (params.containsKey("removeshards")) {
            String shardsStr = params.get("shards");
            if (shardsStr == null) {
                ps.println("no shards parameter specified");
                return 0;
            }
            String[] range = shardsStr.split("-");
            if (range.length != 2) {
                ps.println("invalid shards parameter");
                return 0;
            }

            int start = Integer.parseInt(range[0]);
            int stop = Integer.parseInt(range[1]);
            removeShards(index, start, stop, params, ps);
            return 0;
        }

        SearchRequest searchRequest =
            new SearchRequestBase(ctx, index, index.config());
        if (params.containsKey("checkxurls")) {
            String text = params.get("text");
            ((XURLSOuterDeduplicator)
                outerGroupFunctionFactory.apply(
                    "x_urls",
                    searchRequest.fieldToIndex()))
                .checkString(text);
            ps.println("ok");
            return 0;
        }

        if (params.containsKey("reload_xurls_patterns")) {
            try {
                outerGroupFunctionFactory.reloadRules(ctx.logger());
            } catch (Exception e) {
                printHeader(ps, request);
                ps.println(e.getMessage());
                throw e;
            }
            printHeader(ps, request);
            ps.println("ok");
            return 0;
        }


        if (params.containsKey("copyindex")) {
            copyIndex(params, ps, index);
            return 0;
        }

        if (params.containsKey("listjobs")) {
            boolean json = getBoolean(params, "json", false);
            printHeader(ps, request);
            List<Job> jobs = index.getJobsManager().getJobs();
            for (Job job : jobs) {
                if (json) {
                    ps.println(job.jsonString());
                } else {
                    ps.println(job.toString());
                }
            }
            return 0;
        }

        if (params.containsKey("dumpindex")) {
            return parseDumpIndex(params, ps, os, index, ctx);
        }


        if (params.containsKey("rungc")) {
            System.gc();
            printHeader(ps, request);
            ps.println("OK");
            return 0;
        }

        if (params.containsKey("forceclose")) {
            printHeader(ps, request);
            index.close();
            ps.println("OK");
            return 0;
        }

        if (params.containsKey("optimize")) {
            int optimize = getInt(params, "optimize", 3);
            int parallel = getInt(params, "parallel", 3);
            int startShard = getInt(params, "start_shard", 0);
            int endShard = getInt(params, "end_shard", index.shardsCount() - 1);
            printHeader(ps, request);
            ps.println("Running optimize to " + optimize + " segments");
            ps.flush();
            index.optimize(optimize, ps, parallel, startShard, endShard);
            ps.println("Optimize finished");
            ps.flush();
            return 0;
        }

        if (params.containsKey("expunge")) {
            int parallel = getInt(params, "parallel", 3);
            Integer version = getInt(params, "setversion", null);
            Boolean all = getBoolean(params, "expunge_all", false);
            Boolean convert = getBoolean(params, "convert", false);
            int startShard = getInt(params, "start_shard", 0);
            int endShard = getInt(params, "end_shard", index.shardsCount() - 1);
            double minPct = getDouble(params, "min_pct", 0);

            printHeader(ps, request);
            ps.println("Running expunge in " + parallel +
                " threads setting version to: " + version);
            ps.flush();
            index.expunge(ps, parallel, version, all, convert, minPct, startShard, endShard);
            ps.println("expunge finished");
            ps.flush();
            return 0;
        }

        if (params.containsKey("listattaches")) {
            int count = 0;
            Prefix user;
            String sortby = "name";
            printHeader(ps, request);

            if (params.containsKey("user")) {
                user = index.config().prefixParser().apply(params.get("user"));
            } else {
                return 0;
            }


            IndexReader reader = null;
            TermsEnum te = null;
            DocsEnum td = null;
            Bits skipDocs = new Bits.MatchNoBits(0);
            Searcher searcher = null;
            try {
                BytesRef term;
                BytesRef prefix = new BytesRef(user.toString() + '#');
                ArrayList<Integer> docsId = new ArrayList<Integer>();
                ArrayList<Document> docs = new ArrayList<Document>();
                HashSet<String> deduplicator = new HashSet<String>();
                searcher = index.getSearcher(user, true);
                reader = searcher.reader();
                te = MultiFields.getFields(reader).terms("attachname_keyword").iterator();
                te.seek(prefix);
                while ((term = te.next()) != null) {
                    if (!term.startsWith(prefix)) {
                        break;
                    }
                    td = te.docs(skipDocs, td);
                    int docId;
                    while ((docId = td.nextDoc()) != DocIdSetIterator.NO_MORE_DOCS) {
                        docsId.add(docId);
                    }
                }
                Collections.sort(docsId);
                for (int i = 0; i < docsId.size(); i++) {
                    Document doc;
                    doc = reader.document(docsId.get(i));
                    if (!deduplicator.contains(doc.get("url"))) {
                        docs.add(doc);
                        deduplicator.add(doc.get("url"));
                    }
                }

                OutputStreamWriter writer = new OutputStreamWriter(ps);
                JsonWriter jw = new JsonWriter(writer);
                jw.startObject();
                jw.key("hitsCount");
                jw.value(docs.size());

                jw.key("hitsArray");
                jw.startArray();
                int offset = 0;
                int length = 9999999;
                if (params.containsKey("offset")) {
                    offset = Integer.parseInt(params.get("offset"));
                }
                if (params.containsKey("length")) {
                    length = Integer.parseInt(params.get("length"));
                }
                for (int i = offset; i < docs.size() && i < offset + length; i++) {
                    Document doc;
                    String mid;
                    String hid;
                    doc = docs.get(i);

                    mid = doc.get("url");
                    hid = mid.substring(mid.indexOf('/') + 1);
                    mid = mid.substring(0, mid.indexOf('/'));

                    jw.startObject();
                    jw.key("mid");
                    jw.value(mid);
                    jw.key("hid");
                    jw.value(hid);
                    jw.key("name");
                    jw.value(doc.get("attachname"));
                    jw.key("type");
                    jw.value(doc.get("attachtype"));
                    Long size;
                    try {
                        size = Long.parseLong(doc.get("attachsize"));
                    } catch (Exception e) {
                        size = new Long(0);
                    }
                    size++;
                    size *= 100 * 1024;
                    jw.key("size");
                    jw.value(size.toString());
                    jw.key("suid");
                    jw.value(doc.get("suid"));

                    jw.endObject();
                }
                count = docs.size();
                jw.endArray();
                jw.endObject();
                jw.flush();
                writer.close();

            } finally {
                if (searcher != null) {
                    searcher.free();
                }
                ps.flush();
            }
            return count;
        }

        if (params.containsKey("op") && params.get("op") != null) {
            if (params.get("op").equals("delete")) {
                if (!(params.get("yes_i_want_this") != null && params.get("yes_i_want_this").equals("ugu"))) {
                    ctx.logger().severe("Delete password missed");
                    return 0;
                }
                String text = params.get("text");
                if (text == null) {
                    ps.print("failed");
                    ctx.logger().severe("Invalid deletion: empty text");
                    return 0;
                }
                String user = params.get("user");
                if (user == null) {
                    ps.print("failed");
                    ctx.logger().severe("Invalid deletion: empty user");
                    return 0;
                }
                printHeader(ps, request);
                try {
                    JsonDeleteMessage message = new JsonDeleteMessage(
                        new MessageContext() {
                            @Override
                            public Logger logger() {
                                return ctx.logger();
                            }
                        },
                        "/delete?text=" + text + "&prefix=" + user,
                        Message.Priority.ONLINE,
                        QueueShard.EMPTY_SHARD_ID,
                        MessageQueueId.MAGIC,
                        index.config(),
                        index.config().useJournal());
                    index.dispatch(message, false, true).get();
                    ps.print("ok");
                } catch (Exception e) {
                    if (ctx.logger().isLoggable(Level.SEVERE)) {
                        ctx.logger().log(
                            Level.SEVERE,
                            "Can't delete " + text + " : " + e.toString(),
                            e);
                    }
                    ps.print("failed");
                }
                return 0;
            }
        }

        if (params.containsKey("gcstats") || request.startsWith("/gcstats")) {
            List<GarbageCollectorMXBean> gcms = ManagementFactory.getGarbageCollectorMXBeans();
            long totalCount = 0;
            long totalTime = 0;
            Iterator<GarbageCollectorMXBean> iter = gcms.iterator();
            while (iter.hasNext()) {
                GarbageCollectorMXBean gcm = iter.next();
                totalCount += gcm.getCollectionCount();
                totalTime += gcm.getCollectionTime();
            }
            MemoryMXBean mb = ManagementFactory.getMemoryMXBean();
            MemoryUsage mem = mb.getHeapMemoryUsage();
            printHeader(ps, request);
            ps.println("Collections: " + totalCount + "\nTime: " + totalTime);
            ps.println("Memory used: " + mem.getUsed() + "\nMemory commited: " + mem.getCommitted() + "\nMemory max: "
                + mem.getMax());
            ps.println("Finalization pending: " + mb.getObjectPendingFinalizationCount());
            return 0;
        }


        if (params.containsKey("printkeys")) {
            Prefix user = null;
            String field = "mid";
            BytesRef term = null;
            BytesRef prefix = null;
            int limit = 100000;

            if (params.containsKey("prefix") && params.get("prefix") != null) {
                prefix = new BytesRef(params.get("prefix"));
//		term = new BytesRef( prefix.utf8ToString() );
            }

            if (params.containsKey("user") && params.get("user") != null) {
                user = index.config().prefixParser().apply(params.get("user"));
                final boolean checkCopyness =
                    getBoolean(params, "check-copyness", index.config().checkCopyness());
                if (checkCopyness) {
                    try {
                        index.checkShardCopied(user);
                    } catch (IOException e) {
                        throw new ServerException(
                            HttpStatus.SC_SERVICE_UNAVAILABLE, e);
                    }
                }

            }

            if (params.containsKey("field") && params.get("field") != null) {
                field = params.get("field");
            }

            if (params.containsKey("limit")) {
                limit = Integer.parseInt(params.get("limit"));
            }

            printHeader(ps, request);

            Searcher s = null;
            int lines = 0;
            try {
                s = index.getSearcher(user, true);
                Fields fields = MultiFields.getFields(s.reader());
                if (fields == null) {
                    return 0;
                }
                Terms terms = fields.terms(field);
                if (terms == null) {
                    return 0;
                }
                TermsEnum te = terms.iterator();
                if (prefix != null) {
                    te.seek(prefix);
                } else {
                    te.next();
                }
//    		if( te.term() == null ) term = te.next();
//		term = te.next();
                term = te.term();
//		while( (term = te.next()) != null )
                while (term != null) {
//		    if( field != null && !field.equals( term.field() ) ) break;
                    if (prefix != null && !term.startsWith(prefix)) {
                        break;
                    }
                    ps.println(term.utf8ToString());
                    if ((lines % 100000) == 0) {
                        ctx.checkAbort();
                    }
                    term = te.next();
                    lines++;
                    if (limit > 0 && lines == limit) {
                        break;
                    }
                }
            } finally {
                if (s != null) {
                    s.free();
                }
            }
            return lines;
        }

        boolean jsonFormat =
            params.containsKey("format")
                && params.get("format").toLowerCase().equals("json");
        Set<String> getFields = new LinkedHashSet<>();
        if (!jsonFormat) {
            getFields.add(StringHelper.intern("mid"));
            getFields.add(StringHelper.intern("hid"));
        } else if (params.containsKey("getfields")) {
            for (String field : params.get("getfields").split(",")) {
                getFields.add(StringHelper.intern(field));
            }
        } else {
            getFields.add(StringHelper.intern("mid"));
            getFields.add(StringHelper.intern("received_date"));
        }

        int offset = 0;
        int length = 9999999;
        if (params.containsKey("offset")) {
            offset = Integer.parseInt(params.get("offset"));
        }
        if (params.containsKey("length")) {
            length = Integer.parseInt(params.get("length"));
        }
        String orderBy = params.getOrDefault("orderby", "received_date");
        SortFunc sortFunc = SortFuncFactory.INSTANCE.apply(
            orderBy,
            searchRequest.fieldToIndex());
        String outerGroup = params.get("outergroup");
        OuterGroupFunction ogf;
        if (outerGroup == null) {
            ogf = null;
        } else {
            ogf = outerGroupFunctionFactory.apply(
                outerGroup,
                searchRequest.fieldToIndex());
        }
        String innerGroup = params.getOrDefault("groupby", "mid");
        GroupFunc groupFunc = GroupFuncFactory.INSTANCE.apply(
            innerGroup,
            searchRequest.fieldToIndex());

        MergeFunc mergeFunc = MergeFuncFactory.create(
            params.get("merge_func"),
            index.config().primaryKey() != null);
        searchRequest.setGetFields(getFields);
        searchRequest.setOffset(offset);
        searchRequest.setLength(length);
        searchRequest.setGroup(groupFunc);
        searchRequest.setMergeFunc(mergeFunc);
        searchRequest.setSort(sortFunc);
        searchRequest.setSortDirection(getBoolean(params, "asc", false));
        searchRequest.setOuterGroup(ogf);
        searchRequest.setScorerFactory(new NullScorerFactory(index.config()));
        final ParametrizedDocCollector col;
        if (
            sortFunc != null
                && sortFunc instanceof SimpleSortFunc
                && !params.containsKey("near")
                && index.config().autoPruning()) {
            final String sortField =
                sortFunc.loadFields().iterator().next();
            if (index.config().fieldConfig(sortField) != null) {
                col = new PruningCollector(searchRequest, null);
            } else { // else this is a generated field
                col = new SortedCollector(searchRequest, null);
            }
        } else {
            col = new SortedCollector(searchRequest, null);
        }

        String nearField = null;
        String nearValue = null;
        if (params.containsKey("near")) {
            String near = params.get("near");
            int sep = near.indexOf(':');
            if (sep != -1) {
                nearField = StringHelper.intern(near.substring(0, sep));
                nearValue = near.substring(sep + 1);
                col.setIgnoreEmpty(true);
                search(index, params, near, request, col, false);
                col.setIgnoreEmpty(false);
                if (!col.getGetFields().contains(nearField)) {
                    nearField = null;
                    nearValue = null;
                }
            }
        }

        long searchTime = System.currentTimeMillis();
        search(index, params, null, request, col, wildCardAllowed(params));
        Set<? extends YaDoc3> hits = col.hits();
        col.close();
        printHeader(ps, request);
        Iterator<? extends YaDoc3> iter = hits.iterator();
        boolean returnEmptySet = false;
        if (nearField != null && nearValue != null) {
            int needlePos = 0;
            boolean found = false;
            while (iter.hasNext()) {
                YaDoc3 yadoc = iter.next();
                String val = yadoc.getString(nearField);
                if (val != null && val.equals(nearValue)) {
                    found = true;
                    break;
                }
                needlePos++;
            }
            if (!found) {
                returnEmptySet = true;
                if (ctx.logger().isLoggable(Level.SEVERE)) {
                    ctx.logger().severe("Near not found: " + nearField + ":"
                        + nearValue);
                }
            } else {
                if (ctx.logger().isLoggable(Level.SEVERE)) {
                    ctx.logger().severe("Near found!!!: " + nearField + ":"
                        + nearValue);
                }
            }
            offset = needlePos + offset;
            iter = hits.iterator();
        }
        int pos = 0;
        if (jsonFormat) {
            boolean human = false;
            if (params.containsKey("human")) {
                human = true;
            }
            OutputStreamWriter writer = new OutputStreamWriter(ps);
            JsonWriter jw = new JsonWriter(writer);
            jw.startObject();
            jw.key("hitsCount");
            if (returnEmptySet) {
                jw.value(0);
            } else {
                jw.value(col.uniqCount());
            }

            jw.key("hitsArray");
            jw.startArray();
            if (human) {
                writer.write('\n');
            }
            if (!returnEmptySet) {
                getFields = col.getGetFields();
                while (iter.hasNext()) {
                    YaDoc3 yadoc = iter.next();
                    if (pos++ < offset) {
                        continue;
                    }
                    if (pos > offset + length) {
                        break;
                    }
                    try {
                        yadoc.writeJson(jw, getFields);
                        if (human) {
                            writer.write('\n');
                        }
                    } catch (Exception e) {
                        e.printStackTrace();
                    }
                    if (ps.checkError()) {
                        ctx.logger().warning(
                            "Connection closed by remote side. Aborting.: "
                                + request);
                        break;
                    }
                }
            }
            jw.endArray();
            jw.endObject();
            jw.flush();
            writer.close();
            ps.flush();
        } else {
            ps.println("Ok:\n" + col.uniqCount() + "\n0");
            while (iter.hasNext()) {
                YaDoc3 yadoc = iter.next();
                if (pos++ < offset) {
                    continue;
                }
                if (pos > offset + length) {
                    break;
                }
                try {
                    String mid = yadoc.getString("mid");
                    ps.println(mid + ' ' + mid + '/' + yadoc.getString("hid") + " 0 0");
                } catch (Exception e) {
                    if (ctx.logger().isLoggable(Level.WARNING)) {
                        ctx.logger().log(
                            Level.WARNING,
                            "Failed to print result #" + pos,
                            e);
                    }
                    break;
                }
            }
        }
        int docs = hits.size();
        if (ctx.logger().isLoggable(Level.INFO)) {
            ctx.logger().info(
                request + ": Totalcount: " + col.getTotalCount()
                    + '/' + col.uniqCount());
        }
        hits.clear();
        return docs;
    }

    boolean printHeaders(File targ, PrintStream ps) throws IOException {
        boolean ret = false;
        int rCode = 0;
        if (!targ.exists()) {
            rCode = HTTP_NOT_FOUND;
            ps.print("HTTP/1.0 " + HTTP_NOT_FOUND + " not found");
            ps.write(EOL);
            ret = false;
        } else {
            rCode = HTTP_OK;
            ps.print("HTTP/1.0 " + HTTP_OK + " OK");
            ps.write(EOL);
            ret = true;
        }
//        log(new Date() + " From " +s.getInetAddress().getHostAddress()+": GET " +
//            targ.getAbsolutePath()+"-->"+rCode);
        ps.print("Server: Simple java");
        ps.write(EOL);
        ps.print("Date: " + (new Date()));
        ps.write(EOL);
        if (ret) {
            if (!targ.isDirectory()) {
                ps.print("Content-length: " + targ.length());
                ps.write(EOL);
                ps.print("Last Modified: " + (new
                    Date(targ.lastModified())));
                ps.write(EOL);
                String name = targ.getName();
                int ind = name.lastIndexOf('.');
                String ct = null;
                ct = "text/html";
                ps.print("Content-type: " + ct);
                ps.write(EOL);
            } else {
                ps.print("Content-type: text/html");
                ps.write(EOL);
            }
        }
        return ret;
    }

    void printHeader(PrintStream ps, String request) throws IOException {
        printHeader(ps, request, HTTP_OK);
    }

    void printHeader(PrintStream ps, String request, int code) throws IOException {
        ps.print("HTTP/1.0 " + code + " OK");
        ps.write(EOL);
//        log((new Date()).toGMTString() + " From " +s.getInetAddress().getHostAddress()+": GET " + request + " --> Ok");
        ps.print("Server: Simple java");
        ps.write(EOL);
        ps.print("Date: " + (new Date()));
        ps.write(EOL);
        ps.print("Content-type: text/plain; charset=utf-8;");
        ps.write(EOL);
        ps.print("Connection: close");
        ps.write(EOL);
        ps.write(EOL);
    }

    void printHeaderHtml(PrintStream ps, String request) throws IOException {
        ps.print("HTTP/1.0 " + HTTP_OK + " OK");
        ps.write(EOL);
//        log((new Date()).toGMTString() + " From " +s.getInetAddress().getHostAddress()+": GET " + request + " --> Ok");
        ps.print("Server: Simple java");
        ps.write(EOL);
        ps.print("Date: " + (new Date()));
        ps.write(EOL);
        ps.print("Content-type: text/html; charset=utf-8;");
        ps.write(EOL);
        ps.print("Connection: close");
        ps.write(EOL);
        ps.write(EOL);
    }

    void send404(PrintStream ps) throws IOException {
        ps.write(EOL);
        ps.write(EOL);
        ps.println("Not Found\n\n" +
            "The requested resource was not found.\n");
    }

    void sendFile(File targ, PrintStream ps) throws IOException {
        InputStream is = null;
        ps.write(EOL);
        is = new FileInputStream(targ.getAbsolutePath());

        try {
            int n;
            while ((n = is.read(buf)) > 0) {
                ps.write(buf, 0, n);
            }
        } finally {
            is.close();
        }
    }

    private static class FakeInetConnection implements HttpInetConnection {
        private final Socket socket;

        public FakeInetConnection(final Socket socket) {
            this.socket = socket;
        }

        @Override
        public void close() {
        }

        @Override
        public boolean isOpen() {
            return true;
        }

        @Override
        public boolean isStale() {
            return false;
        }

        @Override
        public void setSocketTimeout(final int timeout) {
        }

        @Override
        public int getSocketTimeout() {
            return 0;
        }

        @Override
        public void shutdown() {
        }

        @Override
        public HttpConnectionMetrics getMetrics() {
            return null;
        }

        @Override
        public InetAddress getLocalAddress() {
            return socket.getLocalAddress();
        }

        @Override
        public int getLocalPort() {
            return socket.getLocalPort();
        }

        @Override
        public InetAddress getRemoteAddress() {
            return socket.getInetAddress();
        }

        @Override
        public int getRemotePort() {
            return socket.getPort();
        }
    }
}

