package ru.yandex.msearch;

import java.io.CharConversionException;
import java.io.IOException;
import java.nio.channels.WritableByteChannel;
import java.nio.charset.CharacterCodingException;
import java.nio.charset.MalformedInputException;
import java.nio.charset.UnsupportedCharsetException;
import java.text.ParseException;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.function.Function;
import java.util.logging.Logger;

import org.apache.http.Header;
import org.apache.http.HttpException;
import org.apache.http.HttpRequest;
import org.apache.http.HttpResponse;
import org.apache.http.HttpStatus;
import org.apache.http.entity.StringEntity;
import org.apache.http.protocol.HttpContext;
import org.apache.http.protocol.HttpRequestHandler;
import org.apache.lucene.index.CorruptIndexException;
import org.apache.lucene.store.BlockCompressedInputStreamBase;

import ru.yandex.collection.Pattern;
import ru.yandex.http.server.sync.BaseHttpServer;
import ru.yandex.http.server.sync.HttpEntityEnclosingRequestHandlerAdapter;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.ForbiddenException;
import ru.yandex.http.util.HeadersParser;
import ru.yandex.http.util.NotFoundException;
import ru.yandex.http.util.ServerException;
import ru.yandex.http.util.ServiceUnavailableException;
import ru.yandex.http.util.YandexHeaders;
import ru.yandex.http.util.YandexHttpStatus;
import ru.yandex.http.util.request.RequestHandlerMapper;
import ru.yandex.http.util.server.ImmutableBaseServerConfig;
import ru.yandex.analyzer.preprocessor.FieldPreprocessorException;
import ru.yandex.msearch.index.AddHandler;
import ru.yandex.msearch.index.DeleteByPrimaryKeyHandler;
import ru.yandex.msearch.index.DeleteByQueryHandler;
import ru.yandex.msearch.index.ModifyHandler;
import ru.yandex.msearch.index.UpdateHandler;
import ru.yandex.msearch.util.SumLimiter;
import ru.yandex.parser.uri.CgiParams;
import ru.yandex.search.prefix.Prefix;
import ru.yandex.search.prefix.PrefixParser;
import ru.yandex.util.string.StringUtils;

public class HttpIndexerServer
    extends BaseHttpServer<ImmutableBaseServerConfig>
{
    private static final Function<String, BadRequestException> BRE_GENERATOR =
        (x) -> new BadRequestException("Unable to find database " + x);

    private final Config config;
    private final DatabaseManager dbManager;
    private final IndexLagStorage lagStorage;

    public static SumLimiter limiter;

    public HttpIndexerServer(
        final Config config,
        final ImmutableBaseServerConfig serverConfig,
        final DatabaseManager dbManager)
        throws IOException
    {
        super(serverConfig);
        this.config = config;
        this.dbManager = dbManager;
        this.lagStorage = new IndexLagStorage(config, serverConfig);
        limiter =
            new SumLimiter(config.parallelDocsSize(), config.indexThreads());
        register(
            new Pattern<>("/add", false),
            new HttpEntityEnclosingRequestHandlerAdapter(
                new AddHandler(dbManager, limiter, this)),
            RequestHandlerMapper.POST);
        register(
            new Pattern<>("/delete", false),
            new DeleteByQueryHandler(dbManager, limiter, this),
            RequestHandlerMapper.GET);
        register(
            new Pattern<>("/delete", false),
            new HttpEntityEnclosingRequestHandlerAdapter(
                new DeleteByPrimaryKeyHandler(dbManager, limiter, this)),
            RequestHandlerMapper.POST);
        register(
            new Pattern<>("/modify", false),
            new HttpEntityEnclosingRequestHandlerAdapter(
                new ModifyHandler(dbManager, limiter, this)),
            RequestHandlerMapper.POST);
        register(
            new Pattern<>("/update", false),
            new HttpEntityEnclosingRequestHandlerAdapter(
                new UpdateHandler(dbManager, limiter, this)),
            RequestHandlerMapper.POST);
        ReopenHandler rh = new ReopenHandler();
        register(new Pattern<>("/reopen", false), rh);
        register(new Pattern<>("/flush", false), rh);
        register(new Pattern<>("/getQueueId", false), new QueueIdHandler());
        register(new Pattern<>("/delay", false), new DelayHandler());
        register(new Pattern<>("/lags", false), new LagsHandler(lagStorage), RequestHandlerMapper.GET);
        register(
            new Pattern<>("/index-status", false),
            new IndexStatusHandler(dbManager, config),
            RequestHandlerMapper.GET);
        register(
            new Pattern<>("/dropp-inndex", false),
            new DropIndexHandler());
        register(
            new Pattern<>("/shard-drop-and-copy", false),
            new ShardDropAndCopyHandler(dbManager));
        register(
            new Pattern<>("/schema_update", false),
            new UpdateSchemaHandler(dbManager));
        register(
            new Pattern<>("/index-copy-rate-limit", false),
            new IndexCopyRateLimitHandler());

        register(
            new Pattern<>("/active-prefixes", false),
            new ActivePrefixesHandler());

        register(
            new Pattern<>("/delete-active-prefix", false),
            new DeleteActivePrefixHandler());

        register(
            new Pattern<>("/ssd-cache-enable", false),
            new SsdCacheHandler());

        dbManager.registerIndexStaters(this);
        registerStater(lagStorage);
    }

    @Override
    public void close() throws IOException {
        limiter.stop();
        super.close();
    }

    @Override
    public Map<String, Object> status(final boolean verbose) {
        Map<String, Object> status = super.status(verbose);
        status.put("size_limiter", limiter.status());
        return status;
    }

    public void storeLag(final HttpRequest request)
        throws BadRequestException
    {
        HeadersParser headersParser = new HeadersParser(request);
        Long timestamp = headersParser.getLong(
            YandexHeaders.X_INDEX_OPERATION_TIMESTAMP,
            null);
        if (timestamp == null) {
            timestamp = headersParser.getLong(YandexHeaders.ZOO_CTIME, null);
        }
        if (timestamp != null) {
            String queue = headersParser.getString(
                    YandexHeaders.X_INDEX_OPERATION_QUEUE,
                    null);
            if (queue == null) {
                queue = headersParser.getString(YandexHeaders.ZOO_QUEUE, null);
            }
            if (queue != null) {
                lagStorage.storeLag(
                    queue,
                    headersParser.getInt(YandexHeaders.ZOO_SHARD_ID),
                    timestamp);
            }
        }
    }

    public static void handleException(final Throwable t,
        final HttpResponse response)
        throws HttpException, IOException
    {
        if (t instanceof PrimaryKeyNotFoundException) {
            throw new NotFoundException(t);
        } else if (t instanceof ParseException
            || t instanceof org.apache.http.ParseException
            || t instanceof CharConversionException
            || t instanceof MalformedInputException
            || t instanceof UnsupportedCharsetException
            || t instanceof CharacterCodingException
            || t instanceof FieldFunctionIOException
            || t instanceof FieldPreprocessorException)
        {
            throw new BadRequestException(t);
        } else if (t instanceof ConflictPrimaryKeyException) {
            throw new ServerException(HttpStatus.SC_CONFLICT, t);
        } else if (t instanceof CorruptIndexException) {
            throw new ServerException(HttpStatus.SC_INTERNAL_SERVER_ERROR, t);
        } else if (t instanceof QueueIdOutOfOrderException) {
            response.setStatusCode(HttpStatus.SC_ACCEPTED);
        } else if (t instanceof ReadOnlyIndexException) {
            throw new ForbiddenException(t);
        } else if (t instanceof IndexRequestsLimitException) {
            throw new ServerException(YandexHttpStatus.SC_REQUESTS_LIMIT_REACHED, t);
        } else if (t instanceof IOException) {
            throw (IOException) t;
        } else if (t instanceof HttpException) {
            throw (HttpException) t;
        } else {
            throw new ServiceUnavailableException(t);
        }
    }

    public static JournalableMessage loggingMessage(
        final JournalableMessage message,
        final Logger logger)
    {
        return new FilterJournalableMessage(message) {
            @Override
            public void handle(final MessageHandler handler)
                throws IOException, ParseException
            {
                logger.fine("Processing message");
                long start = System.currentTimeMillis();
                super.handle(handler);
                logger.fine("Processing finished in "
                    + (System.currentTimeMillis() - start)
                    + " ms");
            }

            @Override
            public int writeTo(final WritableByteChannel channel)
                throws IOException
            {
                if (journalable()) {
                    logger.fine("Writing journal");
                    long start = System.currentTimeMillis();
                    int written = super.writeTo(channel);
                    logger.fine(
                        written + " bytes written to journal in "
                        + (System.currentTimeMillis() - start) + " ms");
                    return written;
                } else {
                    logger.fine("Jounaling explicitly disabled for message");
                    return 0;
                }
            }
        };
    }

    private class ReopenHandler implements HttpRequestHandler {
        public void handle(HttpRequest request, HttpResponse response,
            HttpContext context) throws HttpException, IOException
        {
            try {
                CgiParams params = new CgiParams(request);
                final Logger logger = (Logger) context.getAttribute(LOGGER);
                MessageContext messageContext = new MessageContext() {
                    @Override
                    public Logger logger() {
                        return logger;
                    }
                };
                Index index = dbManager.index(params);
                index.dispatcher(true).dispatch(
                    new ReopenMessage(
                        messageContext,
                        params,
                        Message.Priority.ONLINE));
            } catch (Throwable t) {
                handleException(t, response);
            }
        }

        @Override
        public String toString() {
            return "https://wiki.yandex-team.ru/ps/Documentation/Lucene/IndexH"
                + "andlers#/?";
        }
    }

    private class IndexCopyRateLimitHandler implements HttpRequestHandler {
        public void handle(HttpRequest request, HttpResponse response,
            HttpContext context) throws HttpException, IOException
        {
            try {
                CgiParams params = new CgiParams(request);
                Index index = HttpIndexerServer.this.dbManager.index(params);
                final int rateLimitMb = params.getInt("limit");
                index.setIndexCopyRateLimit(rateLimitMb);
            } catch (Throwable t) {
                handleException(t, response);
            }
        }

        @Override
        public String toString() {
            return "Limits all index-copy operations network bandwidth usage.";
        }
    }

    private class DropIndexHandler implements HttpRequestHandler {
        private final String password;

        public DropIndexHandler() {
            this.password = dbManager.daemonConfig().dropPassword();
        }

        public void handle(
            final HttpRequest request,
            final HttpResponse response,
            final HttpContext context)
            throws HttpException, IOException
        {
            try {
                Header passwordHeader = request.getFirstHeader("ticket");

                if (passwordHeader == null
                    || !password.equalsIgnoreCase(
                    passwordHeader.getValue()))
                {
                    throw new BadRequestException(
                        "Ugu, You want it, but Authentification failed");
                }

                CgiParams params = new CgiParams(request);
                Index index = dbManager.index(params);
                if (index == null) {
                    throw new BadRequestException("Database not found");
                }

                index.dropIndex();
            } catch (Throwable t) {
                handleException(t, response);
            }
        }

        @Override
        public String toString() {
            return "Completle remove index and shutdown server.";
        }
    }

    private class QueueIdHandler implements HttpRequestHandler {
        @Override
        public void handle(
            final HttpRequest request,
            final HttpResponse response,
            final HttpContext context)
            throws HttpException, IOException
        {
            try {
                CgiParams params = new CgiParams(request);
                final boolean checkCopyness =
                    params.getBoolean("check-copyness", true);
                final String service =
                    params.getString("service", QueueShard.DEFAULT_SERVICE);
                Integer shard = params.getInt("shard", null);
                List<Index> indexes = dbManager.indexByService(service);
                Prefix prefix = null;
                if (indexes.size() > 0) {
                    prefix =
                        params.get("prefix", null, indexes.get(0).config().indexPrefixParser());
                }

                if (!(shard == null ^ prefix == null)) {
                    throw new BadRequestException(
                        "Exactly one of &shard or &prefix "
                        + "parameters must be used");
                }
                QueueShard queueShard;
                long queueId;
                if (prefix == null) {
                    queueShard = new QueueShard(service, shard);
                    queueId = dbManager.queueId(queueShard, checkCopyness);
                } else {
                    queueShard = new QueueShard(service, prefix);
                    queueId = dbManager.queueId(queueShard, prefix, checkCopyness);
                }
                Logger logger = (Logger) context.getAttribute(LOGGER);
                logger.info(
                    "For shard " + queueShard + " position is: " + queueId
                    + " and dirty flag is set to "
                    + dbManager.isDirty(queueShard));
                response.setEntity(new StringEntity(Long.toString(queueId)));
            } catch (IOException e) {
                throw new ServiceUnavailableException(e);
            } catch (Throwable t) {
                handleException(t, response);
            }
        }

        @Override
        public String toString() {
            return "Returns shard position in queue";
        }
    }

    private class DelayHandler implements HttpRequestHandler {
        public void handle(HttpRequest request, HttpResponse response,
            HttpContext context) throws HttpException, IOException
        {
            try {
                HttpRequestContext ctx = new HttpRequestContext(context);
                CgiParams params = new CgiParams(request);
                int delay = params.getInt("delay");
                int length = params.getInt("length", 1);
                boolean compressed = params.getBoolean("compressed", false);

                long contentLength = compressed ? -1L : length;
                Logger logger = (Logger) context.getAttribute(LOGGER);
                if (contentLength == -1L) {
                    limiter.waitIdleLimit();
                } else {
                    logger.fine("Message size: " + contentLength);
                    limiter.acquire(contentLength);
                }
                try {
                    if (contentLength == -1L) {
                        logger.fine("Message size: " + length);
                        contentLength = length;
                        limiter.acquireAfterIdle(contentLength);
                    }
                    MessageContext messageContext = new MessageContext() {
                        @Override
                        public Logger logger() {
                            return logger;
                        }
                    };
                    Index index = dbManager.indexOrException(params, BRE_GENERATOR);
                    index.dispatcher(true).dispatch(
                        new DelayMessage(messageContext, delay));
                } finally {
                    if (contentLength == -1L) {
                        limiter.releaseIdle();
                    } else {
                        limiter.release(contentLength);
                    }
                }
            } catch (Throwable t) {
                handleException(t, response);
            }
        }

        @Override
        public String toString() {
            return "Do nothing, just sleep for &delay= milliseconds";
        }
    }

    private class ActivePrefixesHandler implements HttpRequestHandler {
        @Override
        public void handle(
            final HttpRequest request,
            final HttpResponse response,
            final HttpContext context)
            throws HttpException, IOException
        {
            try {
                HttpRequestContext ctx = new HttpRequestContext(context);
                CgiParams params = new CgiParams(request);
                Index index = dbManager.indexOrException(
                    params,
                    BRE_GENERATOR);

                Integer shard = params.getInt("shard", null);
                long timeout = params.getLongDuration("timeout", 0L);
                Collection<String> activePrefixes;
                if (shard == null) {
                    activePrefixes = index.sortedPrefixesStrings(timeout);
                } else {
                    activePrefixes =
                        index.getShard(shard).sortedPrefixesStrings(timeout);
                }
                response.setEntity(
                    new StringEntity(
                        StringUtils.join(activePrefixes, '\n')));
            } catch (IOException e) {
                throw new ServiceUnavailableException(e);
            } catch (Throwable t) {
                handleException(t, response);
            }
        }

        @Override
        public String toString() {
            return "Returns shard position in queue";
        }
    }

    private class DeleteActivePrefixHandler implements HttpRequestHandler {
        @Override
        public void handle(
            final HttpRequest request,
            final HttpResponse response,
            final HttpContext context)
            throws HttpException, IOException
        {
            try {
                HttpRequestContext ctx = new HttpRequestContext(context);
                CgiParams params = new CgiParams(request);
                Index index = dbManager.indexOrException(
                    params,
                    BRE_GENERATOR);

                Prefix prefix =
                    params.get("prefix", index.config().indexPrefixParser());

                index.removePrefixActivity(prefix);
                response.setEntity(new StringEntity("OK"));
            } catch (IOException e) {
                throw new ServiceUnavailableException(e);
            } catch (Throwable t) {
                handleException(t, response);
            }
        }

        @Override
        public String toString() {
            return "Returns shard position in queue";
        }
    }

    private class SsdCacheHandler implements HttpRequestHandler {
        @Override
        public void handle(
            final HttpRequest request,
            final HttpResponse response,
            final HttpContext context)
            throws HttpException, IOException
        {
            try {
                HttpRequestContext ctx = new HttpRequestContext(context);
                CgiParams params = new CgiParams(request);

                boolean enable = params.getBoolean(
                    "enable",
                    !params.getBoolean("disable", false));

                BlockCompressedInputStreamBase.ssdCacheEnable(enable);
                response.setEntity(new StringEntity("OK"));
            } catch (IOException e) {
                throw new ServiceUnavailableException(e);
            } catch (Throwable t) {
                handleException(t, response);
            }
        }

        @Override
        public String toString() {
            return "Enable/Disable ssd cache";
        }
    }
}

