package ru.yandex.msearch;

import java.io.IOException;
import java.util.Map;

import org.apache.lucene.store.BlockCompressedInputStreamBase;

import ru.yandex.collection.Pattern;
import ru.yandex.collection.PatternMap;
import ru.yandex.function.GenericFunction;
import ru.yandex.http.server.sync.BaseHttpServer;
import ru.yandex.http.util.request.RequestHandlerMapper;
import ru.yandex.http.util.request.RequestInfo;
import ru.yandex.http.util.server.ImmutableBaseServerConfig;
import ru.yandex.msearch.collector.outergroup.OuterGroupFunctionFactory;
import ru.yandex.msearch.printkeys.PrintKeysHandler;
import ru.yandex.msearch.util.IOScheduler;
import ru.yandex.msearch.util.IOStater;
import ru.yandex.parser.config.ConfigException;
import ru.yandex.stater.ImmutableStaterConfig;
import ru.yandex.stater.Stater;
import ru.yandex.stater.StatsConsumer;

public class HttpSearchServer
    extends BaseHttpServer<ImmutableBaseServerConfig>
{
    public HttpSearchServer(
        final DatabaseManager index,
        final Config config,
        final OuterGroupFunctionFactory outerGroupFunctionFactory,
        final ImmutableBaseServerConfig serverConfig)
        throws ConfigException, IOException
    {
        super(serverConfig);
        PatternMap<RequestInfo, IOStater> staters =
            serverConfig.staters().staters().transform(new IOStaterFactory());

        register(
                new Pattern<>("/hnsw-search", true),
                new HnswSearchHandler(
                        index,
                        config,
                        outerGroupFunctionFactory,
                        staters),
                RequestHandlerMapper.GET,
                RequestHandlerMapper.HEAD);
        register(
            new Pattern<>("/search", true),
            new SearchHandler(
                index,
                config,
                outerGroupFunctionFactory,
                staters),
            RequestHandlerMapper.GET,
            RequestHandlerMapper.HEAD);
        register(new Pattern<>("/execute", true),
            new LymphoExecuteHandler(index, config,outerGroupFunctionFactory, staters),
            RequestHandlerMapper.POST);
        register(
            new Pattern<>("/index-status", false),
            new IndexStatusHandler(index, config),
            RequestHandlerMapper.GET);
        register(
            new Pattern<>("/numdocs", false),
            new NumDocsHandler(index),
            RequestHandlerMapper.GET);
        register(
            new Pattern<>("/printkeys", true),
            new PrintKeysHandler(index, config, staters),
            RequestHandlerMapper.GET,
            RequestHandlerMapper.HEAD);
        register(new Pattern<>("/clusterize", true),
            new ClusterizeHandler(index, config, staters),
            RequestHandlerMapper.GET,
            RequestHandlerMapper.HEAD);
        register(new Pattern<>("/scan-cluster", true),
            new ScanClusterHandler(index, config, staters),
            RequestHandlerMapper.GET,
            RequestHandlerMapper.HEAD);
        register(new Pattern<>("/drop-caches", false),
            new DropCachesHandler(),
            RequestHandlerMapper.GET);
        register(
            new Pattern<>("/dumpCacheInputs", false),
            new DumpCacheInputs(),
            RequestHandlerMapper.GET);

        registerStater(new CacheStater());
        registerStater(
            IOScheduler.instance().stater(serverConfig.metricsTimeFrame()));
    }

    @Override
    public Map<String, Object> status(final boolean verbose) {
        Map<String, Object> status = super.status(verbose);
        status.put("cache-fill", BlockCompressedInputStreamBase.cacheFill());
        status.put(
            "compressed-cache-fill",
            BlockCompressedInputStreamBase.compressedCacheFill());
        status.put("cache-size", BlockCompressedInputStreamBase.capacity());
        status.put(
            "compressed-cache-size",
            BlockCompressedInputStreamBase.compressedCacheCapacity());
        boolean shrunkCache =
            BlockCompressedInputStreamBase.compressedCacheCapacity()
            <
            BlockCompressedInputStreamBase.initialCompressedCacheCapacity();
        boolean expandedCache =
            BlockCompressedInputStreamBase.compressedCacheCapacity()
            >
            BlockCompressedInputStreamBase.initialCompressedCacheCapacity();
        status.put("compressed-cache-shrunk", shrunkCache);
        status.put("compressed-cache-expanded", expandedCache);
        status.put(
            "ssd-cache-size",
            BlockCompressedInputStreamBase.ssdCacheCapacity());
        status.put(
            "ssd-cache-fill",
             BlockCompressedInputStreamBase.ssdCacheFill());
        return status;
    }

    private class IOStaterFactory implements GenericFunction<
        ImmutableStaterConfig,
        IOStater,
        RuntimeException>
    {
        @Override
        public IOStater apply(final ImmutableStaterConfig config) {
            IOStater stater =
                new IOStater(config.metricsTimeFrame(), config.prefix());
            registerStater(stater);
            return stater;
        }
    }

    private class CacheStater implements Stater {
        @Override
        public <E extends Exception> void stats(
            final StatsConsumer<? extends E> statsConsumer)
            throws E
        {
            statsConsumer.stat(
                "cache-fill_axxx",
                BlockCompressedInputStreamBase.cacheFill());
            statsConsumer.stat(
                "compressed-cache-fill_axxx",
                BlockCompressedInputStreamBase.compressedCacheFill());
            statsConsumer.stat(
                "ssd-cache-fill_axxx",
                BlockCompressedInputStreamBase.ssdCacheFill());
            statsConsumer.stat(
                "cache-fill_avvv",
                BlockCompressedInputStreamBase.cacheFill());
            statsConsumer.stat(
                "compressed-cache-fill_avvv",
                BlockCompressedInputStreamBase.compressedCacheFill());
            statsConsumer.stat(
                "ssd-cache-fill_avvv",
                BlockCompressedInputStreamBase.ssdCacheFill());
            statsConsumer.stat(
                "cache-size_axxx",
                BlockCompressedInputStreamBase.capacity());
            statsConsumer.stat(
                "cache-size_ammm",
                BlockCompressedInputStreamBase.capacity());
            statsConsumer.stat(
                "cache-size_annn",
                BlockCompressedInputStreamBase.capacity());
            statsConsumer.stat(
                "compressed-cache-size_axxx",
                BlockCompressedInputStreamBase.compressedCacheCapacity());
            statsConsumer.stat(
                "compressed-cache-size_ammm",
                BlockCompressedInputStreamBase.compressedCacheCapacity());
            statsConsumer.stat(
                "compressed-cache-size_annn",
                BlockCompressedInputStreamBase.compressedCacheCapacity());
            statsConsumer.stat(
                "ssd-cache-size_axxx",
                BlockCompressedInputStreamBase.ssdCacheCapacity());
            statsConsumer.stat(
                "ssd-cache-size_ammm",
                BlockCompressedInputStreamBase.ssdCacheCapacity());
            statsConsumer.stat(
                "ssd-cache-size_annn",
                BlockCompressedInputStreamBase.ssdCacheCapacity());
            boolean shrunkCache =
                BlockCompressedInputStreamBase.compressedCacheCapacity()
                <
                BlockCompressedInputStreamBase.initialCompressedCacheCapacity();
            boolean expandedCache =
                BlockCompressedInputStreamBase.compressedCacheCapacity()
                >
                BlockCompressedInputStreamBase.initialCompressedCacheCapacity();
            statsConsumer.stat("compressed-cache-shrunk_ammm", shrunkCache);
            statsConsumer.stat("compressed-cache-expanded_ammm", expandedCache);
            BlockCompressedInputStreamBase.stats(statsConsumer);
        }
    }
}

