package ru.yandex.msearch;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.PrintStream;
import java.io.StringReader;
import java.nio.charset.StandardCharsets;
import java.text.ParseException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.PrimitiveIterator;
import java.util.Set;
import java.util.TreeSet;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.Callable;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Future;
import java.util.concurrent.FutureTask;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.RejectedExecutionException;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.atomic.AtomicReference;
import java.util.logging.Level;
import java.util.logging.Logger;

import org.apache.lucene.index.CorruptIndexException;
import org.apache.lucene.index.IndexReader;
import org.apache.lucene.index.SharedThreadPoolMergeScheduler;
import org.apache.lucene.index.Term;
import org.apache.lucene.index.codecs.Codec;
import org.apache.lucene.index.codecs.CodecProvider;
import org.apache.lucene.index.codecs.CoreCodecProvider;
import org.apache.lucene.index.codecs.fast_commit.FastCommitCodec;
import org.apache.lucene.index.codecs.yandex.YandexCodec;
import org.apache.lucene.index.codecs.yandex2.Yandex2Codec;
import org.apache.lucene.search.BooleanQuery;
import org.apache.lucene.search.Query;
import org.apache.lucene.search.TermQuery;
import org.apache.lucene.store.AesflateCompressor;
import org.apache.lucene.store.BlockCompressedInputStreamBase;
import org.apache.lucene.store.BrotliCompressor;
import org.apache.lucene.store.Compressor;
import org.apache.lucene.store.DeflateCompressor;
import org.apache.lucene.store.Directory;
import org.apache.lucene.store.EOFException;
import org.apache.lucene.store.FSDirectory;
import org.apache.lucene.store.LZ4Compressor;
import org.apache.lucene.store.LZMACompressor;
import org.apache.lucene.store.ZstdCompressor;

import ru.yandex.analyzer.ThreadLocalAnalyzerProvider;
import ru.yandex.collection.BlockingBlockingQueue;
import ru.yandex.collection.IntInterval;
import ru.yandex.collection.IntSet;
import ru.yandex.concurrent.FailedFuture;
import ru.yandex.concurrent.LifoWaitBlockingQueue;
import ru.yandex.concurrent.NamedThreadFactory;
import ru.yandex.logger.PrefixedLogger;
import ru.yandex.msearch.collector.FlushableCollector;
import ru.yandex.msearch.config.DatabaseConfig;
import ru.yandex.msearch.fieldscache.FieldsCache;
import ru.yandex.msearch.jobs.JobsManager;
import ru.yandex.msearch.parallel.ParaWork;
import ru.yandex.msearch.parallel.ParallelExec;
import ru.yandex.msearch.util.Compress;
import ru.yandex.msearch.util.IOScheduler;
import ru.yandex.parser.config.ConfigException;
import ru.yandex.parser.config.IniConfig;
import ru.yandex.search.prefix.Prefix;
import ru.yandex.stater.GolovanChart;
import ru.yandex.stater.GolovanChartGroup;
import ru.yandex.stater.GolovanPanel;
import ru.yandex.stater.ImmutableGolovanPanelConfig;
import ru.yandex.stater.Stater;
import ru.yandex.stater.StatsConsumer;
import ru.yandex.util.string.StringUtils;

public class Index implements MessageHandler, CommonTaskor, MessageContext, AnalyzerProvider {
    private static final int FIELDS_CACHE_RELOAD_INTERVAL = 60 * 1000;
    private static final int INDEX_SIZE_CALC_INTERVAL = 60 * 1000;
    private static final int INDEX_STATER_TRAVERSE_TIME = 60 * 1000;

    //in seconds
    private static final float THROTTLE_REGULATOR_PERIOD = 1.0f;
    private static final float THROTTLE_REGULATOR_K = 0.5f;
    private static final int RECOMPUTE_MEM_INTERVAL = 1000;

    static class WarmerWork {
        public IndexReader reader, oldReader;
        public boolean closeOld;
        public FieldsCache cache;

        public WarmerWork(
            final FieldsCache c,
            final IndexReader old,
            final IndexReader newReader,
            final boolean close)
        {
            cache = c;
            oldReader = old;
            reader = newReader;
            closeOld = close;
        }
    }

    private class ShardFlushTask implements Callable<Void> {
        private final int shard;

        ShardFlushTask(final int shard) {
            this.shard = shard;
        }

        public Void call() throws Exception {
            try {
                shards[shard].doFlush(false);
            } catch (Exception e) {
                if (logger.isLoggable(Level.SEVERE)) {
                    logger.log(
                        Level.SEVERE,
                        "Error ocurred while trying to flush shard<" + shard
                            + ">",
                        e);
                }
                throw e;
            } finally {
                recomputeRamUsage();
                if (logger.isLoggable(Level.INFO)) {
                    logger.info("Ram usage after shard<" + shard
                        + "> flushing: " + docsInMemTotal + " ("
                        + docsInMemActive + ")");
                }
                synchronized (checkShardsLock) {
                    shardsInFlush[shard] = false;
                }
            }
            return null;
        }
    }

    private class ReopenShardReaderTask implements Callable<Void> {
        private final int shard;
        private final long readerGen;

        ReopenShardReaderTask(final int shard, final long readerGen) {
            this.shard = shard;
            this.readerGen = readerGen;
        }

        public Void call() throws Exception {
            shards[shard].reopenShardReader(readerGen);
            return null;
        }
    }

    private class ReopenShardMultiSearcherTask implements Callable<Void> {
        private final Shard shard;

        ReopenShardMultiSearcherTask(final Shard shard) {
            this.shard = shard;
        }

        public Void call() throws Exception {
            shard.reopenMultiSearcherSync();
            return null;
        }
    }

    private class ThrottleRegulator extends Thread {
        public void run() {
            while(true) {
                try {
                    computeThrottleDelay();
                    Thread.sleep((int)(THROTTLE_REGULATOR_PERIOD * 1000));
                } catch (Throwable t) {
                    t.printStackTrace();
                }
            }
        }
    }

    private final LinkedList<WarmerWork> warmerQueue = new LinkedList<>();
    private final Object checkShardsLock = new Object();
    private final ThreadLocalAnalyzerProvider analyzerProvider;

    private final DatabaseConfig config;
    private final Config daemonConfig;
    private final int shardsCount;
    private final long maxDocsInMem;
    private final long docsInMemFlushThreshold;
    private final AtomicReference<CodecProvider> cp;
    private final AtomicReference<CodecProvider> memCp;
    private final JobsManager jobsManager;
    private final ExecutorService indexExecutor;
    private final Shard[] shards;
    private final ThreadPoolExecutor commonTaskExecutor;
    private final ThreadPoolExecutor partsParallelExecutor;
    private final SharedThreadPoolMergeScheduler concurrentScheduler;
    private final boolean[] shardsInFlush;
    private final boolean hasPrimaryKey;
    private final Map<QueueShard, AtomicLong> queueIdsDirtyMap =
        new ConcurrentHashMap<>();
    private final PrefixedLogger logger;
    private final Logger indexLogger;

    private long    docsInMemTotal;
    private long    docsInMemActive;
    private volatile int throttleDelay = 0;
    private float throttleRegulatorIntegral = 0;
    private float throttleRegulatorDifferential = 0;
    private long throttleRegulatorPrevDiff = 0;
    private ThrottleRegulator regulator;
    private long lastMemRecompute = 0;
    private volatile boolean exit = false;
    private final FieldsCache fieldsCache;
    private final FieldsCacheReloader fieldsCacheReloader;
    private final Directory tmpDirectory;
    private final IndexDispatcher onlineDispatcher;
    private final IndexDispatcher offlineDispatcher;

//    public Index(
//        final File path,
//        final Config config,
//        final PrefixedLogger logger,
//        final Logger indexLogger,
//        final ShardInitListener initListener)
//        throws Exception
//    {
//        this(path, config, config, logger, indexLogger, initListener);
//    }

    public Index(
        final File path,
        final Config daemonConfig,
        final DatabaseConfig databaseConfig,
        final PrefixedLogger logger,
        final Logger indexLogger,
        final ShardInitListener initListener)
        throws Exception
    {
        this.config = databaseConfig;
        this.daemonConfig = daemonConfig;
        this.logger = logger;
        this.indexLogger = indexLogger;
        this.analyzerProvider = new ThreadLocalAnalyzerProvider(config);

        shardsCount = config.shards();
        maxDocsInMem = config.maxMemDocs() * 1024L;
        docsInMemFlushThreshold = maxDocsInMem - (maxDocsInMem >> 2);

        File indexPath = config.indexPath();
        File tmpDir = new File(indexPath, "tmp" );
        tmpDir.mkdirs();
        tmpDirectory = FSDirectory.open( tmpDir );
        String[] oldFiles = tmpDirectory.listAll();
        for( int i = 0; i < oldFiles.length; i++ )
        {
            tmpDirectory.deleteFile( oldFiles[i] );
        }
        // TODO: move to IndexWriterConfig
        org.apache.lucene.index.DocumentsWriter.setMaxRecycledBlocksSize(
            maxDocsInMem / 3);

        // TODO: do not use singleton for CodecProvider and revert
        // registerIfAbsent() function
        logger.info("Index divisors: " + config.fieldsConfig().indexDivisors());

        cp = new AtomicReference<>(createCodecProvider(false));
        memCp = new AtomicReference<>(createCodecProvider(true));

        BooleanQuery.setMaxClauseCount( 1000000 );

        jobsManager = new JobsManager(new File(path, "jobs"), this, daemonConfig, logger);

        shardsInFlush = new boolean[shardsCount];
        for( int i = 0; i < shardsCount; i++ ) shardsInFlush[i] = false;

        commonTaskExecutor = new ThreadPoolExecutor(
                config.commonTaskExecutorThreads(),
                config.commonTaskExecutorThreads(),
                1,
                TimeUnit.HOURS,
                new LinkedBlockingQueue<Runnable>(),
                new NamedThreadFactory("Index-Tasks-"),
                new ThreadPoolExecutor.CallerRunsPolicy()
        );

        int partsThreads = config.partsExecutorThreadCount();
        if (partsThreads <= 0) {
            partsParallelExecutor = null;
        } else {
            partsParallelExecutor = new ThreadPoolExecutor(
                partsThreads,
                partsThreads,
                1,
                TimeUnit.HOURS,
                new LifoWaitBlockingQueue<>(Math.max(partsThreads, 2)),
                new NamedThreadFactory("MemoryIndex-") {
                    @Override
                    public Thread newThread(final Runnable r) {
                        return super.newThread(() -> {
                            IOScheduler.setThreadReadPrio(
                                IOScheduler.IOPRIO_INDEXING);
                            IOScheduler.setThreadWritePrio(
                                IOScheduler.IOPRIO_INDEXING);
                            r.run();
                        });
                    }
                },
                new ThreadPoolExecutor.CallerRunsPolicy()
            );
        }

        concurrentScheduler = new SharedThreadPoolMergeScheduler(
            config.maxMergeThreads());
        concurrentScheduler.setMergeThreadPriority(
            Thread.MIN_PRIORITY,
            Compress.IOPRIO_CLASS_IDLE,
            Compress.IOPRIO_HIGH);

        exit = false;

        docsInMemTotal = 0;
        docsInMemActive = 0;

        shards = new Shard[shardsCount];
        fieldsCache = new FieldsCache(config, logger);

        List<Future<Object>> futures = new ArrayList<>();
        for (int i = 0; i < shardsCount; ++i) {
            final int j = i;
            final Index thisIndex = this;
            futures.add(commonTaskExecutor.submit(new Callable<Object>() {
                @Override
                public Object call() throws IOException {
                    logger.info("Running work: shardinit");
                    IOScheduler.setThreadReadPrio(IOScheduler.MAX_PRIO);
                    IOScheduler.setThreadWritePrio(IOScheduler.MAX_PRIO);
                    Shard shard = null;
                    try {
                        AnalyzerProvider journalAnalyzerProvider =
                            config.useJournal() ? analyzerProvider : null;
                        try {
                            shard =
                                new Shard(
                                    thisIndex,
                                    path,
                                    j,
                                    config,
                                    journalAnalyzerProvider,
                                    concurrentScheduler,
                                    thisIndex,
                                    cp::get,
                                    memCp::get,
                                    fieldsCache,
                                    logger,
                                    indexLogger,
                                    partsParallelExecutor);
                            shards[j] = shard;
                            shard.initialize();
                        } catch (
                            FileNotFoundException
                            |CorruptIndexException
                            |EOFException e)
                        {
                            logger.log(
                                Level.SEVERE,
                                "Initializing shard no: " + j
                                + " failed, trying "
                                + " to move and recopy: ",
                                e);
                            jobsManager.setPending(j);
                            if (shard != null) {
                                shard.move();
                            } else {
                                shard.move(j, path);
                            }
                            shard =
                                new Shard(
                                    thisIndex,
                                    path,
                                    j,
                                    config,
                                    journalAnalyzerProvider,
                                    concurrentScheduler,
                                    thisIndex,
                                    cp::get,
                                    memCp::get,
                                    fieldsCache,
                                    logger,
                                    indexLogger,
                                    partsParallelExecutor);
                            shards[j] = shard;
                            shard.initialize();
                        }
                    } catch(IOException e) {
                        logger.log(
                            Level.SEVERE,
                            "Initializing shard no: " + j + " failed: ",
                            e);
                        throw e;
                    }
                    IOScheduler.setThreadReadPrio(IOScheduler.IOPRIO_FLUSH);
                    IOScheduler.setThreadWritePrio(IOScheduler.IOPRIO_FLUSH);
                    return null;
                }
            }));
        }
        // This is done for several purposes:
        // 1) to propagate exceptions from submitted tasks (in a form of ExecutionException)
        // 2) to block until all tasks are finished
        // We could use invokeAll() on our commonTaskExecutor, but it would swallow the exceptions
        for (Future<Object> f: futures) {
            if (initListener != null) {
                initListener.shardInit();
            }
            f.get();
        }

        BlockingQueue<Runnable> indexerQueue;
        if (config.limitIndexRequests() > 0) {
            int limit = Math.max(2,
                config.limitIndexRequests() - config.indexThreads());
            indexerQueue = new LifoWaitBlockingQueue<Runnable>(limit);
            logger.warning("LIMIT: " + limit);
        } else {
            indexerQueue =
                new BlockingBlockingQueue<>(
                    new LifoWaitBlockingQueue<>(
                        Math.max(
                            2,
                            daemonConfig.indexer().workers())));
        }

        onlineDispatcher = new IndexDispatcher(this, false, logger);
        offlineDispatcher = new IndexDispatcher(this, true, logger);

        indexExecutor = new ThreadPoolExecutor(
            config.indexThreads(),
            config.indexThreads(),
            1,
            TimeUnit.HOURS,
            indexerQueue,
            new NamedThreadFactory("Index-") {
                @Override
                public Thread newThread(final Runnable r) {
                    return super.newThread(() -> {
                        IOScheduler.setThreadReadPrio(
                            IOScheduler.IOPRIO_INDEXING);
                        IOScheduler.setThreadWritePrio(
                            IOScheduler.IOPRIO_INDEXING);
                        r.run();
                    });
                }
            });

        for (int i = 0; i < config.warmerThreads(); i++) {
            IndexWarmer warmer = new IndexWarmer(this);
            Thread t = new Thread(warmer, "IndexWarmer#" + i);
            t.setDaemon(true);
            t.start();
        }

        hasPrimaryKey = config.primaryKey() != null;

        jobsManager.runJobs();

        regulator = new ThrottleRegulator();
        regulator.setDaemon(true);
        regulator.start();
        fieldsCacheReloader = new FieldsCacheReloader();
        fieldsCacheReloader.start();
    }

    public Directory tmpDirectory() {
        return tmpDirectory;
    }

    @Override
    public Logger logger() {
        return logger;
    }

    public FieldsCache fieldsCache() {
        return fieldsCache;
    }

    public void ssdCacheEnable(final boolean enable) {
        BlockCompressedInputStreamBase.ssdCacheEnable(enable);
    }

    public IndexDispatcher dispatcher(final boolean online) {
        if (online) {
            return onlineDispatcher;
        } else {
            return offlineDispatcher;
        }
    }


    public CodecProvider createCodecProvider(final boolean memory) {
        final int compressLevel;
        if (memory) {
            compressLevel = config.memoryCodecCompressorLevel();
        } else {
            compressLevel = config.yandexCodecCompressorLevel();
        }
        System.err.println("CREATE CODEC PROVIDER: inMemoryFieldsIndex: " + config.inMemoryFieldsIndex());
        CodecProvider cp = new CoreCodecProvider();
        final Compressor[] compressors = new Compressor[] {
            new AesflateCompressor(compressLevel),
            new BrotliCompressor(compressLevel),
            new DeflateCompressor(compressLevel),
            new LZ4Compressor(compressLevel),
            new LZMACompressor(compressLevel),
            new ZstdCompressor(compressLevel)
        };
        String defaultCodecName = null;
        ArrayList<Codec> codecs = new ArrayList<>();
        for (Compressor compressor: compressors) {
            codecs.add(
                new YandexCodec(
                    compressor,
                    config.yandexFieldsWriterBufferSize(),
                    config.yandexTermsWriterBlockSize(),
                    config.yandexPostingsWriterBlockSize(),
                    config.yandexCodecGroupFields(),
                    config.yandexCodecBloomSet(),
                    config.fieldsConfig().indexDivisors())
                    .setFieldIndexReadBufferSize(
                        config.fieldIndexReadBufferSize())
                    .useInMemoryFieldsIndex(
                        config.inMemoryFieldsIndex()));
            codecs.add(
                new Yandex2Codec(
                    compressor,
                    config.yandexFieldsWriterBufferSize(),
                    config.yandexTermsWriterBlockSize(),
                    config.yandexPostingsWriterBlockSize(),
                    config.yandexCodecGroupFields(),
                    config.yandexCodecBloomSet(),
                    config.indexedFields(),
                    config.fieldsConfig().indexDivisors())
                    .setFieldIndexReadBufferSize(
                        config.fieldIndexReadBufferSize())
                    .useInMemoryFieldsIndex(
                        config.inMemoryFieldsIndex()));
        }
        codecs.add(new FastCommitCodec(config.yandexCodecBloomSet()));
        for (Codec codec: codecs) {
            if (codec.name.equals(config.defaultFieldCodec())
                || CodecProvider.pureName(codec)
                    .equals(config.defaultFieldCodec()))
            {
                defaultCodecName = codec.name;
            }
            cp.registerIfAbsent(codec);
        }
        cp.setDefaultFieldCodec(defaultCodecName);
        logger.severe("Index.defaultCodecName: " + defaultCodecName);
        return cp;
    }

    public Set<String> sortedPrefixesStrings(final long timeout) {
        TreeSet<String> prefixes = new TreeSet<>();
        for (Shard shard: shards) {
            prefixes.addAll(shard.sortedPrefixesStrings(timeout));
        }
        return prefixes;
    }

    public void removePrefixActivity(final Prefix prefix) {
        int luceneShard = (int) (prefix.hash() % shardsCount);
        shards[luceneShard].removePrefixActivity(prefix);
    }

    public void setIndexCopyRateLimit(final int limitMb) {
        jobsManager.rateLimiter().rateLimitMb(limitMb);
    }

    public boolean dumpAllowed() {
        return jobsManager.hasCopyJobs();
    }

    public Stater stater() {
        final IndexStater stater = new IndexStater(INDEX_SIZE_CALC_INTERVAL);
        stater.start();
        return stater;
    }

    public CodecProvider codecProvider() {
        return cp.get();
    }

    public DatabaseConfig config() {
        return config;
    }

    public Config daemonConfig() {
        return daemonConfig;
    }

    public JobsManager getJobsManager()
    {
	return jobsManager;
    }

    public WarmerWork getWarmerWork() {
        synchronized (warmerQueue) {
            WarmerWork w = null;
            while (true) {
                w = warmerQueue.poll();
                if (w == null) {
                    try {
                        warmerQueue.wait(1000);
                        continue;
                    } catch (Exception e) {
                    }
                }
                return w;
            }
        }
    }

    public int shardsCount()
    {
	return shardsCount;
    }

    public Shard getShard(int shardNo) {
        return shards[shardNo];
    }

    public void close() {
        exit = true;
        indexExecutor.shutdown();

        logger.info("Stopping job manager");
        try {
            jobsManager.stopJobs();
        } catch (IOException e) {
            e.printStackTrace();
        }

        ParallelExec para = new ParallelExec(this, 16, config.indexPrefixParser());
        for (int i = 0; i < shardsCount; i++) {
            Shard shard = shards[i];
            try {
                logger.info("Closing shard: " + i);
                para.addWork(new ParaWork("closeShard"), shard);
            } catch (IOException e) {
                e.printStackTrace();
            }
        }
        para.stop();
        commonTaskExecutor.shutdown();
    }

    public void reopenIndexes(
        IntSet shards,
        final Boolean ro,
        final boolean doWait)
    {
        if (shards == null) {
            shards = new IntInterval(0, shardsCount - 1);
        }
        PrimitiveIterator.OfInt iter = shards.iterator();
        while (iter.hasNext()) {
            int shardNum = iter.next();
            if (shardNum < 0 || shardNum >= shardsCount) {
                continue;
            }
            Shard shard = this.shards[shardNum];
            try {
                if (logger.isLoggable(Level.INFO)) {
                    logger.info("Reopening shard (" + doWait + "): "
                        + shardNum + " flushing docs: " + shard.getDocsInMem());
                }
                if (!doWait) {
                    flushShardAsync(shardNum, ro);
                } else {
                    flushShardSync(shardNum, ro);
                }
            } catch (Exception e) {
                e.printStackTrace();
            }
        }
    }

    private void checkShardBounds(final int shard, final String name)
        throws IOException
    {
        if (shard < 0 || shard >= shardsCount) {
            throw new IOException(
                "Invalid " + name + " parameter: " + shard +
                ". Should be in range 0-" + (shardsCount - 1));
        }
    }

    public void optimize(final int optimize, final PrintStream ps,
        final int threads, final int startShard, final int endShard)
            throws IOException
    {
        checkShardBounds(startShard, "startShard");
        checkShardBounds(endShard, "endShard");
        if (endShard - startShard < 0) {
            throw new IOException(
                "Can't run optimize: startShard is less than endShard");
        }
        ThreadPoolExecutor executor = null;
        boolean failed = false;
        try {
            executor = new ThreadPoolExecutor(
                threads,
                threads,
                0,
                TimeUnit.SECONDS,
                new ArrayBlockingQueue<Runnable>(endShard - startShard + 1),
                new NamedThreadFactory("Optimize-"));
            List<Future<Void>> futures =
                new ArrayList<Future<Void>>(endShard-startShard+1);

            for(int i = startShard; i <= endShard; i++) {
                final Shard shard = shards[i];
                final int num = i;
                Callable<Void> c = new Callable<Void>() {
                    @Override
                    public Void call() throws IOException {
                        logger.info("Optimizing shard: " + num + " to <"
                            + optimize + "> segments.");
                        ps.println("Optimizing shard: " + num + " to <"
                            + optimize + "> segments.");
                        ps.flush();
                        shard.optimize(optimize);
                        logger.info("Optimizing shard: " + num
                            + " has been finished");
                        ps.println("Optimizing shard: " + num
                            + " has been finished");
                        ps.flush();
                        return null;
                    }
                };
                futures.add(executor.submit(c));
            }
            for (Future<Void> f : futures) {
                try {
                    f.get();
                } catch (Exception e) {
                    failed = true;
                    logger.log(
                        Level.SEVERE,
                        "Optimize request failed: ",
                        e);
                }
            }
        } finally {
            if (executor != null) {
                executor.shutdown();
            }
            if (failed) {
                throw new IOException("One or more shards are failed to optimize");
            }
        }
    }

    public void expunge(final PrintStream ps, int threads,
        final Integer version, final boolean all, final boolean convert,
        final double minPct,
        final int startShard, final int endShard)
            throws IOException
    {
        checkShardBounds(startShard, "startShard");
        checkShardBounds(endShard, "endShard");
        if (endShard - startShard < 0) {
            throw new IOException(
                "Can't run expunge: startShard is less than endShard");
        }
        ThreadPoolExecutor executor = null;
        boolean failed = false;
        List<Future<Void>> futures =
            new ArrayList<Future<Void>>(endShard-startShard+1);
        try {
            executor = new ThreadPoolExecutor(
                threads,
                threads,
                0,
                TimeUnit.SECONDS,
                new ArrayBlockingQueue<Runnable>(endShard - startShard + 1),
                new NamedThreadFactory("Expunge-"));

            for(int i = startShard; i <= endShard; i++) {
                final Shard shard = shards[i];
                final int num = i;
                if (version != null && shard.getVersion() >= version) {
                    logger.warning("Skipping shard: " + i);
                    ps.println("Skipping shard: " + i);
                    ps.flush();
                    continue;
                }
                Callable<Void> c = new Callable<Void>() {
                    @Override
                    public Void call() throws IOException {
                        logger.info("Expunging shard: " + num);
                        ps.println("Expunging shard: " + num);
                        ps.flush();

                        shard.expunge(all, version, minPct, convert);

                        logger.info("Expunge of shard: " + num
                            + " has been finished");
                        ps.println("Expunge shard: " + num
                            + " has been finished");
                        ps.flush();
                        return null;
                    }
                };
                futures.add(executor.submit(c));
            }
            for (Future<Void> f : futures) {
                try {
                    f.get();
                } catch (Exception e) {
                    logger.log(
                        Level.SEVERE,
                        "Expunge request failed",
                        e);
                    failed = true;
                }
            }
        } finally {
            if (executor != null) {
                executor.shutdown();
            }
            if (failed) {
                throw new IOException("One or more shards are failed to expunge");
            }
        }
    }

    public DumpSearcher getDumpSearcher(
        final int shard,
        final int outerShardStart,
        final int outerShardEnd)
        throws IOException
    {
        if (jobsManager.shardInCopy(shard, outerShardStart, outerShardEnd)) {
            throw new IOException("Shard <" + shard
                + "> is copying now and cannot be dumped");
        }
        if (!jobsManager.shardsCopied(shard, outerShardStart, outerShardEnd)) {
            throw new IOException("Shard <" + shard
                + "> was not copied and can not be dumped");
        }
        return shards[shard].getDumpSearcher();
    }

    public DumpSearcher getDumpSearcher(
        final Set<Integer> shards,
        final int outerShardStart,
        final int outerShardEnd)
        throws IOException
    {
        for(Integer shard : shards) {
            if (jobsManager.shardInCopy(
                shard,
                outerShardStart,
                outerShardEnd))
            {
                throw new IOException("Shard <" + shard
                    + "> is copying now and cannot be dumped");
            }
            if (!jobsManager.shardsCopied(
                shard,
                outerShardStart,
                outerShardEnd))
            {
                throw new IOException("Shard <" + shard
                    + "> was not copied and can not be dumped");
            }
        }
        DumpSearcher[] searchers = new DumpSearcher[shards.size()];
        int i = 0;
        final Map<QueueShard, Long> queueIds = new HashMap<>();
        for(Integer shard : shards) {
            DumpSearcher shardSearcher = this.shards[shard].getDumpSearcher();
            for (final Map.Entry<QueueShard, Long> entry :
                shardSearcher.queueIds().entrySet())
            {
                final QueueShard queueShard = entry.getKey();
                final Long queueId = entry.getValue();
                if (queueIds.containsKey(queueShard)) {
                    if (queueIds.get(queueShard) < queueId) {
                        queueIds.put(queueShard, queueId);
                    }
                } else {
                    queueIds.put(queueShard, queueId);
                }
            }
            searchers[i++] = shardSearcher;
        }
        DumpSearcher searcher = new DumpSearcher(
            new Searcher.MultiSearcher(searchers),
            queueIds);
        searcher.incRef();
        return searcher;
    }

    public void checkShardCopied(Prefix user) throws IOException {
        if (user == null) {
            if (config.checkCopyness()
                && !jobsManager.indexCopied())
            {
                throw new IOException("Index copy in progress. "
                    + " Dump forbidden.");
            }
        } else {
            int luceneShard = (int) (user.hash() % shardsCount);
            int outerShard =
                Math.abs((int) (user.hash() % QueueShard.SHARDS_MAGIC));
            if (config.checkCopyness()
                    && !jobsManager.shardCopied(luceneShard, outerShard))
            {
                throw new IOException("Shard <" + outerShard + "@" + luceneShard
                    + "> was not copied and can not be dumped");
            }
        }
    }

    public void checkShardCopied(QueueShard shard) throws IOException {
        if (!config.checkCopyness()) {
            return;
        }
        if (shard == null || !config.kosherShards()) {
            if (!jobsManager.indexCopied()) {
                throw new IOException("Index copy in progress. "
                    + " Dump forbidden.");
            }
        } else {
            int luceneShard = shard.shardId() % shardsCount;
            int outerShard = shard.shardId();
            if (!jobsManager.shardCopied(luceneShard, outerShard)) {
                throw new IOException("Shard <" + outerShard + "@" + luceneShard
                    + "> was not copied and can not be dumped");
            }
        }
    }

    public Searcher getDiskSearcher(final Prefix user) throws IOException {
        if (user == null) {
            return getDiskSearcher();
        }
        int luceneShard = (int) (user.hash() % shardsCount);
        return getDiskSearcher(luceneShard);
    }

    public Searcher getDiskSearcher(final int shard) throws IOException {
        return shards[shard].getShardSearcher();
    }

    public Searcher getDiskSearcher() throws IOException {
        final Searcher[] searchers = new Searcher[shardsCount];
        for (int i = 0; i < shardsCount; ++i) {
            searchers[i] = shards[i].getShardSearcher();
        }
        final Searcher searcher = new Searcher.MultiSearcher(searchers);
        searcher.incRef();
        return searcher;
    }

    public Searcher getSearcher(Prefix user, final boolean sync)
        throws IOException
    {
        if (user == null) {
            return getSearcher(sync);
        }
//        checkShardCopied(user);
        int luceneShard = (int) (user.hash() % shardsCount);
        return getSearcher(luceneShard, sync);
    }

    public Searcher getSearcher(final int shard, final boolean sync)
        throws IOException
    {
        if (sync) {
            return shards[shard].getMultiSearcher();
        } else {
            return shards[shard].getMultiSearcherDirty();
        }
    }

    public Searcher getSearcher(final boolean sync) throws IOException {
	Searcher[] searchers = new Searcher[shardsCount];
	for(int i = 0; i < shardsCount; ++i)
	{
	    if (sync) {
                searchers[i] = shards[i].getMultiSearcher();
            } else {
                searchers[i] = shards[i].getMultiSearcherDirty();
            }
	}
	Searcher searcher = new Searcher.MultiSearcher(searchers);
	searcher.incRef();
	return searcher;
    }

    public Searcher getSearcher(final Set<Integer> shards) throws IOException {
	Searcher[] searchers = new Searcher[shards.size()];
	int i = 0;
	for(Integer shard : shards)
	{
	    searchers[i++] = this.shards[shard].getMultiSearcher();
	}
	Searcher searcher = new Searcher.MultiSearcher(searchers);
	searcher.incRef();
	return searcher;
    }

    public void updatePrefixActivity(final Prefix prefix) {
        if (prefix != null && config.updatePrefixActivity()) {
            int shard = (int) (prefix.hash() % shardsCount);
            shards[shard].updatePrefixActivity(prefix);
        }
    }

    private Future<Void> submitTask(final Callable<Void> task,
        final int priority,
        final boolean force)
        throws IndexRequestsLimitException
    {
        FutureTask<Void> f = new PriorityFIFOTask(task, priority);
        try {
            indexExecutor.execute(f);
        } catch (RejectedExecutionException e) {
            if (!force) {
                throw new IndexRequestsLimitException();
            }
            //else execute in current thread
            try {
                task.call();
            } catch (Exception taskException) {
                Exception ee = new ExecutionException(
                    "In place task execution failed",
                    taskException);
                ee.addSuppressed(e);
                return new FailedFuture(ee);
            }
        }
        return f;
    }

    private static class PriorityFIFOTask extends FutureTask<Void>
        implements Comparable<PriorityFIFOTask>
    {
        private final int priority;
        private final static AtomicLong seq = new AtomicLong();
        private final long seqNum;
        private final Callable<Void> callable;
        private Throwable error;

        public PriorityFIFOTask(
            final Callable<Void> callable,
            final int priority)
        {
            super(callable);
            this.callable = callable;
            seqNum = seq.getAndIncrement();
            this.priority = priority;
        }

        @Override
        public int compareTo(final PriorityFIFOTask other) {
            if (priority < other.priority) {
                return 1;
            } else if (priority > other.priority) {
                return -1;
            } else {
                return Long.compare(seqNum, other.seqNum);
            }
        }
    }

    public void reloadFields(final String name, final boolean force, final String schema) throws IOException {
        reloadFields(name, force, schema, -1);
    }

//    2021-02-05 13:29:54.377	WARNING	IndexerServer-1	shard[61]	PRE Flushing shard<61>: ro=false, docs:1144, time: 1
//        2021-02-05 13:29:54.377	WARNING	IndexerServer-1	B62T1D	Failed update schema: java.io.IOException: java.util.concurrent.ExecutionException: java.io.IOException: Seek behind the end of buffer: pos=2550, length=1047
//    at ru.yandex.msearch.DeleteHandlingIndexReader.deleteDocuments(DeleteHandlingIndexReader.java:273)
//    at ru.yandex.msearch.Shard.reopenShardSearcher(Shard.java:1083)
//    at ru.yandex.msearch.Shard.doFlush(Shard.java:1430)
//    at ru.yandex.msearch.Index.flushShardSync(Index.java:1144)
//    at ru.yandex.msearch.Index.reloadFields(Index.java:1068)
//    at ru.yandex.msearch.Index.reloadFields(Index.java:1034)
//    at ru.yandex.msearch.UpdateSchemaHandler.handle(UpdateSchemaHandler.java:61)
//    at ru.yandex.http.server.sync.OverridableHttpRequestHandler.handle(OverridableHttpRequestHandler.java:39)
//    at org.apache.http.protocol.HttpService.doService(HttpService.java:437)
//    at ru.yandex.http.server.sync.BaseHttpService.doService(BaseHttpService.java:134)
//    at org.apache.http.protocol.HttpService.handleRequest(HttpService.java:342)
//    at ru.yandex.http.server.sync.BaseHttpService.handleRequest(BaseHttpService.java:99)
//    at ru.yandex.http.server.sync.BaseHttpServer$Worker.run(BaseHttpServer.java:513)
//    at java.base/java.util.concurrent.ThreadPoolExecutor.runWorker(ThreadPoolExecutor.java:1130)
//    at java.base/java.util.concurrent.ThreadPoolExecutor$Worker.run(ThreadPoolExecutor.java:630)
//    at java.base/java.lang.Thread.run(Thread.java:832)
//    Caused by: java.util.concurrent.ExecutionException: java.io.IOException: Seek behind the end of buffer: pos=2550, length=1047
//    at java.base/java.util.concurrent.FutureTask.report(FutureTask.java:122)
//    at java.base/java.util.concurrent.FutureTask.get(FutureTask.java:191)
//    at ru.yandex.msearch.DeleteHandlingIndexReader.deleteDocuments(DeleteHandlingIndexReader.java:270)
//        ... 15 more
//    Caused by: java.io.IOException: Seek behind the end of buffer: pos=2550, length=1047
//    at org.apache.lucene.store.MutableBlockCompressedInputStream.seek(MutableBlockCompressedInputStream.java:348)
//    at org.apache.lucene.store.MutableBlockCompressedInputStream.seek(MutableBlockCompressedInputStream.java:315)
//    at org.apache.lucene.index.codecs.yandex.YandexPostingsReader$SegmentDocsEnum.reset(YandexPostingsReader.java:456)
//    at org.apache.lucene.index.codecs.yandex.YandexPostingsReader.docs(YandexPostingsReader.java:319)
//    at org.apache.lucene.index.codecs.yandex2.Yandex2TermsReader$FieldReader$SegmentTermsEnum.docs(Yandex2TermsReader.java:1254)
//    at org.apache.lucene.search.TermQuery$TermWeight.scorer(TermQuery.java:103)
//    at org.apache.lucene.search.BooleanQuery$BooleanWeight.scorer(BooleanQuery.java:365)
//    at org.apache.lucene.search.BooleanQuery$BooleanWeight.scorer(BooleanQuery.java:365)
//    at ru.yandex.msearch.DeleteHandlingIndexReader$DeleteHandlingReaderLeave.deleteDocuments(DeleteHandlingIndexReader.java:490)
//    at ru.yandex.msearch.DeleteHandlingIndexReader.deleteDocuments(DeleteHandlingIndexReader.java:294)
//    at ru.yandex.msearch.DeleteHandlingIndexReader.deleteDocumentsBatch(DeleteHandlingIndexReader.java:229)
//    at ru.yandex.msearch.DeleteHandlingIndexReader$1.call(DeleteHandlingIndexReader.java:259)
//    at ru.yandex.msearch.DeleteHandlingIndexReader$1.call(DeleteHandlingIndexReader.java:256)
//    at java.base/java.util.concurrent.FutureTask.run(FutureTask.java:264)
//        ... 3 more
    public synchronized void reloadFields(
        final String name,
        final boolean force,
        final String schema,
        final int shard)
        throws IOException
    {
        // validate
        boolean needUpdate = true;
        if (!force) {
            File file = config.fieldsConfig().dynamicFieldsFiles().get(name);
            try {
                if (file != null && file.exists() && file.isFile()) {
                    IniConfig newConfig = new IniConfig(new StringReader(schema));
                    IniConfig oldConfig = new IniConfig(file);
                    needUpdate = !oldConfig.toString().trim().equals(newConfig.toString().trim());
                }
            } catch (ConfigException ce) {
                throw new IOException("Failed to validate configs, old " + file, ce);
            }
        }

        if (!needUpdate) {
            logger.info("No need to update, skipping");
            return;
        }

        logger.info("Reloading fields");
        try {
            if (shard < 0) {
                for (int i = 0; i < shardsCount; i++) {
                    flushShardSync(i, true);
                }
            } else {
                flushShardSync(shard, true);
            }

            config.reloadDynamicConfig(name, schema);
            cp.set(createCodecProvider(false));
            memCp.set(createCodecProvider(true));
            // we already set new codec provider, what will occur if we just partially updateSchemas?
            if (shard < 0) {
                for (int i = 0; i < shardsCount; i++) {
                    shards[i].updateSchema();
                    logger.info("Fields were reloaded for shard " + i);
                }
            } else {
                shards[shard].updateSchema();
                logger.info("Fields were reloaded for shard " + shard);
            }
        } catch (ConfigException ce) {
            throw new IOException(ce);
        } finally {
            if (shard < 0) {
                for (int i = 0; i < shardsCount; i++) {
                    shards[i].flush(false, false);
                }
            } else {
                shards[shard].flush(false, false);
            }
        }
    }

    @Override
    public void reopenShardsReader(final int shard, final long readerGen)
    {
        commonTaskExecutor.submit(new ReopenShardReaderTask(shard, readerGen));
    }

    public void dropIndex() {
        commonTaskExecutor.submit(
            new Runnable() {
                @Override
                public void run() {
                    logger.severe("Dropping index. Sleep 10. Exiting");
                    try {
                        new File(config.indexPath(), "drop-index").createNewFile();
                        Thread.sleep(10000);
                        System.exit(100500);
                    } catch (Exception e) {
                        logger.log(
                            Level.SEVERE,
                            "Error droping index",
                            e);
                    }
                }
            });
    }

    @Override
    public void flushShard( int shard ) throws IOException
    {
        commonTaskExecutor.submit( new ShardFlushTask(shard) );
    }

    @Override
    public void reopenMultiSearcherAsync(final Shard shard) {
        commonTaskExecutor.submit(
            new ReopenShardMultiSearcherTask(shard));
    }

    public final boolean indexingEnabled(final int shard) {
        return shards[shard].indexingEnabled();
    }

    public final void flushShardSync(int shard, Boolean ro) throws IOException {
        shards[shard].flush(false, ro);
        Future sync = shards[shard].doFlush(true);
        if (sync != null) {
            try {
                sync.get();
            } catch (ExecutionException e) {
                throw new IOException(
                    "Shard<" + shard + "> sync flush failed",
                    e);
            } catch (InterruptedException e) {
                Thread.currentThread().interrupt();
            }
        }
    }

    private final void flushShardAsync(final int shard, final Boolean ro) {
        try {
            shards[shard].flush(true, ro);
        } catch (Exception e) {
            logger.log(
                Level.SEVERE,
                "Error ocurred while trying to flush shard<" + shard + ">",
                e);
            recomputeRamUsage();
            synchronized( checkShardsLock )
            {
                shardsInFlush[shard] = false;
            }
        }
    }

    private final void recomputeRamUsage()
    {
        synchronized( checkShardsLock )
        {
            docsInMemTotal = 0;
            docsInMemActive = 0;
            for (int i = 0; i < shardsCount; i++) {
                docsInMemActive += shards[i].getActiveRamUsed();
                docsInMemTotal += shards[i].getRamUsed();
            }
            lastMemRecompute = System.currentTimeMillis();
            if (docsInMemTotal < maxDocsInMem) {
                checkShardsLock.notifyAll();
            }
        }
    }

    private void checkShards(final long ramUsed) throws IOException {
        int maxShard = -1;
        synchronized (checkShardsLock) {
            //first - recompute actual memory consumption, because memoryindex may change it without notifying us.
            docsInMemTotal += ramUsed;
            docsInMemActive += ramUsed;
            if (docsInMemActive >= docsInMemFlushThreshold) {
                recomputeRamUsage();
                if (logger.isLoggable(Level.INFO)) {
                    logger.info("Ram usage: " + docsInMemTotal
                        + " (" + docsInMemActive + ")");
                }
                //all ok after recompute
                if (docsInMemActive < docsInMemFlushThreshold) return;

                //else enqueue as much shards for flushing so
                //docsInMemActive became less then docsInMemFlushThreshold
                while (docsInMemActive >= docsInMemFlushThreshold) {
                    int maxShardNumber = -1;
                    long maxShardRamUsed = 0;
                    for (int i = 0; i < shardsCount; i++) {
                        if (shardsInFlush[i]) {
                            continue;
                        }
                        long shardRamUsed = shards[i].getActiveRamUsed();
                        if (shardRamUsed > maxShardRamUsed) {
                            maxShardRamUsed = shardRamUsed;
                            maxShardNumber = i;
                        }
                    }
                    if (maxShardNumber == -1) {
                        if (logger.isLoggable(Level.INFO)) {
                            logger.info("checkShards: All shards are "
                                + "flushing nothing to select");
                        }
                        break;
                    }
                    docsInMemActive -= maxShardRamUsed;
//                    docsInMemTotal -= maxShardRamUsed;
                    shardsInFlush[maxShardNumber] = true;
                    Shard shard = shards[maxShardNumber];
                    AbstractPart.PartsStats pStats =
                        shard.getPartsStats();
                    if (logger.isLoggable(Level.INFO)) {
                        logger.info("Ram usage: " + docsInMemTotal
                            + " (" + docsInMemActive + ")"
                            + ", trying to flush shard no: " + maxShardNumber
                            + ", with ram consumption: "
                            + shard.getRamUsed()
                            + " (" + shard.getActiveRamUsed() + ") "
                            + "/ docsCount:" + shard.getDocsInMem()
                            + ", parts.stats: " + pStats.toString());
                    }
                    flushShardAsync(maxShardNumber, null);
                }
                if (docsInMemActive < docsInMemFlushThreshold) return;
            }

            //if shards are flushing but used memory continues to grow
            //throttle indexing
//            if (docsInMemTotal >= maxDocsInMem) {
            final int delay = throttleDelay;
            if (delay > 0) {
                logger.warning("Ram usage is increasing to fast."
                    +" Throttling indexing.");
                try {
                    checkShardsLock.wait(delay);
                } catch (InterruptedException e) {
                    Thread.currentThread().interrupt();
                }
            }
        }
    }

    private String prevRamUsage = "";
    private String prevLimiterStats = "";
    private void computeThrottleDelay() {
        synchronized (checkShardsLock) {
            if ((System.currentTimeMillis() - lastMemRecompute)
                > RECOMPUTE_MEM_INTERVAL)
            {
                recomputeRamUsage();
                String ramUsage = "Ram usage: " + docsInMemTotal
                    + " (" + docsInMemActive + ")";
                if (!ramUsage.equals(prevRamUsage)) {
                    if (logger.isLoggable(Level.INFO)) {
                        logger.info(ramUsage);
                    }
                    prevRamUsage = ramUsage;
                }
                if (HttpIndexerServer.limiter != null) {
                    String limiterStats = "SumLimiter ram: "
                        + HttpIndexerServer.limiter.status().toString();
                    if (!limiterStats.equals(prevLimiterStats)) {
                        if (logger.isLoggable(Level.CONFIG)) {
                            logger.config(limiterStats);
                        }
                        prevLimiterStats = limiterStats;
                    }
                }
            }
//            if (docsInMemTotal >= maxDocsInMem) {
                long diff = (docsInMemTotal - maxDocsInMem) / 1024 / 1024;
//                if (docsInMemTotal < docsInMemFlushThreshold) {
//                    diff = 0;
//                }
//                diff = Math
//                if (diff < 0) diff = 0;
                float delay = THROTTLE_REGULATOR_K
                    * (diff + (1.0f/THROTTLE_REGULATOR_PERIOD)
                    * throttleRegulatorIntegral
                    + THROTTLE_REGULATOR_PERIOD
                    * throttleRegulatorDifferential);

                throttleRegulatorIntegral += diff;
                if (throttleRegulatorIntegral < 0) {
                    //we can delay indexing but we can not speed up it
                    throttleRegulatorIntegral = 0;
                }
                throttleRegulatorDifferential =
                    diff - throttleRegulatorPrevDiff;
                throttleRegulatorPrevDiff = diff;
//                if ((int)delay < 1) throttleDelay = 1;
//                else throttleDelay = (int)delay;
                throttleDelay = (int)delay;
                if (throttleDelay > 0) {
                    if (logger.isLoggable(Level.INFO)) {
                        logger.info("calculateThrottleDelay: delay=" + delay
                            + ", throttleDelay=" + throttleDelay
                            + ", throttleRegulatorIntegral="
                            + throttleRegulatorIntegral
                            + ", throttleRegulatorDifferential="
                            + throttleRegulatorDifferential
                            + ", diff=" + diff);
                    }
                }
//            if (throttleDelay < 1) throttleDelay = 1;
//            }
//            else {
//                throttleDelay = 0;
//                throttleRegulatorPrevDiff = 0;
//                throttleRegulatorIntegral = 0;
//                throttleRegulatorDifferential = 0;
//            }
        }
    }

    public AnalyzerProvider analyzerProvider() {
        return analyzerProvider;
    }

    @Override
    public PrefixingAnalyzerWrapper indexAnalyzer() {
        return analyzerProvider.indexAnalyzer();
    }

    @Override
    public PrefixingAnalyzerWrapper searchAnalyzer() {
        return analyzerProvider.searchAnalyzer();
    }

    public void search(
        final PrefixedQuery query,
        final FlushableCollector collector)
        throws IOException
    {
        if (query.prefix == null) {
            search(query.query, collector);
        } else {
            search(query.query, collector, query.prefix);
        }
    }

    public void search(
        final Query query,
        final FlushableCollector collector)
        throws IOException
    {
        for (Shard shard: shards) {
            shard.search(query, collector);
        }
    }

    public void search(
        final Query query,
        final FlushableCollector collector,
        final Prefix prefix)
        throws IOException
    {
        shards[(int) (prefix.hash() % shardsCount)].search(query, collector);
    }

    @Override
    public long dirtyShardQueueId(final QueueShard shard) {
        AtomicLong queueId = queueIdsDirtyMap.get(shard);
        if (queueId == null) {
            return -1;
        }
        return queueId.get();
    }

    @Override
    public void dirtifyQueueShard(final QueueShard shard, final long id) {
        if (config.kosherShards()) {
            return;
        }
        AtomicLong queueId = queueIdsDirtyMap.get(shard);
        if (queueId == null) {
            AtomicLong newId = new AtomicLong(id);
            queueId = queueIdsDirtyMap.putIfAbsent(shard, newId);
            if (queueId == null) {
                queueId = newId;
            }
        }
        long oldId;
        while ((oldId = queueId.get()) < id) {
            queueId.compareAndSet(oldId, id);
        }
    }

    @Override
    public void clearDirtyQueueShard(final QueueShard shard) {
        queueIdsDirtyMap.remove(shard);
    }

    public boolean isDirty(final QueueShard shard) {
        return queueIdsDirtyMap.get(shard) != null;
    }

    public void pushQueueId(
        final QueueShard queueShard,
        final QueueId id,
        final int skipShard)
    {
        for (int i = 0; i < shardsCount; i++) {
            if (i == skipShard) {
                continue;
            }
            final Shard shard = shards[i];
            final long currentId = shard.queueId(queueShard);
            if (currentId < id.get()) {
                final String json = "{\"prefix\":" + i + ", \"docs\":[]}";
//                Logger.debug("Pushing: " + queueShard + "=" + id.get()
//                     + " to shard: " + shard.shardNo);
                try {
                    shard.process(
                        new JsonBatchDeleteMessage(
                            this,
                            json.getBytes(StandardCharsets.UTF_8),
                            StandardCharsets.UTF_8,
                            Message.Priority.ONLINE,
                            queueShard,
                            new MessageQueueId(id.get(), id.get(), false),
                            config,
                            config.useJournal(),
                            config.orderIndependentUpdate()));
                } catch (IOException|ParseException e) {
                    if (logger.isLoggable(Level.WARNING)) {
                        logger.log(
                            Level.WARNING,
                            "Can't push fake queueid update message: "
                                + "currentId: " + currentId
                                + ", push: " + id.get()
                                + ", reloadCurrentId:"
                                    + shard.queueId(queueShard)
                                + ", memoryIndexQueueId: "
                                    + shard.memoryIndex.queueId(queueShard)
                                + ", shard: " + i,
                                e);
                    }
                }
            }
        }
    }

    @Override
    public Future pushQueueIds(
        final Map<QueueShard, QueueId> ids,
        final Shard from)
    {
        final Runnable updater = new Runnable() {
            @Override
            public void run() {
                if (logger.isLoggable(Level.INFO)) {
                    logger.info("Running queue ids pusher from shard "
                        + from.shardNo);
                }
                for (Map.Entry<QueueShard, QueueId> entry : ids.entrySet()) {
                    final QueueShard shard = entry.getKey();
                    final QueueId id = entry.getValue();
                    if (id.dirty()) {
                        pushQueueId(shard, entry.getValue(), from.shardNo);
                    }
                }
            }};
        return commonTaskExecutor.submit(updater);
    }

    public long queueId(
        final QueueShard queueShard,
        final Prefix prefix,
        final boolean checkCopyness)
        throws IOException
    {
        if (checkCopyness) {
            checkShardCopied(queueShard);
        }
        return shards[(int)(prefix.hash() % shardsCount)].queueId(queueShard);
    }

    @Override
    public long maxSavedQueueId(final QueueShard shard) {
        long maxId = -1;
        for (Shard s : shards) {
            long queueId = s.savedQueueId(shard);
            if (maxId < queueId) {
                maxId = queueId;
            }
        }
        return maxId;
    }

    public long queueId(
        final QueueShard shard,
        final boolean checkCopyness)
        throws IOException
    {
        if (checkCopyness) {
            checkShardCopied(shard);
        }
        final AtomicLong dirtyfiedId = queueIdsDirtyMap.get(shard);
        if (dirtyfiedId != null) {
            return dirtyfiedId.get();
        }
        if (config.kosherShards()) {
            int luceneShard = shard.shardId() % config.shards();
            return shards[luceneShard].queueId(shard);
        } else if (config.returnMaxQueueId()) {
            return maxQueueId(shard);
        } else {
            return minQueueId(shard);
        }
    }

    public long minQueueId(final QueueShard shard) throws IOException {
        long minQueueId = Long.MAX_VALUE;
        for (Shard s : shards) {
            long queueId = s.queueId(shard);
            if (queueId != -1 && queueId < minQueueId) {
                minQueueId = queueId;
            }
        }
        if (minQueueId == Long.MAX_VALUE) {
            minQueueId = -1;
        }
        return minQueueId;
    }


    public long maxQueueId(final QueueShard shard) throws IOException {
        long maxQueueId = -1;
        for (Shard s : shards) {
            long queueId = s.queueId(shard);
            if (queueId > maxQueueId) {
                maxQueueId = queueId;
            }
        }
        return maxQueueId;
    }

    public void deleteDocument(final PrimaryKey key)
        throws IOException, ParseException
    {
        processMessage(
            new DeleteByQueryNonJournalable(
                this,
                key.prefix(),
                key.queryProducer()),
            true);
    }

    public void deleteTerm(final Prefix prefix, final Term term)
        throws IOException, ParseException
    {
        final TermQuery query = new TermQuery(term);
        processMessage(
            new DeleteByQueryNonJournalable(this, prefix, query),
            true);
    }

    public void processMessage(
        final JournalableMessage message,
        final boolean checkShards)
        throws IOException, ParseException
    {
        long memUsage = shards[(int) (message.prefix().hash() % shardsCount)]
            .process(message);
        if (checkShards) {
            checkShards(memUsage);
        }
    }

    private class ShardTask implements Callable<Void> {
        private final JournalableMessage message;
        private final boolean checkShards;

        public ShardTask(
            final JournalableMessage message,
            final boolean checkShards)
        {
            this.message = message;
            this.checkShards = checkShards;
        }

        @Override
        public Void call() throws IOException, ParseException {
            processMessage(message, checkShards);
            return null;
        }
    }

    public Future<Void> dispatch(
        final JournalableMessage message,
        final boolean force,
        final boolean checkShards)
        throws IOException, ParseException
    {
        return submitTask(
            new ShardTask(message, checkShards),
            message.priority(),
            force);
    }

    @Override
    public void add(DocumentsMessage docs, Map<String, String> conditions) throws IOException {
        throw new UnsupportedOperationException(
            "Index is unable to handle dispatchable message");
    }

    @Override
    public void modify(DocumentsMessage message, Map<String, String> conditions) throws IOException {
        throw new UnsupportedOperationException(
            "Index is unable to handle dispatchable message");
    }

    @Override
    public void delete(DocumentsMessage message) throws IOException {
        throw new UnsupportedOperationException(
            "Index is unable to handle dispatchable message");
    }

    @Override
    public void delete(DeleteMessage message) throws IOException {
        throw new UnsupportedOperationException(
            "Index is unable to handle dispatchable message");
    }

    @Override
    public void update(DocumentsMessage message, Map<String, String> conditions, boolean addIfNotExists, boolean orderIndependentUpdate, Set<String> preserveFields) throws IOException {
        throw new UnsupportedOperationException(
            "Index is unable to handle dispatchable message");
    }

    @Override
    public void updateIfNotMatches(UpdateIfNotMatchesMessage message,
                                   Map<String, String> conditions,
                                   boolean addIfNotExists,
                                   boolean orderIndependentUpdate) throws IOException {
        throw new UnsupportedOperationException(
            "Index is unable to handle dispatchable message");
    }

    @Override
    public void update(UpdateMessage message,
                       Map<String, String> conditions,
                       boolean addIfNotExists,
                       boolean orderIndependentUpdate) throws IOException {
        throw new UnsupportedOperationException(
            "Index is unable to handle dispatchable message");
    }

    @Override
    public long journalMessage( JournalableMessage message ) throws IOException {
        throw new UnsupportedOperationException(
            "Index is unable to handle dispatchable message");
    }

    @Override
    public void reopen() {
        reopenIndexes(null, false, false);
    }

    @Override
    public void reopen(
        final IntSet shards,
        final Boolean ro,
        final boolean doWait)
        throws IOException
    {
        reopenIndexes(shards, ro, doWait);
    }

    public void traverseShards(final ShardVisitor visitor) throws IOException {
        for (int i = 0; i < shardsCount; ++i) {
            visitor.visit(i, shards[i]);
        }
    }

    private class IndexStater extends Thread implements Stater {
        private volatile long indexSizeMb = 0;
        private final int sizeCalcInterval;
        private volatile long numDocs = 0;
        private volatile long allActivePrefixes = 0;
        private volatile long timedActivePrefixes = 0;

        public IndexStater(int sizeCalcInterval) {
            super("IdxSizeCalc");
            setDaemon(true);
            this.sizeCalcInterval = sizeCalcInterval;
        }

        @Override
        public void run() {
            long perShardSleep = INDEX_STATER_TRAVERSE_TIME / shards.length;
            while(!exit) {
                try {
                    long indexSizeMb = 0;
                    long numDocs = 0;
                    long allActivePrefixes = 0;
                    long timedActivePrefixes = 0;
                    for (Shard shard : shards) {
                        indexSizeMb += shard.indexSizeMb();
                        numDocs += shard.numDocsLong();
                        Thread.sleep(perShardSleep);
                        allActivePrefixes += shard.activePrefixes();
                        timedActivePrefixes += shard.timedActivePrefixes();
                    }
                    this.indexSizeMb = indexSizeMb;
                    this.numDocs = numDocs;
                    this.allActivePrefixes = allActivePrefixes;
                    this.timedActivePrefixes = timedActivePrefixes;
                    Thread.sleep(sizeCalcInterval);
                } catch (InterruptedException e) {
                    Thread.currentThread().interrupt();
                    break;
                } catch (Exception e) {
                    e.printStackTrace();
                }
            }
        }

        private String signalNameWithDb(final String name) {
            if (config.name() == DatabaseManager.DEFAULT_DATABASE) {
                return name;
            }

            return StringUtils.concat(config.name(), '-', name);
        }

        @Override
        public <E extends Exception> void stats(
            final StatsConsumer<? extends E> statsConsumer)
            throws E
        {
            statsConsumer.stat(
                signalNameWithDb("merges_ammm"),
                concurrentScheduler.mergesInProgress());
            statsConsumer.stat(
                signalNameWithDb("queued_merges_ammx"),
                concurrentScheduler.queuedMerges());
            statsConsumer.stat(signalNameWithDb("index-size_mb_ammx"), indexSizeMb);
            statsConsumer.stat(signalNameWithDb("index-size_mb_axxx"), indexSizeMb);
            statsConsumer.stat(signalNameWithDb("index-size_mb_avvv"), indexSizeMb);
            statsConsumer.stat(signalNameWithDb("index-size_mb_annn"), indexSizeMb);

            statsConsumer.stat(signalNameWithDb("index-numdocs_ammx"), numDocs);
            statsConsumer.stat(signalNameWithDb("index-numdocs_axxx"), numDocs);
            statsConsumer.stat(signalNameWithDb("index-numdocs_avvv"), numDocs);
            statsConsumer.stat(signalNameWithDb("index-numdocs_annn"), numDocs);

            statsConsumer.stat(signalNameWithDb("active-prefixes_ammx"), allActivePrefixes);
            statsConsumer.stat(signalNameWithDb("active-prefixes_axxx"), allActivePrefixes);
            statsConsumer.stat(signalNameWithDb("active-prefixes_avvv"), allActivePrefixes);

            statsConsumer.stat(
                signalNameWithDb("timed-active-prefixes_ammx"),
                timedActivePrefixes);
            statsConsumer.stat(
                signalNameWithDb("timed-active-prefixes_axxx"),
                timedActivePrefixes);
            statsConsumer.stat(
                signalNameWithDb("timed-active-prefixes_avvv"),
                timedActivePrefixes);
//            statsConsumer.stat("active-prefixes_annn", activePrefixes);

            statsConsumer.stat(signalNameWithDb("index-throttle-delay_axxx"), throttleDelay);
        }

        @Override
        public void addToGolovanPanel(
            final GolovanPanel panel,
            final String statsPrefix)
        {
            ImmutableGolovanPanelConfig config = panel.config();
            GolovanChartGroup group =
                new GolovanChartGroup(statsPrefix, statsPrefix);

            GolovanChart merges = new GolovanChart(
                "merges",
                " merges in progress",
                false,
                true,
                0d);
            merges.addSplitSignal(
                config,
                "mul(" + statsPrefix + signalNameWithDb("merges_ammm") + ",5)",
                0,
                false,
                false);
            group.addChart(merges);

            GolovanChart queuedMerges = new GolovanChart(
                "queued-merges",
                " queued merges",
                false,
                false,
                0d);
            queuedMerges.addSplitSignal(
                config,
                statsPrefix + signalNameWithDb("queued_merges_ammx"),
                0,
                false,
                false);
            group.addChart(queuedMerges);

            GolovanChart numdocs = new GolovanChart(
                "numdocs",
                " number of documents",
                false,
                false,
                null);
            numdocs.addSplitSignal(
                config,
                statsPrefix + signalNameWithDb("index-numdocs_ammx"),
                0,
                false,
                false);
            group.addChart(numdocs);

            panel.addCharts(
                "index",
                null,
                group);
        }
    }

    private class FieldsCacheReloader extends Thread {
        public FieldsCacheReloader() {
            super("FCReloader");
            setDaemon(true);
        }

        @Override
        public void run() {
            IOScheduler.setThreadReadPrio(IOScheduler.IOPRIO_WARM);
            IOScheduler.setThreadWritePrio(IOScheduler.IOPRIO_WARM);
            while(true) {
                try {
                    for (Shard shard: shards) {
                        shard.checkReloadFieldsCache();
                    }
                    Thread.sleep(FIELDS_CACHE_RELOAD_INTERVAL);
                } catch (Exception e) {
                    e.printStackTrace();
                }
            }
        }
    }
}

class IndexWarmer implements Runnable {
    Index index;

    public IndexWarmer(Index index) {
        this.index = index;
    }

    public void run() {
        while (true) {
            Index.WarmerWork w = null;
            try {
                w = index.getWarmerWork();
                if (w != null) {
//		    w.cache.cacheDeprecated( w.oldReader, w.reader );
                    if (w.closeOld == true) {
                        try {
                            w.oldReader.close();
                        } catch (Exception e) {
                            e.printStackTrace();
                        }
                    }
                }
            } catch (Exception e) {
                e.printStackTrace();
                if (w != null) {
                    System.err.println("Warmer: failed reader: " + w.reader.toString() + ", oldReader=" + w.oldReader);
                }
                continue;
            }
        }
    }
}
