package ru.yandex.msearch;

import java.io.OutputStream;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Set;

import java.util.logging.Level;
import java.util.logging.Logger;

import org.apache.lucene.index.IndexReader;
import org.apache.lucene.index.IndexWriter;
import org.apache.lucene.index.IndexWriterConfig;
import org.apache.lucene.index.LogByteSizeMergePolicy;
import org.apache.lucene.util.Version;
import org.apache.lucene.analysis.SimpleAnalyzer;
import org.apache.lucene.index.codecs.CodecProvider;
import org.apache.lucene.index.codecs.CoreCodecProvider;
import org.apache.lucene.index.codecs.yandex.YandexCodec;

import ru.yandex.logger.PrefixedLogger;

import ru.yandex.search.prefix.LongPrefix;

public class IndexDumper
{
    private static final long MIN_FLUSH_INTERVAL = 3600 * 1000;
    private static final int MAX_BATCH_DOCS = 2000000;
    private Index index;
    private boolean fullDump;
    private int outShards;
    private int userShardStart;
    private int userShardEnd;
    private int outShard;
    private Set<String> shardingFields;
    private final boolean lockIndex;
    private CodecProvider    cp;
    private final int dumpMerge;
    private final Logger logger;

    public IndexDumper(
        final Index index,
        final int shard,
        final int totalShards,
        final int userShardStart,
        final int userShardEnd,
        final Set<String> shardingFields,
        final boolean lockIndex,
        final PrefixedLogger logger)
    {
        this.index = index;
        fullDump = false;
        this.userShardStart = userShardStart;
        this.userShardEnd = userShardEnd;
        this.lockIndex = lockIndex;
        this.outShards = totalShards;
        this.outShard = shard;
        this.shardingFields = shardingFields;
        //this.config = config;
        if (outShards == index.shardsCount()) {
            dumpMerge = 1;
        } else {
            dumpMerge = Math.max(1, index.shardsCount() / outShards);
        }
        this.logger = logger.replacePrefix(
            "IndexDumper[s:" + userShardStart + '-' + userShardEnd
            + ", out:" + shard + '/' + totalShards + "]");
    }

    public boolean dump(final OutputStream outStream) throws IOException {
        DumpSearcher s = null;
        Boolean ro = null;
        final Set<Integer> affectedShards = new LinkedHashSet<>();
        try {
            if (outShards == index.shardsCount()) {
                if (lockIndex) {
                    ro = !index.getShard(outShard).indexingEnabled();
                    index.getShard(outShard).flush(false, true);
                } else {
                    index.getShard(outShard).flush(false, null);
                }
                index.getShard(outShard).doFlush(true);
                s = index.getDumpSearcher(outShard, userShardStart, userShardEnd);
            } else {
                final int maxShards = Math.max(index.shardsCount(), outShards);
                final int minShards = Math.min(index.shardsCount(), outShards);
                if (maxShards % minShards == 0) {
                    for (int i = 0; i < maxShards; i++) {
                        int outerMod = i % outShards;
                        int thisMod = i % index.shardsCount();
                        if (outerMod == outShard) {
                            affectedShards.add(thisMod);
                        }
                    }
                } else {
                    for (int i = 0; i < index.shardsCount(); i++) {
                        affectedShards.add(i);
                    }
                }
                for(Integer shard : affectedShards) {
                    if (
                        System.currentTimeMillis()
                            - index.getShard(shard).lastFlush()
                            > MIN_FLUSH_INTERVAL)
                    {
                        index.getShard(shard).flush(false, null);
                        index.getShard(shard).doFlush(true);
                    }
                }
                s = index.getDumpSearcher(
                    affectedShards,
                    userShardStart,
                    userShardEnd);
            }
            logger.info("affectedShards=" + affectedShards);
            ArrayList<IndexReader> filteredReaders = new ArrayList<>();
            filterReaders(filteredReaders, s.reader());

            logger.info("filteredReader.size() = "
                + filteredReaders.size() + ", dumpMerge=" + dumpMerge);

            //TODO rewrite using Visitor pattern

            Iterator<IndexReader> iter = filteredReaders.iterator();
            ArrayList<IndexReader> batch = new ArrayList<>();
            int batchDocs = 0;
            while (iter.hasNext()) {
                IndexReader reader = iter.next();
                int numDocs = reader.numDocs();
                if (batch.size() == 0) {
                    batch.add(reader);
                    batchDocs += numDocs;
                } else {
                    if (batch.size() >= dumpMerge
                        || batchDocs + reader.numDocs() > MAX_BATCH_DOCS)
                    {
                        dumpReaders(
                            batch.toArray(new IndexReader[batch.size()]),
                            outStream);
                        batch.clear();
                        batchDocs = 0;
                    }
                    batch.add(reader);
                    batchDocs += numDocs;
                }
            }
            if (batch.size() > 0) {
                dumpReaders(
                    batch.toArray(new IndexReader[batch.size()]),
                    outStream);
                batch.clear();
                batchDocs = 0;
            }

            PrintStreamDirectory outDir = new PrintStreamDirectory( outStream, index.tmpDirectory() );
            outDir.sendQueueIds(s.queueIds(), userShardStart, userShardEnd);
            outDir.finished();

            //EOF
            outDir = new PrintStreamDirectory( outStream, index.tmpDirectory() );
            outDir.finished();
            return true;
        } catch (Exception e) {
            e.printStackTrace();
            throw new IOException( e );
        } finally {
            if (lockIndex) {
                if (outShards == index.shardsCount()) {
                    index.getShard(outShard).flush(true, ro);
                }
            }
            if (s != null) {
                s.free();
            }
        }
    }

    public boolean dumpSearcher( Searcher searcher, OutputStream outStream ) throws IOException
    {
        IndexReader origReader = searcher.reader();
        dumpReader( origReader, outStream );
        PrintStreamDirectory outDir = new PrintStreamDirectory( outStream, index.tmpDirectory() );
        outDir.finished();
        return true;
    }

    public void filterReaders(
        final List<IndexReader> filtered,
        final IndexReader reader)
        throws IOException
    {
        IndexReader[] subReaders = reader.getSequentialSubReaders();
        if (subReaders != null) {
            for (int i = 0; i < subReaders.length; i++) {
                filterReaders(filtered, subReaders[i]);
            }
            return;
        }

        SuidFilterIndexReader filterReader =
            new SuidFilterIndexReader(
                reader,
                userShardStart,
                userShardEnd,
                outShards,
                outShard,
                shardingFields,
                index.config(),
                logger);
//        IndexReader filterReader = reader;
        if (filterReader.numDocs() > 0) {
            filtered.add(filterReader);
        }
    }

    public boolean dumpReaders(
        final IndexReader[] readers,
        final OutputStream outStream)
        throws IOException
    {
        if (logger.isLoggable(Level.INFO)) {
            logger.info("Dumping readers: "
                + java.util.Arrays.toString(readers));
        }
        IndexWriter outWriter = null;
        try {
            PrintStreamDirectory outDir = new PrintStreamDirectory( outStream, index.tmpDirectory() );
            IndexWriterConfig config = new IndexWriterConfig( Version.LUCENE_40, new SimpleAnalyzer( true ) );
            config.setTermIndexInterval(48);
            config.setRAMBufferSizeMB(10);
            config.setFieldsWriterBufferSize(index.config().yandexFieldsWriterBufferSize());
            LogByteSizeMergePolicy mergePolicy = new LogByteSizeMergePolicy();
            mergePolicy.setUseCompoundFile( false );
            mergePolicy.setMergeFactor( 2 );
            config.setMergePolicy( mergePolicy );

//          cp = new CoreCodecProvider();
//          cp.setDefaultFieldCodec("Standard");
//          config.setCodecProvider(cp);
            config.setCodecProvider(index.codecProvider());

            outWriter = new IndexWriter(
                outDir,
                config,
                index.config().storedFields(),
                index.config().indexedFields());

            outWriter.addIndexes(readers);
            outWriter.commit();
            outDir.close();
        } finally {
            if (outWriter != null) {
                outWriter.close();
            }
        }
        return true;
    }

    public boolean dumpReader(
        final IndexReader reader,
        final OutputStream outStream)
        throws IOException
    {
        IndexReader[] subReaders = reader.getSequentialSubReaders();
        if (subReaders != null) {
            for (int i = 0; i < subReaders.length; i++) {
                dumpReader(subReaders[i], outStream);
            }
            return true;
        }

        SuidFilterIndexReader filterReader = null;
        IndexWriter outWriter = null;

        try {
            if (!fullDump) {
                filterReader =
                    new SuidFilterIndexReader(
                        reader,
                        userShardStart,
                        userShardEnd,
                        outShards,
                        outShard,
                        shardingFields,
                        index.config(),
                        logger);
		if( filterReader.numDeletedDocs() == filterReader.maxDoc() )
		{
		    PrintStreamDirectory outDir = new PrintStreamDirectory( outStream, index.tmpDirectory() );
		    outDir.close();
		    return true;
		}
	    }
	    PrintStreamDirectory outDir = new PrintStreamDirectory( outStream, index.tmpDirectory() );
	    IndexWriterConfig config = new IndexWriterConfig( Version.LUCENE_40, new SimpleAnalyzer( true ) );
	    config.setTermIndexInterval( 128 );
	    config.setRAMBufferSizeMB( 10 );
            config.setFieldsWriterBufferSize(index.config().yandexFieldsWriterBufferSize());
	    LogByteSizeMergePolicy mergePolicy = new LogByteSizeMergePolicy();
	    mergePolicy.setUseCompoundFile( false );
	    mergePolicy.setMergeFactor( 2 );
	    config.setMergePolicy( mergePolicy );

//	    cp = new CoreCodecProvider();
//	    cp.setDefaultFieldCodec("Standard");
//	    config.setCodecProvider(cp);
	    config.setCodecProvider( index.codecProvider() );

	    outWriter = new IndexWriter(
                outDir,
                config,
                index.config().storedFields(),
                index.config().indexedFields());

	    outWriter.addIndexes( fullDump ? reader : filterReader );
	    outWriter.commit();
	    outDir.close();
	    filterReader.close();
	}
	finally
	{
	    if( outWriter != null ) outWriter.close();
	    if( filterReader != null ) filterReader.close();
	}
	
	return true;
    }

}
