package ru.yandex.msearch;

import java.io.InputStream;
import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.PrintStream;
import java.io.IOException;

import java.net.InetSocketAddress;
import java.net.Socket;

import java.io.File;

import java.util.HashMap;
import java.util.Map;
import java.util.Set;

import java.util.logging.Logger;

import org.apache.lucene.store.Directory;
import org.apache.lucene.store.FSDirectory;

import org.apache.lucene.index.IndexReader;
import org.apache.lucene.index.IndexWriter;
import org.apache.lucene.index.IndexWriterConfig;
import org.apache.lucene.index.TieredMergePolicy;

import org.apache.lucene.util.Version;

import org.apache.lucene.analysis.SimpleAnalyzer;

import org.apache.lucene.index.codecs.CodecProvider;

import ru.yandex.msearch.InputStreamDirectory.PartHandler;

import ru.yandex.msearch.config.DatabaseConfig;
import ru.yandex.msearch.jobs.DownloadLimiter;

public class IndexGetter {
    private final Index index;
    private final String url;
    private final int outShard;
    private final int userShardStart;
    private final int userShardEnd;
    private final int timeout;
    private final DatabaseConfig config;
    private final Set<String> shardingFields;
    private final Logger logger;
    private boolean abort = false;
    private File recvPath = null;
    private File optimizePath = null;
    private Directory recvDir = null;
    private Directory optimizeDir = null;
    private InputStreamDirectory inputDir = null;
    private IndexWriter optimizerWriter = null;
    private Socket sock = null;
    private InputStream is = null;
    private PrintStream out = null;
    private Object lock = new Object();
    private CodecProvider cp;
    private final boolean lockIndex;
    private final boolean checkDups;
    private final boolean replace;
    private final boolean disableIndexing;
    private DownloadLimiter limiter = null;

    public IndexGetter(
        final Index index,
        final String url,
        final int outShard,
        final int userShardStart,
        final int userShardEnd,
        final int timeout,
        final Set<String> shardingFields,
        final boolean lockIndex,
        final boolean checkDups,
        final boolean replace,
        final boolean disableIndexing,
        final Logger logger)
    {
        this.index = index;
        this.url = url;
        this.outShard = outShard;
        this.userShardStart = userShardStart;
        this.userShardEnd = userShardEnd;
        this.timeout = timeout;
        this.config = index.config();
        this.shardingFields = shardingFields;
        this.lockIndex = lockIndex;
        this.checkDups = checkDups;
        this.replace = replace;
        this.disableIndexing = disableIndexing;
        this.logger = logger;
    }

    public void setLimiter(final DownloadLimiter limiter) {
        this.limiter = limiter;
    }

    public boolean get() throws IOException {
        final Boolean ro = !index.getShard(outShard).indexingEnabled();
		try {
			if (replace || disableIndexing) {
				index.getShard(outShard).flush(false, disableIndexing);
				index.getShard(outShard).doFlush(true);
			}

			synchronized (lock) {
				if (abort) {
					throw new IOException("Copy aborted");
				}
				recvPath = index.getShard(outShard).createTemporaryDirectory();
				optimizePath =
					index.getShard(outShard).createTemporaryDirectory();
				recvDir = FSDirectory.open(recvPath);
				optimizeDir = FSDirectory.open(optimizePath);
			}
			sock = createConnection(url,
				index.shardsCount(),
				outShard,
				userShardStart,
				userShardEnd);

			is = sock.getInputStream();
			if (limiter != null) {
				is = limiter.wrapInputStream(is);
			}
			is = new BufferedInputStream(is);
			out =
				new PrintStream(new BufferedOutputStream(sock.getOutputStream
					()));
			StringBuilder fields = new StringBuilder();
			for (String field : shardingFields) {
				fields.append(field);
				fields.append(',');
			}
			fields.setLength(fields.length() - 1);
			out.println("GET /?dumpindex&innershards=" + userShardStart + "-"
				+ userShardEnd + "&outshards=" + index.shardsCount() + ":"
				+ outShard
				+ "&sharding-fields=" + fields.toString()
				+ "&lock-index=" + lockIndex
				+ "&db=" + index.config().name()
				+ " HTTP/1.0\n");
			out.flush();

			synchronized (lock) {
				if (abort) {
					throw new IOException("Copy aborted");
				}
				inputDir = new InputStreamDirectory(is, recvDir);
			}

            IndexWriterConfig config = new IndexWriterConfig(
                Version.LUCENE_40,
                new SimpleAnalyzer(true));
            config.setTermIndexInterval(48);
            config.setRAMBufferSizeMB(10);
            config.setFieldsWriterBufferSize(
                this.config.yandexFieldsWriterBufferSize());
            TieredMergePolicy mergePolicy = new TieredMergePolicy();
            mergePolicy.setUseCompoundFile(false);
            mergePolicy.setReclaimDeletesWeight(
                this.config.reclaimDeletesWeight());
            mergePolicy.setConvert(true);
            config.setMergePolicy(mergePolicy);
            config.setMergeScheduler(index.getShard(outShard).mergeScheduler());

			config.setCodecProvider(index.codecProvider());

			synchronized (lock) {
				if (abort) {
					throw new IOException("Copy aborted");
				}
				optimizerWriter = new IndexWriter(
					optimizeDir,
					config,
					this.config.storedFields(),
					this.config.indexedFields());
			}

			IndexPartHandler indexPartHandler =
				new IndexPartHandler(optimizerWriter, config);

			inputDir.recv(indexPartHandler);

			synchronized (lock) {
				if (abort) {
					throw new IOException("Copy aborted");
				}
				if (this.config.copyIndexUpdateQueueIdsOnEmptyDump()
                    || indexPartHandler.receivedCount() > 0)
				{
				    if (indexPartHandler.receivedCount() <= 0) {
                        logger.warning(
                            "No docs in dump, but we still updating "
                                + "queueIds because of config property");
                    }
//                    mergePolicy.setExpungeAll(true);
//                    mergePolicy.exp
//                    optimizerWriter.expungeDeletes(true);
					optimizerWriter.commit();
					optimizerWriter.close();
					optimizerWriter = null;
					if (checkDups) {
						IndexReader reader =
							IndexReader.open(
								optimizeDir,
								false,
								index.codecProvider());
						index.getShard(outShard).removeDuplicates(reader);
						boolean optimize = false;
						float deletedPercent = (float) reader.numDeletedDocs()
							/ (float) reader.numDocs();
						if (deletedPercent > 0.1) {
							optimize = true;
						}
						reader.close();
						if (optimize) {
							optimizerWriter = new IndexWriter(
								optimizeDir,
								config,
								this.config.storedFields(),
								this.config.indexedFields());
							optimizerWriter.expungeDeletes();
							optimizerWriter.commit();
							optimizerWriter.close();
							optimizerWriter = null;
						}
					}

					index.getShard(outShard).addIndexes(
						true,
						optimizeDir,
						replace,
						indexPartHandler.queueIds());
				} else {
				    logger.warning(
				        "No docs in dump, skipping update queueIds");
                }
			}

		} finally {
			if (out != null) {
				out.close();
			}
			if (is != null) {
				is.close();
			}
			if (sock != null) {
				sock.close();
			}
			if (inputDir != null) {
				inputDir.close();
			}
			if (optimizerWriter != null) {
				optimizerWriter.close();
			}
			if (optimizeDir != null) {
				optimizeDir.close();
			}
			if (recvDir != null) {
				recvDir.close();
			}
			if (recvPath != null) {
				index.getShard(outShard).deleteTemporaryDirectory(recvPath);
			}
			if (optimizePath != null) {
				index.getShard(outShard).deleteTemporaryDirectory
					(optimizePath);
			}
			if (replace || disableIndexing) {
				index.getShard(outShard).flush(true, ro);
			}
		}

		return true;
    }

	public void abort() {
		logger.warning("Aborting index get");
		
		synchronized (lock) {
			if (inputDir != null) {
				inputDir.abort();
			}
			try {
				sock.shutdownInput();
				if (optimizerWriter != null) {
					optimizerWriter.close(false);
				}
			} catch (Exception e) {
			}
		}
	}

    private Socket createConnection(
        String url,
        int totalShards,
        int shard,
        int userShardStart,
        int userShardEnd) throws IOException
    {
        String host;
        int port = 80;
        //	String[] urlParts = url.split(":");
        int sep = url.lastIndexOf(':');
        if (sep == -1) {
            throw new IOException("Invalid url: " + url);
        }
        host = url.substring(0, sep);
        try {
            port = Integer.parseInt(url.substring(sep + 1));
        } catch (Exception e) {
            throw new IOException("Invalid url: can't parse port number", e);
        }
        Socket sock = new Socket();
        sock.setSoTimeout(
            timeout * 1000); //FIXME: make timeout configurable from conf file
        sock.setTcpNoDelay(true);
        sock.setKeepAlive(true);
        sock.connect(
            new InetSocketAddress(host, port),
            10000); //FIXME: make timeout configurable from conf file
        return sock;
    }

    private static class IndexPartHandler implements PartHandler {
        IndexWriterConfig conf;
        IndexWriter out;
        int prevCount = 0;
        int receivedCount = 0;
        private final HashMap<QueueShard, Long> queueIds = new HashMap<>();

        public IndexPartHandler(IndexWriter out, IndexWriterConfig conf) {
            this.conf = conf;
            this.out = out;
        }

        public Map<QueueShard, Long> queueIds() {
            return queueIds;
        }

        @Override
        public void handlePart(Directory dir) throws IOException
        {
            out.addIndexes(true, dir);
            out.commit();
            if (out.maxDoc() != prevCount) {
                receivedCount++;
                prevCount = out.maxDoc();
            }
//	    out.expungeDeletes(false);
        }

        @Override
        public void handleQueueIds(final Map<QueueShard, Long> newQueueIds) {
            for (Map.Entry<QueueShard, Long> entry : newQueueIds.entrySet()) {
                if (queueIds.containsKey(entry.getKey())) {
                    if (queueIds.get(entry.getKey()) < entry.getValue()) {
                        queueIds.put(entry.getKey(), entry.getValue());
                    }
                } else {
                    queueIds.put(entry.getKey(), entry.getValue());
                }
            }
        }

        public int receivedCount()
        {
            return receivedCount;
        }
    }
}
