package ru.yandex.msearch;

import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

import ru.yandex.concurrent.TimeFrameQueue;
import ru.yandex.http.util.server.ImmutableBaseServerConfig;
import ru.yandex.stater.SimpleHistogramMetric;
import ru.yandex.stater.Stater;
import ru.yandex.stater.StatsConsumer;

public class IndexLagStorage implements Stater {
    private final SimpleHistogramMetric lagHistMetric;
    // queue -> shard -> lag
    private final Map<String, Map<Integer, Lag>> lags =
        new ConcurrentHashMap<>();

    private final TimeFrameQueue<Long> lagsQueue;

    public IndexLagStorage(
        final Config config,
        final ImmutableBaseServerConfig serverConfig)
    {
        lagsQueue = new TimeFrameQueue<>(serverConfig.metricsTimeFrame());
        lagHistMetric = config.indexLagHistogram();
    }

    @Override
    public <E extends Exception> void stats(
        final StatsConsumer<? extends E> statsConsumer) throws E
    {
        if (lagHistMetric != null) {
            SimpleHistogramMetric requestMetric
                = new SimpleHistogramMetric(lagHistMetric);

            for (Long ts: lagsQueue) {
                requestMetric.accept(ts);
            }

            requestMetric.stats(statsConsumer);
        }
    }

    public Map<String, Map<Integer, Lag>> lags() {
        return lags;
    }

    public void storeLag(
        final String queue,
        final int shard,
        final long operationTimestamp)
    {
        long now = System.currentTimeMillis();
        long lag = Math.max(now - operationTimestamp, 0L);
        lags.computeIfAbsent(queue, key -> new ConcurrentHashMap<>())
            .put(shard, new Lag(lag, now));

        lagsQueue.accept(lag);
    }

    public static class Lag {
        private final long lag;
        private final long timestamp;

        public Lag(final long lag, final long timestamp) {
            this.lag = lag;
            this.timestamp = timestamp;
        }

        public long lag() {
            return lag;
        }

        public long timestamp() {
            return timestamp;
        }
    }
}
