package ru.yandex.msearch;

import java.io.Closeable;
import java.io.IOException;
import java.io.Writer;
import java.util.Map;
import java.util.Map.Entry;

import org.apache.http.HttpException;
import org.apache.http.HttpRequest;
import org.apache.http.HttpResponse;
import org.apache.http.HttpStatus;
import org.apache.http.entity.ContentType;
import org.apache.http.entity.EntityTemplate;
import org.apache.http.protocol.HttpContext;
import org.apache.http.protocol.HttpRequestHandler;

import ru.yandex.http.server.sync.ContentProducerWriter;
import ru.yandex.http.server.sync.ContentWriter;

import ru.yandex.http.util.CharsetUtils;
import ru.yandex.http.util.ServerException;

import ru.yandex.json.writer.JsonType;
import ru.yandex.json.writer.JsonTypeExtractor;
import ru.yandex.json.writer.JsonWriter;

import ru.yandex.parser.uri.CgiParams;

public class IndexStatusHandler implements HttpRequestHandler {
    protected final DatabaseManager dbManager;
    protected final Config config;

    public IndexStatusHandler(final DatabaseManager dbManager, final Config config) {
        this.dbManager = dbManager;
        this.config = config;
    }

    @Override
    public void handle(final HttpRequest request,
        final HttpResponse response,
        final HttpContext context)
        throws HttpException, IOException
    {
        CgiParams params = new CgiParams(request);
        Index index = dbManager.index(params);

        IndexStatusParams indexStatusParams =
            new IndexStatusParams(params);
        EntityTemplate entity = new EntityTemplate(
            new ContentProducerWriter(
                new IndexStatusProducer(index, indexStatusParams),
                request));
        entity.setChunked(true);
        ContentType contentType;
        if (indexStatusParams.jsonType() == null) {
            contentType = ContentType.TEXT_PLAIN;
        } else {
            contentType = ContentType.APPLICATION_JSON;
        }
        entity.setContentType(contentType.withCharset(
            CharsetUtils.acceptedCharset(request)).toString());
        response.setEntity(entity);
    }

    @Override
    public String toString() {
        return "Print index status information";
    }

    private class IndexStatusParams {
        private final JsonType jsonType;

        public IndexStatusParams(final CgiParams params)
            throws HttpException
        {
            jsonType = JsonTypeExtractor.NULL.extract(params);
        }

        public JsonType jsonType() {
            return jsonType;
        }
    }

    private interface IndexStatusWriter extends Closeable {
        void startIndex() throws IOException;
        void endIndex() throws IOException;
        void startShard(final int shardNumber) throws IOException;
        void endShard() throws IOException;
        void statusInfo(final String infoName, final String infoValue)
            throws IOException;
    }

    private static class PlainIndexStatusWriter implements IndexStatusWriter {
        private final StringBuilder sb = new StringBuilder();
        private final Writer writer;

        public PlainIndexStatusWriter(final Writer writer) {
            this.writer = writer;
        }

        @Override
        public void close() throws IOException {
            if (sb.length() > 0) {
                writer.append(sb);
                sb.setLength(0);
            }
            writer.close();
        }

        @Override
        public void startIndex() throws IOException {
        }

        @Override
        public void endIndex() throws IOException {
        }

        @Override
        public void startShard(final int shardNumber) throws IOException {
            sb.append("shard: ");
            sb.append(shardNumber);
        }

        @Override
        public void endShard() throws IOException {
            sb.append('\n');
        }

        @Override
        public void statusInfo(final String infoName, final String infoValue)
            throws IOException
        {
            sb.append(' ');
            sb.append(infoName);
            sb.append(": ");
            sb.append(infoValue);
        }
    }

    private static class JsonIndexStatusWriter implements IndexStatusWriter {
        private final JsonWriter writer;
        private boolean closed = false;

        public JsonIndexStatusWriter(final JsonWriter writer)
            throws IOException
        {
            this.writer = writer;
            writer.startObject();
        }

        @Override
        public void close() throws IOException {
            if (!closed) {
                closed = true;
                writer.endObject();
                writer.close();
            }
        }

        @Override
        public void startIndex() throws IOException {
            writer.key("shards");
            writer.startArray();
        }

        @Override
        public void endIndex() throws IOException {
            writer.endArray();
        }

        @Override
        public void startShard(final int shardNumber) throws IOException {
            writer.startObject();
            writer.key("shard");
            writer.value(shardNumber);
        }

        @Override
        public void endShard() throws IOException {
            writer.endObject();
        }

        @Override
        public void statusInfo(final String infoName, final String infoValue)
            throws IOException
        {
            writer.key(infoName);
            writer.value(infoValue);
        }
    }

    private static class IndexStatusProducer implements ContentWriter {
        private final Index index;
        private final IndexStatusParams params;

        public IndexStatusProducer(
            final Index index,
            final IndexStatusParams params)
            throws IOException
        {
            this.index = index;
            this.params = params;
        }

        private IndexStatusWriter createWriter(final Writer out)
            throws IOException
        {
            if (params.jsonType() == null) {
                return new PlainIndexStatusWriter(out);
            } else {
                return new JsonIndexStatusWriter(params.jsonType().create(out));
            }
        }

        @Override
        public void writeTo(final Writer out) throws IOException {
            try (IndexStatusWriter writer = createWriter(out)) {
                writer.startIndex();
                index.traverseShards(
                    new ShardVisitor() {
                        @Override
                        public void visit(int shardNo, Shard shard)
                            throws IOException
                        {
                            Shard.ShardStatus status = shard.status();
                            writer.startShard(shardNo);
                            for (Map.Entry<String,Object> info :
                                status.infos().entrySet())
                            {
                                writer.statusInfo(info.getKey(),
                                info.getValue().toString());
                            }
                            writer.endShard();
                        }
                    });
                writer.endIndex();
            }
        }
    }
}

