package ru.yandex.msearch;

/**
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

import org.apache.lucene.store.Directory;
import org.apache.lucene.store.IndexOutput;
import org.apache.lucene.store.SingleInstanceLockFactory;

import java.io.IOException;
import java.io.InputStream;
import java.util.HashMap;
import java.util.Map;
import java.util.Iterator;
import java.util.concurrent.ConcurrentHashMap;
import ru.yandex.msearch.PrintStreamDirectory.REMOTE_CODES;

/**
 * A memory-resident {@link Directory} implementation.  Locking
 * implementation is by default the {@link SingleInstanceLockFactory}
 * but can be changed with {@link #setLockFactory}.
 */
public class InputStreamDirectory {
  protected final Map<String,InputStreamFile> fileMap = new ConcurrentHashMap<String,InputStreamFile>();
  protected final Map<Integer,InputStreamFile> fileNoMap = new ConcurrentHashMap<Integer,InputStreamFile>();
  protected InputStreamFile currentFile = null;
  protected int currentFileNo = -1;
  boolean closed = false;
  private InputStream inputStream;
  private Directory recvDir;
  private boolean dirOpened = false;
  private boolean abort = false;

  /** Constructs an empty {@link Directory}. */
  public InputStreamDirectory( InputStream stream, Directory recvDir ) {
    this.inputStream = stream;
    this.recvDir = recvDir;
  }

  /** Closes the store to future operations, releasing associated memory. */
  public void close()  throws IOException {
    Iterator<Map.Entry<String,InputStreamFile>> iter = fileMap.entrySet().iterator();
    while( iter.hasNext() )
    {
	Map.Entry<String,InputStreamFile> entry = iter.next();
	InputStreamFile f = entry.getValue();
	f.close();
    }
    fileMap.clear();
    fileNoMap.clear();
  }

    private void readQueueIds(final PartHandler handler)
        throws IOException
    {
        Map<QueueShard, Long> ids = new HashMap<>();
        while (readVInt() == REMOTE_CODES.QUEUE_IDS_NEXT_ID) {
            final QueueShard shard = new QueueShard(readString(), readVInt());
            ids.put(shard, readVLong());
        }
        handler.handleQueueIds(ids);
    }
  
  public final byte readByte() throws IOException
  {
    int b = inputStream.read();
    if( b == -1 ) throw new IOException( "End of channel" );
    return (byte)b;
  }
  
  public final void readBytes( byte[] b, int off, int len ) throws IOException
  {
    int received = 0;
    int ret;
    while( received < len )
    {
	ret = inputStream.read( b, off + received, len - received );
	if( ret == -1 ) throw new IOException( "End of channel" );
	received += ret;
    }
  }
  
  public int readVInt() throws IOException
  {
    byte b = readByte();
    int i = b & 0x7f;
    for (int shift = 7; b < 0; shift += 7) {
      b = readByte();
      i |= (b & 0x7f) << shift;
    }
    return i;
  }

  public long readVLong() throws IOException
  {
    byte b = readByte();
    long i = b & 0x7fL;
    for (int shift = 7; b < 0; shift += 7) {
      b = readByte();
      i |= (b & 0x7fL) << shift;
    }
    return i;
  }

  public String readString() throws IOException
  {
    int length = readVInt();
    final byte[] bytes = new byte[length];
    readBytes(bytes, 0, length);
    return new String(bytes, 0, length, "UTF-8");
  }

  private void checkOpen() throws IOException
  {
    if( !dirOpened )
	throw new IOException( "Received command for file operation while directory is not opened..." );
  }

  private void clearDir() throws IOException
  {
    close();
    String[] files = recvDir.listAll();
    for( int i = 0; i < files.length; i++ )
    {
	recvDir.deleteFile( files[i] );
    }
  }

  public void recv(final PartHandler handler) throws IOException {
    int filecount = 0;
    int cmd = 0;
    while( !abort )
    {
	cmd = readVInt();
	if( cmd == -1 ) break;
//	System.err.println( "RECEOVED CMD: " + cmd );
	switch( cmd )
	{
	    case REMOTE_CODES.FINISHED:
		clearDir();
		return;
	    case REMOTE_CODES.NEW_DIR:
		dirOpened = true;
		clearDir();
		currentFileNo = -1;
		currentFile = null;
		filecount = 0;
		break;
	    case REMOTE_CODES.NEW_FILE:
		filecount++;
		checkOpen();
		rmtNewFile();
		break;
	    case REMOTE_CODES.FILE_FLUSH:
	    case REMOTE_CODES.FILE_CLOSE:
	    case REMOTE_CODES.FILE_SEEK:
	    case REMOTE_CODES.FILE_WRITE_BYTE:
	    case REMOTE_CODES.FILE_WRITE_BYTES:
		checkOpen();
		rmtFileOp( cmd );
		break;
	    case REMOTE_CODES.DELETE_FILE:
		checkOpen();
		rmtDeleteFile();
		break;
	    case REMOTE_CODES.QUEUE_IDS_START:
	        readQueueIds(handler);
	        break;
	    case REMOTE_CODES.CLOSE_DIR:
		try
		{
		    if( filecount > 0 )  handler.handlePart( recvDir );
		}
		finally
		{
		    clearDir();
		}
		dirOpened = false;
		break;
	    default:
		throw new IOException( "Unknown command <" + cmd + "> received from server... Possible data corruption or version difference...Abadoning..." );
	}
    }
    if( dirOpened )
    {
	clearDir();
	if( abort ) throw new IOException( "Aborting index receiving by user request" );
	else throw new IOException( "Unknown command <" + cmd + "> received from server... Possible data corruption or version difference...Abadoning..." );
    }
  }
  
  private void rmtNewFile() throws IOException
  {
    int fileNo = readVInt();
    String name = readString();
    InputStreamFile file = new InputStreamFile( this, recvDir.createOutput(name), name );
    fileMap.put(name, file);
    fileNoMap.put(fileNo, file);
    currentFile = file;
    currentFileNo = fileNo;
  }
  
  private void rmtDeleteFile() throws IOException
  {
    int fileNo = readVInt();
    InputStreamFile file = fileNoMap.get(fileNo);
    if( file != null )
    {
	file.close();
	recvDir.deleteFile( file.name() );
    }
  }
  
  private void rmtFileOp( int cmd ) throws IOException
  {
    int fileNo = readVInt();
    InputStreamFile file;
    if( fileNo == currentFileNo ) file = currentFile;
    else
    {
	file = fileNoMap.get(fileNo);
	currentFileNo = fileNo;
	currentFile = file;
	if( file == null ) throw new IOException( "Invalid file id received: uknown file" );
    }

    file.remoteCmd( cmd );
  }
  
  private static class InputStreamFile
  {
    private IndexOutput out;
    private String name;
    private boolean closed = false;
    private InputStreamDirectory parent;
    private static final int BUFFER_SIZE = 1024;
    private byte[] buffer = null;
    
    public InputStreamFile( InputStreamDirectory parent, IndexOutput out, String name )
    {
	this.out = out;
	this.name = name;
	this.parent = parent;
    }
    
    public void close() throws IOException
    {
	if( !closed )
	    out.close();
	closed = true;
    }
    
    public String name()
    {
	return name;
    }
    
    public void remoteCmd( int cmd ) throws IOException
    {
	switch( cmd )
	{
	    case REMOTE_CODES.FILE_FLUSH:
		out.flush();
		break;
	    case REMOTE_CODES.FILE_CLOSE:
		close();
		break;
	    case REMOTE_CODES.FILE_SEEK:
		out.seek( parent.readVLong() );
		break;
	    case REMOTE_CODES.FILE_WRITE_BYTE:
		out.writeByte( parent.readByte() );
		break;
	    case REMOTE_CODES.FILE_WRITE_BYTES:
		readBytes();
		break;
	}
    }
    
    private void readBytes() throws IOException
    {
	if( buffer == null ) buffer = new byte[BUFFER_SIZE];
	int len = parent.readVInt();
//	System.err.println( "readBytes: " + len );
	while( len > 0 )
	{
	    int toRead;
	    toRead = BUFFER_SIZE < len ? BUFFER_SIZE : len;
	    parent.readBytes( buffer, 0, toRead );
	    out.writeBytes( buffer, 0, toRead );
	    len -= toRead;
	}
    }
  }
  
  public void abort()
  {
    abort = true;
  }

    public interface PartHandler {
        public void handlePart(Directory d) throws IOException;
        public void handleQueueIds(Map<QueueShard, Long> queueIds);
    }
  
/*  
  protected void finalize()
  {
    if( !closed )
    {
	System.err.println( "FINALIZING DIRECTORY!!!" );
    }
  }
*/  
}
