package ru.yandex.msearch;

import java.io.BufferedInputStream;
import java.io.Closeable;
import java.io.EOFException;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;

import java.nio.channels.FileChannel;
import java.nio.file.StandardOpenOption;

import java.util.Arrays;
import java.util.Comparator;
import java.util.EnumMap;
import java.util.Locale;
import java.util.Map;
import java.util.concurrent.atomic.AtomicInteger;

import java.util.logging.Logger;

import ru.yandex.msearch.config.DatabaseConfig;
import ru.yandex.msearch.util.IOScheduler;

public class Journal implements Closeable {
    private final Logger logger;

    private static class JournalFile {
        public final File file;
        public final FileChannel output;
        public long position = 0;
        public boolean failed = false;
        private static AtomicInteger fileNumber = new AtomicInteger();
        public static String getSuffix(Message.Type type) {
            return "." + type.name().toLowerCase(Locale.ENGLISH) + ".journal";
        }
        public JournalFile(Message.Type type, String path) throws IOException {
            file = File.createTempFile( "ram" + String.format("%010d", fileNumber.incrementAndGet()), getSuffix(type), new File(path) );
            output = FileChannel.open(
                file.toPath(),
                StandardOpenOption.CREATE,
                StandardOpenOption.WRITE);
        }

        public void close() throws IOException
        {
            output.close();
            file.delete();
        }
    }

    private final Map<Message.Type, JournalFile> files =
        new EnumMap<Message.Type, JournalFile>(Message.Type.class);

    public Journal(final String path, final Logger logger) throws IOException {
        this.logger = logger;
        for (Message.Type type: Message.Type.values()) {
            files.put(type, new JournalFile(type, path));
        }
    }

    public long journalMessage( JournalableMessage message ) throws IOException
    {
        JournalFile file = files.get(message.type());
        return IOScheduler.instance().writeOp(
            IOScheduler.IOPRIO_JOURNAL,
            () -> {
                synchronized (file) {
                    if (file.failed) {
                        file.output.position(file.position);
                    } else {
                        file.failed = true;
                    }
                    try {
                        if (message.writeTo(file.output) != 0) {
                            file.position = file.output.position();
                        }
                        file.failed = false;
                    } catch (IOException e) {
                        try {
                            file.output.position(file.position);
                            file.failed = false;
                        } catch (IOException ex) {
                            e.addSuppressed(ex);
                        }
                        throw e;
                    }
                    return file.position;
                }
            });
    }

    public void close() throws IOException
    {
        for (JournalFile file: files.values()) {
            file.close();
        }
    }

    public static void replayJournal(
        final JournalHandler handler,
        final String path,
        final DatabaseConfig config,
        final Logger logger)
    {
        File dir = new File(path);
        File[] files = dir.listFiles();
        Arrays.sort(files, new Comparator<File>() {
            @Override
            public int compare(File lhs, File rhs) {
                return lhs.getName().compareTo(rhs.getName());
            }

            @Override
            public boolean equals(Object obj) {
                return this == obj;
            }
        });
        for( int i = 0; i < files.length; i++ )
        {
            for (Message.Type type: Message.Type.values()) {
                if (files[i].getName().endsWith(JournalFile.getSuffix(type))) {
                    if (files[i].length() == 0
                        || replayJournal(
                            files[i],
                            type,
                            handler,
                            config,
                            logger))
                    {
                        files[i].delete();
                    } else {
                        files[i].renameTo(new File(files[i].getPath() + ".corrupted"));
                    }
                    break;
                }
            }
        }
        try {
            handler.close();
        } catch (IOException e) {
            e.printStackTrace();
        }
    }

    public static boolean replayJournal(
        final File f,
        final Message.Type type,
        final MessageHandler handler,
        final DatabaseConfig config,
        final Logger logger)
    {
        boolean success = true;
        int msgCount = 0;
        logger.info("Replaying journal: " + f.getAbsolutePath());
        try (InputStream in = new BufferedInputStream(new FileInputStream(f));
            MessageReader reader =
                type.reader(in, Message.Priority.JOURNAL, config, logger))
        {
            for (Message message = reader.next(); message != null;
                message = reader.next())
            {
                try {
                    if (message instanceof JournalableMessage) {
                        //just to check Queue ID
                        handler.journalMessage((JournalableMessage)message);
                    }
                    message.handle(handler);
                } catch (PrimaryKeyNotFoundException e) {
                    // Keep Calm and Carry On
                } catch (ConflictPrimaryKeyException e) {
                    // Keep Calm and Carry On
                } catch (QueueIdOutOfOrderException e) {
                    logger.fine("Skipping already indexed message" +
                        " with id: " + ((JournalableMessage)message).queueId());
                    // Keep Calm and Carry On
                } finally {
                    if (message instanceof JournalableMessage) {
                        ((JournalableMessage)message).close();
                    }
                }
                msgCount++;
            }
        } catch( EOFException e ) {
            // do nothing
        } catch( Exception e ) {
            e.printStackTrace();
            success = false;
        } finally {
            try {
                handler.reopen();
            } catch(IOException e) {
                e.printStackTrace();
                success = false;
            }
        }
        logger.info("Replaying journal: " + f.getAbsolutePath()
            + " finished: "
            + (success ? ("successfull: msgCount: " + msgCount) : "failed"));
        return success;
    }
}

