package ru.yandex.msearch;

import java.io.IOException;

import java.text.ParseException;

import java.util.Collection;
import java.util.HashMap;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.Callable;
import java.util.concurrent.atomic.AtomicLong;

import java.util.logging.Logger;

import org.apache.lucene.analysis.SimpleAnalyzer;
import org.apache.lucene.index.IndexReader;
import org.apache.lucene.index.IndexWriter;
import org.apache.lucene.index.IndexWriterConfig;
import org.apache.lucene.index.LogByteSizeMergePolicy;
import org.apache.lucene.search.IndexSearcher;
import org.apache.lucene.search.QueryProducer;
import org.apache.lucene.store.Directory;
import org.apache.lucene.util.Version;

import ru.yandex.collection.IntSet;

import ru.yandex.logger.PrefixedLogger;

import ru.yandex.msearch.config.DatabaseConfig;
import ru.yandex.search.prefix.Prefix;

public class JournalHandler
    implements AnalyzerProvider, IndexAccessor, IndexManager, MessageHandler
{
    private static final int DELETED_CACHE_SIZE = 1000000;

    private final IndexWriter fsWriter;
    private final IndexWriterConfig config;
    private final PrefixingAnalyzerWrapper searchAnalyzer;
    private final PrefixingAnalyzerWrapper indexAnalyzer;
    private final Map<QueueShard, QueueId> queueIds;
    private final Map<Prefix, Long> prefixActivity;
    private final Integer currentVersion;
    private final PartFactory partFactory;
    private final DatabaseConfig daemonConfig;
    private final PrefixedLogger logger;
    private final Logger indexLogger;

    private AbstractPart part = null;
    private IndexReader reader = null;
    private IndexSearcher searcher = null;
    private boolean searcherClosed = true;
    private boolean searcherDirty = false;

    public JournalHandler(
        final IndexWriter fsWriter,
        final IndexWriterConfig config,
        final AnalyzerProvider analyzerProvider,
        final Map<QueueShard, QueueId> queueIds,
        final Map<Prefix, Long> prefixActivity,
        final Integer currentVersion,
        final DatabaseConfig daemonConfig,
        final PrefixedLogger logger,
        final Logger indexLogger)
        throws IOException
    {
        this.fsWriter = fsWriter;
        this.config = config;
        this.indexAnalyzer = analyzerProvider.indexAnalyzer();
        this.searchAnalyzer = analyzerProvider.searchAnalyzer();
        this.queueIds = queueIds;
        this.prefixActivity = prefixActivity;
        this.currentVersion = currentVersion;
        this.daemonConfig = daemonConfig;
        this.logger = logger;
        this.indexLogger = indexLogger;
        partFactory = PartFactory.constructFactory(daemonConfig);
        part = partFactory.create(this, this, this, queueIds, 0);
    }

    @Override
    public PrefixingAnalyzerWrapper indexAnalyzer() {
        return indexAnalyzer;
    }

    @Override
    public PrefixingAnalyzerWrapper searchAnalyzer() {
        return searchAnalyzer;
    }

    private void closeSearcher() throws IOException {
        if (searcher != null) {
            reader.close();
            reader = null;
            searcher.close();
            searcher = null;
            searcherClosed = true;
        }
    }

    @Override
    public CloseableSearcher searcher() throws IOException {
        if (searcherDirty) {
            closeSearcher();
        }
        if (searcher == null) {
//            fsWriter.commit();
            reader = IndexReader.open(fsWriter, true);
            searcher = new IndexSearcher(reader);
            searcherDirty = false;
        }
        return new CloseableSearcher() {
            @Override
            public IndexSearcher searcher() {
                return searcher;
            }

            @Override
            public void close() {
            }
        };
    }

    @Override
    public void executeTasks(Collection<Callable<Void>> tasks) throws IOException {
        try {
            for (Callable task : tasks) {
                task.call();
            }
        } catch (IOException e) {
            throw e;
        } catch (Exception e) {
            throw new IOException(e);
        }
    }

    @Override
    public void primaryKeyAtomicOp(final PrimaryKey primaryKey,
        final AtomicOp op)
        throws IOException
    {
        op.run();
    }

    @Override
    public void prefixReadAtomicOp(final Prefix prefix,
        final PrefixOp op)
        throws IOException, ParseException
    {
        op.run();
    }
    @Override
    public void prefixWriteAtomicOp(final Prefix prefix,
        final PrefixOp op)
        throws IOException, ParseException
    {
        op.run();
    }

    @Override
    public void deleteDocuments(final QueryProducer query) throws IOException {
        fsWriter.deleteDocuments(query);
        searcherDirty = true;
//        closeSearcher();
    }

    @Override
    public void traverseParts(final PartVisitor visitor, final long version) throws IOException {
        visitor.visit(part);
    }

    @Override
    public IndexWriter createWriter(final Directory dir) throws IOException {
        IndexWriterConfig newConfig = new IndexWriterConfig(Version.LUCENE_40,
            new SimpleAnalyzer(true));
        newConfig.setReaderTermsIndexDivisor(
            config.getReaderTermsIndexDivisor());
        newConfig.setTermIndexInterval(config.getTermIndexInterval());
        newConfig.setOpenMode(IndexWriterConfig.OpenMode.CREATE);
        newConfig.setMaxBufferedDocs(1 * 1024 * 1024);
        newConfig.setRAMBufferSizeMB(IndexWriterConfig.DISABLE_AUTO_FLUSH);
        newConfig.setFieldsWriterBufferSize(config.getFieldsWriterBufferSize());
        LogByteSizeMergePolicy mergePolicy = new LogByteSizeMergePolicy();
        mergePolicy.setUseCompoundFile(false);
        mergePolicy.setMergeFactor(1024);
        newConfig.setMergePolicy(mergePolicy);
        return new IndexWriter(
            dir,
            newConfig,
            daemonConfig.storedFields(),
            daemonConfig.indexedFields());
    }

    @Override
    public Journal createJournal() {
        return null;
    }

    @Override
    public void add(DocumentsMessage message, Map<String, String> conditions)
        throws IOException, ParseException
    {
        part.add(message, conditions);
    }

    @Override
    public void modify(DocumentsMessage message, Map<String, String> conditions)
        throws IOException, ParseException
    {
        part.modify(message, conditions);
    }

    @Override
    public void delete(DocumentsMessage message)
        throws IOException, ParseException
    {
        part.delete(message);
    }

    @Override
    public void delete(DeleteMessage message)
        throws IOException, ParseException
    {
        part.delete(message);
    }

    @Override
    public void update(DocumentsMessage message, Map<String, String> conditions, boolean addIfNotExists, boolean orderIndependentUpdate, Set<String> preservedFields)
        throws IOException, ParseException
    {
        part.update(message, conditions, addIfNotExists, orderIndependentUpdate, preservedFields);
    }

    @Override
    public void updateIfNotMatches(UpdateIfNotMatchesMessage message, Map<String, String> conditions, boolean addIfNotExists, boolean orderIndependentUpdate)
        throws IOException, ParseException
    {
        part.updateIfNotMatches(message, conditions, addIfNotExists, orderIndependentUpdate);
    }

    @Override
    public void update(UpdateMessage message,
                       Map<String, String> conditions,
                       boolean addIfNotExists,
                       boolean orderIndependentUpdate)
        throws IOException, ParseException
    {
        part.update(message, conditions, addIfNotExists, orderIndependentUpdate);
    }

    public long journalMessage( JournalableMessage message ) throws IOException {
        return part.journalMessage(message);
    }

    @Override
    public long queueId(final QueueShard shard) {
        final AtomicLong id = queueIds.get(shard);
        if (id != null) {
            return id.get();
        }
        return -1;
    }

    @Override
    public void updateQueueIds(final Map<QueueShard, QueueId> ids) {
        queueIds.putAll(ids);
    }

    private Map<String,String> commitData() {
        final Map<String,String> commitData = new HashMap<String,String>();

        if (currentVersion != null) {
            commitData.put("version", currentVersion.toString());
        }

        for (Map.Entry<QueueShard, QueueId> entry : queueIds.entrySet()) {
            commitData.put(
                entry.getKey().toString(),
                Long.toString(entry.getValue().get()));
        }
        for (final Map.Entry<Prefix, Long> entry:
            prefixActivity.entrySet())
        {
            commitData.put(
                Shard.ACTIVE_PREFIX + entry.getKey().toString(),
                Long.toString(entry.getValue()));
        }
        return commitData;
    }

    @Override
    public void reopen() throws IOException {
        closeSearcher();
        part.atomicWriteTo(fsWriter);
        updateQueueIds(part.getQueueIds());
        fsWriter.commit(commitData());
        part.delete();
        part = partFactory.create(this, this, this, queueIds, 0);
    }

    @Override
    public void reopen(
        final IntSet shards,
        final Boolean ro,
        final boolean doWait)
        throws IOException
    {
    }

    @Override
    public void dirtifyQueueShard(final QueueShard shard, final long queueId) {
    }

    @Override
    public PrefixedLogger logger() {
        return logger;
    }

    @Override
    public Logger indexLogger() {
        return indexLogger;
    }

    public void close() throws IOException {
        closeSearcher();
        part.close();
        part.delete();
    }
}

