package ru.yandex.msearch;

import java.io.IOException;

import java.nio.ByteBuffer;
import java.nio.channels.WritableByteChannel;
import java.nio.charset.StandardCharsets;

import java.text.ParseException;

import org.apache.http.NameValuePair;

import org.apache.lucene.analysis.Analyzer;
import org.apache.lucene.queryParser.QueryParser;
import org.apache.lucene.search.Query;
import org.apache.lucene.util.Version;

import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.YandexHeaders;

import ru.yandex.msearch.config.DatabaseConfig;
import ru.yandex.parser.string.NonEmptyValidator;

import ru.yandex.parser.uri.CgiParams;
import ru.yandex.parser.uri.UriParser;

import ru.yandex.queryParser.YandexQueryParser;

import ru.yandex.search.prefix.Prefix;

public class JsonDeleteMessage implements DeleteMessage {
    private static final String ZOO_QUEUE_ID_TO_CHECK =
        "\r\n" + YandexHeaders.ZOO_QUEUE_ID_TO_CHECK + ": ";
    private static final String ZOO_QUEUE_ID =
        "\r\n" + YandexHeaders.ZOO_QUEUE_ID + ": ";
    private static final String ZOO_SHARD_ID =
        "\r\n" + YandexHeaders.ZOO_SHARD_ID + ": ";
    private static final String ZOO_QUEUE =
        "\r\n" + YandexHeaders.ZOO_QUEUE + ": ";

    private final int priority;
    private final MessageQueueId queueId;
    private final DatabaseConfig config;
    private final byte[] dump;
    private final Prefix prefix;
    private final QueueShard queueShard;
    private final String text;
    private final MessageContext context;

    public JsonDeleteMessage(
        final MessageContext context,
        final String uri,
        final int priority,
        final QueueShard queueShard,
        final MessageQueueId queueId,
        final DatabaseConfig config,
        final boolean journalable)
        throws ParseException
    {
        this.context = context;
        this.priority = priority;
        this.queueId = queueId;
        this.queueShard = queueShard;
        this.config = config;
        if (journalable) {
            StringBuilder sb = new StringBuilder("GET ");
            sb.append(uri);
            sb.append(" HTTP/1.1");
            if (queueShard.shardId() != -1) {
                sb.append(ZOO_SHARD_ID);
                sb.append(queueShard.shardId());
                sb.append(ZOO_QUEUE);
                sb.append(queueShard.service());
            }
            if (queueId.weakCheck()) {
                sb.append(ZOO_QUEUE_ID_TO_CHECK);
            } else {
                sb.append(ZOO_QUEUE_ID);
            }
            sb.append(queueId.queueId());
            sb.append("\r\n\r\n");
            dump = new String(sb).getBytes(StandardCharsets.UTF_8);
        } else {
            dump = null;
        }
        CgiParams params = new CgiParams(new UriParser(uri).queryParser());
        try {
            prefix = params.get("prefix", config.indexPrefixParser());
            text = params.get("text", NonEmptyValidator.INSTANCE);
        } catch (BadRequestException e) {
            ParseException ex = new ParseException("Bad delete request", 0);
            ex.initCause(e);
            throw ex;
        }
    }

    @Override
    public void close() {
    }

    @Override
    public void handle(final MessageHandler handler)
        throws IOException, ParseException
    {
        handler.delete(this);
    }

    @Override
    public Type type() {
        return Type.JSON;
    }

    @Override
    public Prefix prefix() {
        return prefix;
    }

    @Override
    public QueueShard queueShard() {
        if (queueShard.shardId() == -1) {
            return new QueueShard(queueShard.service(), prefix());
        }
        return queueShard;
    }

    @Override
    public MessageQueueId queueId() {
        return queueId;
    }

    @Override
    public boolean journalable() {
        return dump != null;
    }

    @Override
    public int writeTo(final WritableByteChannel channel) throws IOException {
        ByteBuffer buf = ByteBuffer.wrap(dump);
        while (buf.hasRemaining()) {
            channel.write(buf);
        }
        return dump.length;
    }

    @Override
    public Query query(final Analyzer analyzer) throws ParseException {
        QueryParser parser =
            new YandexQueryParser(Version.LUCENE_40, null, analyzer, config);
        parser.setAllowLeadingWildcard(true);
        parser.setAnalyzeRangeTerms(true);
        try {
            return parser.parse(text);
        } catch (org.apache.lucene.queryParser.ParseException exc) {
            ParseException e = new ParseException(exc.getMessage(), 0);
            e.initCause(exc);
            throw e;
        }
    }

    @Override
    public int priority() {
        return priority;
    }

    @Override
    public MessageContext context() {
        return context;
    }
}

