package ru.yandex.msearch;

import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.nio.charset.UnsupportedCharsetException;

import java.io.EOFException;
import java.io.IOException;
import java.io.InputStream;

import java.text.ParseException;

import java.util.logging.Logger;

import org.apache.http.ConnectionClosedException;
import org.apache.http.Header;
import org.apache.http.HttpException;
import org.apache.http.HttpRequest;
import org.apache.http.RequestLine;
import org.apache.http.config.MessageConstraints;
import org.apache.http.entity.ContentType;
import org.apache.http.impl.io.DefaultHttpRequestParser;
import org.apache.http.impl.io.HttpTransportMetricsImpl;
import org.apache.http.impl.io.SessionInputBufferImpl;
import org.apache.http.io.HttpMessageParser;
import org.apache.http.protocol.HTTP;

import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.HeadersParser;
import ru.yandex.http.util.YandexHeaders;

import ru.yandex.msearch.config.DatabaseConfig;
import ru.yandex.msearch.index.BaseIndexHandler;
import ru.yandex.parser.uri.CgiParams;

public class JsonReader implements MessageReader {
    private static final int BUFFER_SIZE = 65536;

    private final InputStream in;
    private final int priority;
    private final DatabaseConfig config;
    private final SessionInputBufferImpl buffer;
    private final HttpMessageParser<HttpRequest> parser;
    private final Logger logger;

    public JsonReader(
        final InputStream in,
        final int priority,
        final DatabaseConfig config,
        final Logger logger)
    {
        this.in = in;
        this.priority = priority;
        this.config = config;
        this.logger = logger;
        buffer = new SessionInputBufferImpl(
            new HttpTransportMetricsImpl(),
            BUFFER_SIZE,
            BUFFER_SIZE,
            MessageConstraints.DEFAULT,
            StandardCharsets.UTF_8.newDecoder());
        buffer.bind(in);
        parser = new DefaultHttpRequestParser(buffer);
    }

    @Override
    public void close() throws IOException {
        buffer.clear();
        in.close();
    }

    @Override
    public Message next() throws IOException, ParseException {
        HttpRequest request;
        try {
            request = parser.parse();
        } catch (ConnectionClosedException e) {
            return null;
        } catch (HttpException e) {
            ParseException ex = new ParseException("Failed to parse json", 0);
            ex.initCause(e);
            throw ex;
        }
        try {
            MessageQueueId queueId = BaseIndexHandler.extractQueueId(request);
            final HeadersParser headers = new HeadersParser(request);
            final String service =
                headers.getString(
                    YandexHeaders.ZOO_QUEUE,
                    QueueShard.DEFAULT_SERVICE);
            final int shardId =
                headers.getInt(
                    YandexHeaders.ZOO_SHARD_ID,
                    -1);
            final QueueShard shard = new QueueShard(service, shardId);
            MessageContext context = new MessageContext() {
                @Override
                public Logger logger() {
                    return logger;
                }
            };
            RequestLine requestLine = request.getRequestLine();
            final CgiParams params = new CgiParams(request);
            if (requestLine.getMethod().equals("GET")) {
                if (requestLine.getUri().startsWith("/delete?")) {
                    return new JsonDeleteMessage(
                        context,
                        requestLine.getUri(),
                        priority,
                        shard,
                        queueId,
                        config,
                        false);
                } else if (requestLine.getUri().startsWith("/reopen")) {
                    return new ReopenMessage(context, params, priority);
                }
            } else if (requestLine.getMethod().equals("POST")) {
                int len = Integer.valueOf(
                    request.getFirstHeader(HTTP.CONTENT_LEN).getValue());
                byte[] dump = new byte[len];
                for (int off = 0; off < len;) {
                    int read = buffer.read(dump, off, len - off);
                    if (read == -1) {
                        throw new EOFException("Read " + off + " of " + len
                            + " bytes");
                    }
                    off += read;
                }
                Header contentType = request.getFirstHeader(HTTP.CONTENT_TYPE);
                Charset charset = null;
                if (contentType != null) {
                    try {
                        charset = ContentType.parse(contentType.getValue())
                            .getCharset();
                    } catch (org.apache.http.ParseException
                        | UnsupportedCharsetException e)
                    {
                        ParseException ex = new ParseException(
                            "Failed to parse content-type: "
                            + contentType.getValue(), 0);
                        ex.initCause(e);
                        throw ex;
                    }
                }

                if (charset == null) {
                    charset = StandardCharsets.UTF_8;
                }

                final boolean orderIndependentUpdate = params.getBoolean("order-independent-update",
                        config.orderIndependentUpdate());

                switch (requestLine.getUri()) {
                    case "/add":
                        return new JsonAddMessage(
                            context,
                            dump,
                            charset,
                            priority,
                            shard,
                            queueId,
                            config,
                            false,
                            orderIndependentUpdate);
                    case "/modify":
                        return new JsonModifyMessage(
                            context,
                            dump,
                            charset,
                            priority,
                            shard,
                            queueId,
                            config,
                            false,
                            orderIndependentUpdate);
                    case "/delete":
                        return new JsonBatchDeleteMessage(
                            context,
                            dump,
                            charset,
                            priority,
                            shard,
                            queueId,
                            config,
                            false,
                            orderIndependentUpdate);
                    case "/update":
                        return new JsonUpdateMessage(
                            context,
                            dump,
                            charset,
                            priority,
                            shard,
                            queueId,
                            config,
                            false,
                            orderIndependentUpdate);
                    default:
                        if (requestLine.getUri().startsWith("/add?")) {
                            return new JsonAddMessage(
                                context,
                                dump,
                                charset,
                                priority,
                                shard,
                                queueId,
                                config,
                                false,
                                orderIndependentUpdate);
                        }
                        break;
                }
            }
        } catch (BadRequestException e) {
            ParseException ex = new ParseException("Failed to parse json", 0);
            ex.initCause(e);
            throw ex;
        } catch (RuntimeException e) {
            ParseException ex = new ParseException("Failed to parse json", 0);
            ex.initCause(e);
            throw ex;
        }
        return null;
    }
}

