package ru.yandex.msearch;

import java.io.IOException;
import java.nio.charset.Charset;
import java.text.ParseException;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.lucene.analysis.Analyzer;
import org.apache.lucene.queryParser.QueryParser;
import org.apache.lucene.search.Query;
import org.apache.lucene.util.Version;

import ru.yandex.msearch.config.DatabaseConfig;
import ru.yandex.msearch.indexprocessor.IndexDocProcessor;
import ru.yandex.msearch.indexprocessor.IndexDocProcessorFactory;
import ru.yandex.queryParser.YandexQueryParser;
import ru.yandex.search.json.UpdateDocumentsMapCollector;
import ru.yandex.search.json.UpdateMessageRootHandler;
import ru.yandex.search.json.fieldfunction.FieldFunction;
import ru.yandex.search.prefix.Prefix;

public class JsonUpdateMessage
    extends AbstractJsonMessage
{
    private final Prefix prefix;
    private final DocumentsMessage documentsMessage;
    private final UpdateIfNotMatchesMessage updateINMMessage;
    private final UpdateMessage updateMessage;
    private final boolean addIfNotExists;
    private final boolean updateIfNotMatches;
    private final boolean orderIndependentUpdate;
    private final Map<String,String> conditions;
    private final Set<String> preserveFields;

    public JsonUpdateMessage(
        final MessageContext context,
        final byte[] dump,
        final Charset charset,
        final int priority,
        final QueueShard queueShard,
        final MessageQueueId queueId,
        final DatabaseConfig config,
        final boolean journalable,
        final boolean orderIndependentUpdate)
        throws IOException, ParseException
    {
        super(
            context,
            "/update",
            dump,
            charset,
            priority,
            queueShard,
            queueId,
            config,
            journalable);

        UpdateDocumentsMapCollector collector =
            new UpdateDocumentsMapCollector(config.indexPrefixParser());
        parse(
            dump,
            manager -> new UpdateMessageRootHandler(manager, collector));
        prefix = collector.getPrefix();
        if (prefix == null) {
            throw new ParseException("prefix not set", 0);
        }
        addIfNotExists = collector.getAddIfNotExists();
        updateIfNotMatches = collector.getUpdateIfNotMatches();
        this.orderIndependentUpdate = orderIndependentUpdate;

        preserveFields = collector.getPreserveFields();

        if (preserveFields != null) {
            for (String field: collector.getPreserveFields()) {
                if (!config.knownFields().contains(field)) {
                    throw new ParseException(
                        "Unknown preserve field " + field, 0);
                }

                preserveFields.add(field);
            }

            if (config.primaryKey() != null) {
                for (String field: config.primaryKey()) {
                    preserveFields.add(field);
                }
            }

            preserveFields.add(Config.QUEUE_ID_FIELD_KEY);
            preserveFields.add(Config.QUEUE_NAME_FIELD_KEY);
        }

        conditions = collector.getConditions();
        List<Map<String, FieldFunction>> documents = collector.getDocuments();
        if (documents.isEmpty()) {
            throw new ParseException(
                "no documents passed for update message",
                0);
        }

        IndexDocProcessor docProcessor =
            IndexDocProcessorFactory.INTSANCE.create(
                collector.docProcessor(),
                config,
                context.logger());

        String query = collector.getQuery();
        if (query == null) {
            if (config.primaryKey() == null || updateIfNotMatches) {
                throw new ParseException("query not set", 0);
            }
            updateMessage = null;
            updateINMMessage = null;
            documentsMessage =
                new JsonUpdateDocumentsMessage(
                    context,
                    this,
                    docProcessor,
                    documents,
                    orderIndependentUpdate,
                    config);
        } else {
            if (updateIfNotMatches) {
                updateMessage = null;
                documentsMessage = null;
                updateINMMessage =
                    new JsonUpdateIfNotMatchesMessage(
                        this,
                        documents,
                        query,
                        orderIndependentUpdate,
                        config);
            } else {
                documentsMessage = null;
                updateINMMessage = null;
                updateMessage =
                    new JsonUpdateByQueryMessage(
                        this,
                        documents,
                        docProcessor,
                        query,
                        config);
            }
        }
    }

    @Override
    public void close() throws IOException {
        if (documentsMessage != null) {
            documentsMessage.close();
        }
        if (updateINMMessage != null) {
            updateINMMessage.close();
        }
        if (updateMessage != null) {
            updateMessage.close();
        }
    }

    @Override
    public Prefix prefix() {
        return prefix;
    }

    @Override
    public Type type() {
        return Type.JSON;
    }

    @Override
    public void handle(final MessageHandler handler)
        throws IOException, ParseException
    {
        if (documentsMessage != null) {
            handler.update(documentsMessage, conditions, addIfNotExists, orderIndependentUpdate, preserveFields);
        } else if (updateINMMessage != null) {
            handler.updateIfNotMatches(updateINMMessage, conditions, addIfNotExists, orderIndependentUpdate);
        } else {
            handler.update(updateMessage, conditions, addIfNotExists, orderIndependentUpdate);
        }
    }

    private static class JsonUpdateDocumentsMessage
        extends FilterJournalableMessage
        implements DocumentsMessage
    {
        private final HTMLDocument[] docs;
        private final IndexDocProcessor docProcessor;

        public JsonUpdateDocumentsMessage(
            final MessageContext context,
            final JournalableMessage message,
            final IndexDocProcessor docProcessor,
            final List<Map<String, FieldFunction>> documents,
            final boolean orderIndependentUpdate,
            final DatabaseConfig config)
            throws ParseException
        {
            super(message);
            this.docProcessor = docProcessor;
            HTMLDocument[] docs = new HTMLDocument[documents.size()];
            Map<PrimaryKey, UpdateOnlyHTMLDocument> deduper = new HashMap<>();
            Set<String> primaryKeyFields = config.primaryKey();
            int d = 0;
            boolean success = false;
            UpdateOnlyHTMLDocument uohd = null;
            try {
                for (int i = 0; i < documents.size(); ++i) {
                    uohd =
                        new UpdateOnlyHTMLDocument(
                            documents.get(i),
                            prefix(),
                            queueId().phantomQueueId(),
                            queueShard().service(),
                            orderIndependentUpdate,
                            config,
                            context.logger());
                    UpdateOnlyHTMLDocument existing =
                        deduper.get(uohd.primaryKey());
                    if (existing != null) {
                        boolean canMerge = true;
                        for (String field : documents.get(i).keySet()) {
                            if (primaryKeyFields.contains(field)) {
                                continue;
                            }
                            if (existing.updateFields().containsKey(field)) {
                                canMerge = false;
                                break;
                            }
                        }
                        if (canMerge) {
                            existing.updateFields().putAll(documents.get(i));
                        } else {
                            deduper.put(uohd.primaryKey(), uohd);
                            docs[d++] = uohd;
                        }
                    } else {
                        deduper.put(uohd.primaryKey(), uohd);
                        docs[d++] = uohd;
                    }
                    uohd = null;
                }
                if (d < docs.length) {
                    docs = Arrays.copyOf(docs, d);
                }
                success = true;
            } finally {
                if (!success) {
                    for (int i = 0; i < d; i++) {
                        docs[i].close();
                    }
                    if (uohd != null) {
                        uohd.close();
                    }
                }
            }
            this.docs = docs;
        }

        @Override
        public void close() {
            if (docs != null) {
                for (HTMLDocument doc : docs) {
                    doc.close();
                }
            }
        }

        @Override
        public HTMLDocument[] documents() {
            return docs;
        }

        @Override
        public IndexDocProcessor docProcessor() {
            return docProcessor;
        }
    }

    private static class JsonQueryUpdateMessage
        extends FilterJournalableMessage
    {
        private final String query;
        private final DatabaseConfig config;

        public JsonQueryUpdateMessage(
            final JournalableMessage message,
            final String query,
            final DatabaseConfig config)
        {
            super(message);
            this.query = query;
            this.config = config;
        }

        public Query query(final Analyzer analyzer)
            throws ParseException
        {
            QueryParser parser =
                new YandexQueryParser(
                    Version.LUCENE_40,
                    null,
                    analyzer,
                    config);
            parser.setAllowLeadingWildcard(true);
            parser.setAnalyzeRangeTerms(true);
            try {
                Query q = parser.parse(query);
                // Force exception
                q.toString();
                return q;
            } catch (Exception exc) {
                ParseException e = new ParseException(exc.getMessage(), 0);
                e.initCause(exc);
                throw e;
            }
        }
    }

    private static class JsonUpdateIfNotMatchesMessage
        extends JsonQueryUpdateMessage
        implements UpdateIfNotMatchesMessage
    {
        private final HTMLDocument[] docs;

        public JsonUpdateIfNotMatchesMessage(
            final JournalableMessage message,
            final List<Map<String, FieldFunction>> documents,
            final String query,
            final boolean orderIndependentUpdate,
            final DatabaseConfig config)
            throws ParseException
        {
            super(message, query, config);
            docs = new HTMLDocument[documents.size()];
            int i = 0;
            boolean success = false;
            try {
                for (; i < documents.size(); ++i) {
                    docs[i] =
                        new UpdateOnlyHTMLDocument(
                            documents.get(i),
                            prefix(),
                            queueId().phantomQueueId(),
                            queueShard().service(),
                            orderIndependentUpdate,
                            config,
                            context().logger());
                }
                success = true;
            } finally {
                if (!success) {
                    for (int j = 0; j < i; j++) {
                        docs[j].close();
                    }
                }
            }
        }

        @Override
        public void close() {
            if (docs != null) {
                for (HTMLDocument doc : docs) {
                    doc.close();
                }
            }
        }

        @Override
        public HTMLDocument[] documents() {
            return docs;
        }
    }

    private static class JsonUpdateByQueryMessage
        extends JsonQueryUpdateMessage
        implements UpdateMessage
    {
        private final Map<String, FieldFunction> document;
        private final IndexDocProcessor docProcessor;

        public JsonUpdateByQueryMessage(
            final JournalableMessage message,
            final List<Map<String, FieldFunction>> documents,
            final IndexDocProcessor docProcessor,
            final String query,
            final DatabaseConfig config)
            throws ParseException
        {
            super(message, query, config);
            if (documents.size() > 1) {
                throw new ParseException(
                    "Only one document allowed for update messages", 0);
            }

            this.docProcessor = docProcessor;
            document = Collections.unmodifiableMap(documents.get(0));
            if (document.isEmpty()) {
                throw new ParseException("no fields for update specified", 0);
            }
            Set<String> primaryKey = config.primaryKey();
            if (primaryKey != null) {
                for (String key: primaryKey) {
                    if (document.containsKey(key)) {
                        throw new ParseException("Cannot update field '" + key
                            + "' as it is a part of primary key", 0);
                    }
                }
            }
        }

        @Override
        public IndexDocProcessor docProcessor() {
            return docProcessor;
        }

        @Override
        public void close() {
        }

        @Override
        public Map<String, FieldFunction> document() {
            return document;
        }
    }
}

