package ru.yandex.msearch;

import java.io.IOException;
import java.nio.charset.Charset;
import java.util.Map;

import org.apache.http.HttpException;
import org.apache.http.HttpRequest;
import org.apache.http.HttpResponse;
import org.apache.http.entity.ContentType;
import org.apache.http.entity.EntityTemplate;
import org.apache.http.protocol.HttpContext;
import org.apache.http.protocol.HttpRequestHandler;

import ru.yandex.http.server.sync.JsonContentProducerWriter;
import ru.yandex.http.util.CharsetUtils;
import ru.yandex.json.writer.JsonTypeExtractor;
import ru.yandex.json.writer.JsonValue;
import ru.yandex.json.writer.JsonWriterBase;
import ru.yandex.parser.string.PositiveLongValidator;
import ru.yandex.parser.uri.CgiParams;

public class LagsHandler implements HttpRequestHandler {
    private final IndexLagStorage lagsStorage;

    public LagsHandler(final IndexLagStorage lagsStorage) {
        this.lagsStorage = lagsStorage;
    }

    @Override
    public void handle(
        final HttpRequest request,
        final HttpResponse response,
        final HttpContext context)
        throws HttpException
    {
        CgiParams params = new CgiParams(request);
        Long ttl = params.get(
            "ttl",
            null,
            PositiveLongValidator.INSTANCE);
        long minTimestamp;
        long expiredValue;
        if (ttl == null) {
            minTimestamp = 0L;
            expiredValue = 0L;
        } else {
            minTimestamp = System.currentTimeMillis() - ttl;
            expiredValue = params.getLong("expired-value");
        }
        Charset charset = CharsetUtils.acceptedCharset(request);
        EntityTemplate entity =
            new EntityTemplate(
                new JsonContentProducerWriter(
                    new LagsWriter(lagsStorage.lags(), minTimestamp, expiredValue),
                    JsonTypeExtractor.NORMAL.extract(params),
                    charset));
        entity.setChunked(true);
        entity.setContentType(
            ContentType.APPLICATION_JSON.withCharset(charset).toString());
        response.setEntity(entity);
    }

    private static class LagsWriter implements JsonValue {
        private final Map<String, Map<Integer, IndexLagStorage.Lag>> lags;
        private final long minTimestamp;
        private final long expiredValue;

        LagsWriter(
            final Map<String, Map<Integer, IndexLagStorage.Lag>> lags,
            final long minTimestamp,
            final long expiredValue)
        {
            this.lags = lags;
            this.minTimestamp = minTimestamp;
            this.expiredValue = expiredValue;
        }

        @Override
        public void writeValue(final JsonWriterBase writer)
            throws IOException
        {
            StringBuilder sb = new StringBuilder(10);
            writer.startObject();
            for (Map.Entry<String, Map<Integer, IndexLagStorage.Lag>> entry
                : lags.entrySet())
            {
                writer.key(entry.getKey());
                writer.startObject();
                for (Map.Entry<Integer, IndexLagStorage.Lag> queueEntry
                    : entry.getValue().entrySet())
                {
                    sb.setLength(0);
                    sb.append(queueEntry.getKey().intValue());
                    writer.key(sb);
                    IndexLagStorage.Lag indexLag = queueEntry.getValue();
                    long lag;
                    if (indexLag.timestamp() < minTimestamp) {
                        lag = expiredValue;
                    } else {
                        lag = indexLag.lag();
                    }
                    writer.value(lag);
                }
                writer.endObject();
            }
            writer.endObject();
        }
    }
}

