package ru.yandex.msearch;

import java.io.IOException;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.util.function.Predicate;

import javax.script.ScriptContext;
import javax.script.ScriptEngine;
import javax.script.ScriptEngineManager;
import javax.script.SimpleScriptContext;

import org.apache.http.HttpEntityEnclosingRequest;
import org.apache.http.HttpException;
import org.apache.http.HttpRequest;
import org.apache.http.HttpResponse;
import org.apache.http.HttpStatus;
import org.apache.http.entity.ContentType;
import org.apache.http.entity.EntityTemplate;
import org.apache.http.protocol.HttpContext;
import org.apache.http.protocol.HttpRequestHandler;

import ru.yandex.collection.PatternMap;
import ru.yandex.http.server.sync.JsonContentProducerWriter;
import ru.yandex.http.server.sync.Utf8JsonContentProducer;
import ru.yandex.http.util.CharsetUtils;
import ru.yandex.http.util.request.RequestInfo;
import ru.yandex.json.writer.JsonTypeExtractor;
import ru.yandex.json.writer.JsonValue;
import ru.yandex.json.writer.JsonWriterBase;
import ru.yandex.json.writer.Utf8JsonValue;
import ru.yandex.json.writer.Utf8JsonWriter;
import ru.yandex.lympho.OnlineLymphoContext;
import ru.yandex.msearch.collector.outergroup.OuterGroupFunctionFactory;
import ru.yandex.msearch.util.Compress;
import ru.yandex.msearch.util.IOScheduler;
import ru.yandex.msearch.util.IOStater;
import ru.yandex.parser.uri.CgiParams;

public class LymphoExecuteHandler extends SearchHandlerBase implements HttpRequestHandler {
    private final OuterGroupFunctionFactory outerGroupFunctionFactory;
    private final ScriptEngine engine;

    public LymphoExecuteHandler(
        final DatabaseManager dbManager,
        final Config config,
        final OuterGroupFunctionFactory outerGroupFunctionFactory,
        final PatternMap<RequestInfo, IOStater> ioStaters)
    {
        super(dbManager, config, ioStaters);

        this.outerGroupFunctionFactory = outerGroupFunctionFactory;
        engine = getEngine();
    }

    private ScriptEngine getEngine() {
        ScriptEngineManager factory = new ScriptEngineManager();
        for (String engineName:
            new String[] {
                "Graal.js",
                "Javascript",
                "nashorn"
            })
        {
            ScriptEngine engine = factory.getEngineByName(engineName);
            if (engine != null) {
                if (engineName.equals("Graal.js")) {
                    engine.put("polyglot.js.allowHostAccess", true);
                    engine.put(
                        "polyglot.js.allowHostClassLookup",
                        (Predicate) s -> true);
                }
                return engine;
            }
        }
        return null;
    }

    @Override
    public void handle(
        final HttpRequest request,
        final HttpResponse response,
        final HttpContext context)
        throws HttpException, IOException
    {
        CgiParams params = new CgiParams(request);

        Index index = dbManager.indexOrException(params, SearchHandler.BRE_GEN);

        IOScheduler.setThreadReadPrio(
            params.getInt("IO_PRIO", IOScheduler.IOPRIO_SEARCH));
        Compress.updateSsdCache(params.getBoolean("update-ssd-cache", true));

        Charset charset = CharsetUtils.acceptedCharset(request);

        String script = CharsetUtils.toString(((HttpEntityEnclosingRequest) request).getEntity());

        LymphoProducer lymphoProducer = new LymphoProducer(context, index, script);
        EntityTemplate entity;
        if (charset.equals(StandardCharsets.UTF_8)) {
            entity =
                new EntityTemplate(
                    new Utf8JsonContentProducer(
                        lymphoProducer,
                        JsonTypeExtractor.NORMAL.extract(params)));
        } else {
            entity =
                new EntityTemplate(
                    new JsonContentProducerWriter(
                        lymphoProducer,
                        JsonTypeExtractor.NORMAL.extract(params),
                        charset));
        }
        entity.setChunked(true);
        entity.setContentType(
            ContentType.APPLICATION_JSON.withCharset(charset).toString());
        response.setEntity(entity);
        response.setStatusCode(HttpStatus.SC_OK);
    }

    private class LymphoProducer implements Utf8JsonValue, JsonValue {
        private final HttpContext httpContext;
        private final Index index;
        private final String script;

        public LymphoProducer(
            final HttpContext httpContext,
            final Index index,
            final String script)
        {
            this.httpContext = httpContext;
            this.script = script;
            this.index = index;
        }

        @Override
        public void writeValue(final JsonWriterBase writer)
            throws IOException
        {
            try (OnlineLymphoContext context =
                     new OnlineLymphoContext(
                         index,
                         httpContext,
                         outerGroupFunctionFactory,
                         ioStaterFor(httpContext),
                         writer))
            {
                ScriptContext scriptContext = new SimpleScriptContext();
                scriptContext.setAttribute(
                    "context",
                    context,
                    ScriptContext.ENGINE_SCOPE);

                engine.eval(script, scriptContext);
            } catch (Exception e) {
                throw new IOException(e);
            }
        }


        @Override
        public void writeValue(final Utf8JsonWriter writer) throws IOException {
            writeValue((JsonWriterBase) writer);
        }
    }
}