package ru.yandex.msearch;

import java.io.IOException;
import java.text.ParseException;
import java.util.Collection;
import java.util.LinkedList;
import java.util.List;
import java.util.ListIterator;
import java.util.Queue;
import java.util.concurrent.Callable;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.locks.ReentrantReadWriteLock;
import java.util.logging.Level;
import java.util.logging.Logger;

import org.apache.lucene.analysis.SimpleAnalyzer;
import org.apache.lucene.index.*;
import org.apache.lucene.search.*;
import org.apache.lucene.store.*;
import org.apache.lucene.index.codecs.CodecProvider;
import org.apache.lucene.util.Version;

import ru.yandex.logger.PrefixedLogger;
import ru.yandex.msearch.config.DatabaseConfig;

public class MemoryIndex implements IndexManager {
    private static final boolean DEBUG = false;

    private final ThreadPoolExecutor parallelExecutor;

    private final AtomicLong partVersion = new AtomicLong(0);
    private final Object reopenSearcherLock = new Object();
    private final Object getSearcherLock = new Object();
    private final AtomicLong indexGeneration = new AtomicLong(0);
    private final AtomicLong searcherGeneration = new AtomicLong(0);
    private final AtomicLong memDocsSize = new AtomicLong();
    private final Queue<AbstractPart> written = new LinkedBlockingQueue<>() ;
    private final CopyOnWriteArrayList<AbstractPart> parts = new CopyOnWriteArrayList<>();
    private final ReentrantReadWriteLock partsLock = new ReentrantReadWriteLock();
    private final ReentrantReadWriteLock indexingLock = new ReentrantReadWriteLock();
    private final IndexWriterConfig config;
    private final String journalPath;
    private final DatabaseConfig daemonConfig;
    private final PartFactory partFactory;
    private final CodecProvider cp;
    private final AnalyzerProvider analyzerProvider;
    private final IndexAccessor indexAccessor;
    private final PrefixedLogger logger;
    private final Logger indexLogger;
    private volatile boolean closed = false;

    private MemorySearcher currentSearcher = null;

    public MemoryIndex(
        final IndexWriterConfig config,
        final String journalPath,
        final AnalyzerProvider analyzerProvider,
        final IndexAccessor indexAccessor,
        final DatabaseConfig daemonConfig,
        final CodecProvider cp,
        final ThreadPoolExecutor parallelExecutor)
        throws IOException
    {
        this.config = config;
        this.journalPath = journalPath;
        this.daemonConfig = daemonConfig;
        this.cp = cp;
        this.logger = indexAccessor.logger();
        this.indexLogger = indexAccessor.indexLogger();
        this.parallelExecutor = parallelExecutor;
        partFactory = PartFactory.constructFactory(daemonConfig);
        if (daemonConfig.useFastCommitCodec()) {
            cp.setDefaultFieldCodec( "FastCommit" );
        }
        this.analyzerProvider = analyzerProvider;
        this.indexAccessor = indexAccessor;
        parts.add(
            partFactory.create(
                analyzerProvider,
                this,
                indexAccessor,
                null,
                partVersion.getAndIncrement()));
        reopenSearcher(indexGeneration.incrementAndGet(), parts);
    }

    @Override
    public void executeTasks(Collection<Callable<Void>> tasks) throws IOException {
        if (parallelExecutor != null) {
            List<Future> futures = new LinkedList<Future>();
            try {
                for (Callable task : tasks) {
                    futures.add(parallelExecutor.submit(task));
                }
                for (Future f : futures) {
                    f.get();
                }
            } catch (ExecutionException e) {
                Throwable cause = e.getCause();
                if (cause instanceof IOException) {
                    throw (IOException)cause;
                } else {
                    throw new IOException(cause);
                }
            } catch (Exception e) {
                throw new IOException(e);
            }
        } else {
            try {
                for (Callable task : tasks) {
                    task.call();
                }
            } catch (IOException e) {
                throw e;
            } catch (Exception e) {
                throw new IOException(e);
            }
        }
    }

    @Override
    public void traverseParts(final PartVisitor partsVisitor, long version)
        throws IOException
    {
	partsLock.readLock().lock();
        ListIterator<AbstractPart> partsIter = parts.listIterator();
	try
	{
	    while( partsIter.hasNext() ) {
	        AbstractPart part = partsIter.next();
	        part.incRef();
	    }
	}
	finally
	{
	    partsLock.readLock().unlock();
	}

        boolean finished = false;
        try {
            while( partsIter.hasPrevious() ) {
                AbstractPart part = partsIter.previous();
                //traverse parts in reverse order from the new to the oldest one to prevent double indexing
                //then we first delete/index doc from older part to new part and then again delete/index it from newest part
                if( part.version() <= version ) {
                    partsVisitor.visit(part);
                }
            }
            finished = true;
        } finally {
            if( !finished ) {
                //first rewind iterator to the start
                while( partsIter.hasPrevious() ) partsIter.previous();
            }
	    while( partsIter.hasNext() ) {
	        AbstractPart part = partsIter.next();
	        part.free();
	    }
        }
    }

    @Override
    public IndexWriter createWriter(final Directory dir) throws IOException {
        IndexWriterConfig newConfig = new IndexWriterConfig( Version.LUCENE_40, new SimpleAnalyzer( true ) );
        newConfig.setReaderTermsIndexDivisor( config.getReaderTermsIndexDivisor() );
        newConfig.setTermIndexInterval( config.getTermIndexInterval() );
        newConfig.setCodecProvider( cp );
        newConfig.setOpenMode( IndexWriterConfig.OpenMode.CREATE );
        newConfig.setMaxBufferedDocs(1 * 1024 * 1024);
        newConfig.setRAMBufferSizeMB(IndexWriterConfig.DISABLE_AUTO_FLUSH);
        newConfig.setMaxThreadStates( config.getMaxThreadStates() );
        newConfig.setFieldsWriterBufferSize(config.getFieldsWriterBufferSize());
        newConfig.setMergeScheduler(new SerialMergeScheduler());
        LogByteSizeMergePolicy mergePolicy = new LogByteSizeMergePolicy();
        mergePolicy.setUseCompoundFile( false );
        mergePolicy.setMergeFactor( 1000 );
        newConfig.setMergePolicy( mergePolicy );
        return new IndexWriter(
            dir,
            newConfig,
            daemonConfig.storedFields(),
            daemonConfig.indexedFields());
    }

    @Override
    public Journal createJournal() throws IOException {
        if (journalPath == null) {
            return null;
        } else {
            return new Journal(journalPath, logger);
        }
    }

    public ProcessingResult process(final JournalableMessage message)
        throws IOException, ParseException
    {
        indexingLock.readLock().lock();
	partsLock.readLock().lock();
        AbstractPart part = null;
        if( closed ) throw new IOException( "MemoryIndex have been closed" );
        ListIterator<AbstractPart> partsIter;
        boolean journaled = false;
        long journalSize;
        try
        {
            part = parts.get(parts.size()-1);
            partsIter = parts.listIterator();
            part.incUses();
            journalSize = part.journalMessage(message);
            journaled = true;
        }
        finally
        {
    	    partsLock.readLock().unlock();
            if( !journaled )
            {
                if( part != null ) part.decUses();
    	        indexingLock.readLock().unlock();
            }
        }
        long messageRamUsage;
        try {
            while( partsIter.hasNext() )
            {
                AbstractPart previous = partsIter.next();
                if( previous == part ) break;
                //delay execution until previous part finishes their requests
                previous.waitFree();
            }
            long usageBefore = part.size();
            message.handle(part);
            // tabolin@ said that negative values can be obtained in case of
            // disk segments merge during operation
            messageRamUsage = Math.abs(part.size() - usageBefore);
            indexGeneration.incrementAndGet();
    	} finally {
    	    part.decUses();
    	    indexingLock.readLock().unlock();
    	}
    	final boolean needFlush = part.needFlush()
            || journalSize >= daemonConfig.maxPartJournalSize();
        return new ProcessingResult(
            messageRamUsage,
            needFlush && part.version() == partVersion.get() - 1);
    }

    public Searcher getSearcher() throws IOException
    {
        long indexGen = indexGeneration.get();
        long searcherGen = searcherGeneration.get();
	if( indexGen > searcherGen || currentSearcher == null )
	{
	    reopenSearcher(indexGen, parts);
	}
	synchronized(getSearcherLock)
	{
	    Searcher retval = currentSearcher;
	    retval.incRef();
	    return retval;
	}
    }

    public void reopenSearcher(long indexMod, List<AbstractPart> parts) throws IOException {
        if (DEBUG) {
            System.err.println("\t\tMemoryIndex.reopenIndex");
        }
        partsLock.readLock().lock();
        try {
            parts = this.parts;
            synchronized (reopenSearcherLock) {
                if (searcherGeneration.get() >= indexMod && currentSearcher != null) {
                    return; //Double check;
                }

                synchronized (getSearcherLock) {
                    Searcher oldSearcher = currentSearcher;
                    currentSearcher = new MemorySearcher(parts);
                    currentSearcher.incRef();
                    if (oldSearcher != null) {
                        oldSearcher.free();
                    }
                }

                searcherGeneration.set(indexMod);
            }
        } finally {
            partsLock.readLock().unlock();
        }
    }

    public long queueId(final QueueShard shard) {
        partsLock.readLock().lock();
        try {
            final AbstractPart current = parts.get(parts.size()-1);
            final AtomicLong id = current.getQueueIds().get(shard);
            if (id != null) {
                return id.get();
            }
            return -1;
        } finally {
            partsLock.readLock().unlock();
        }
    }

    public void removePartFromSearcher( AbstractPart part ) throws IOException
    {
	synchronized(getSearcherLock)
	{
	    if( currentSearcher == null ) return;
	    MemorySearcher oldSearcher = currentSearcher;
	    currentSearcher = oldSearcher.removePart(part);
	    currentSearcher.incRef();
	    oldSearcher.free();
	}
    }

    public void flush() throws IOException {
        partsLock.writeLock().lock();
        try {
            long time = System.currentTimeMillis();
            if (logger.isLoggable(Level.INFO)) {
                logger.info("MemoryIndex.Part.commit: "
                    + (System.currentTimeMillis() - time));
            }
            if( parts.size() == 0 ) return;
            AbstractPart current = parts.get(parts.size()-1);
            parts.add(
                partFactory.create(analyzerProvider, this, indexAccessor, current.getQueueIds(), partVersion.getAndIncrement()));
            reopenSearcher( indexGeneration.incrementAndGet(), parts );
            current.setDirty(false);
            current.preFlush();
            if (logger.isLoggable(Level.INFO)) {
                logger.info("Preflushing memory index: part=" + current);
            }
        } finally {
            partsLock.writeLock().unlock();
        }
    }

    public AbstractPart firstPart()
    {
        partsLock.readLock().lock();
        try
        {
            if( parts.size() > 1 ) return parts.get(0);
            return null;
        }
        finally
        {
	    partsLock.readLock().unlock();
	}
    }

    public void lockParts()
    {
        partsLock.writeLock().lock();
    }

    public void unlockParts()
    {
        if( partsLock.isWriteLockedByCurrentThread() ) partsLock.writeLock().unlock();
    }

    public void lockIndexing()
    {
        indexingLock.writeLock().lock();
    }

    public void unlockIndexing()
    {
        if( indexingLock.isWriteLockedByCurrentThread() ) indexingLock.writeLock().unlock();
    }

    public void removeFirstPart(AbstractPart part) throws IOException {
        AbstractPart removed = parts.remove(0);
        if (removed != part) {
            throw new IOException("MemoryIndex parts consistency failed");
        }
        long newSize = 0;
        for (AbstractPart p : parts) {
            newSize += p.size();
        }
        memDocsSize.set(newSize);
        if (!closed) {
            removePartFromSearcher(part);
        }
    }

    public long size() {
        long size = 0;
        try {
            for (AbstractPart part : parts) {
                size += part.size();
            }
        } finally {
        }
        return size;
    }

    public long activeSize()
    {
	partsLock.readLock().lock();
	long size = 0;
	try
	{
	    if( parts.size() > 0 )
	    {
	        size = parts.get(parts.size()-1).size();
	    }
	}
	finally
	{
	    partsLock.readLock().unlock();
	}
	return size;
    }

    public int docCount() throws IOException
    {
	int docs = 0;
	try
	{
	    for( AbstractPart part : parts )
	    {
	        docs += part.numDocs();
	    }
	}
	finally
	{
	}
	return docs;
    }

    public AbstractPart.PartsStats getPartsStats() throws IOException
    {
	partsLock.readLock().lock();
        AbstractPart.PartStats[] stats = new AbstractPart.PartStats[parts.size()];
	try
	{
	    for( int i = 0; i < parts.size(); i++ )
	    {
	        stats[i] = parts.get(i).stats();
	    }
	}
	finally
	{
	    partsLock.readLock().unlock();
	}
	return new AbstractPart.PartsStats(stats);
    }

    public void close() {
	partsLock.writeLock().lock();
	closed = true;
        partsLock.writeLock().unlock();
    }

    public void free() throws IOException
    {
	partsLock.writeLock().lock();
	try
	{
            if( currentSearcher != null )
            {
                currentSearcher.free();
            }
            AbstractPart current = parts.get(parts.size()-1);
            if( current.dirty() )
            {
                throw new IOException( "Freeing MemoryIndex while current part is dirty and not commited. Will not free part." );
            }
            current.close();
            while( current.refs() > 0 ) current.free();
            parts.remove(current);
        }
        finally
        {
            partsLock.writeLock().unlock();
        }
    }
}
