package ru.yandex.msearch;

import java.io.IOException;

import java.util.Map;
import java.util.concurrent.atomic.AtomicInteger;

import java.util.logging.Level;

import org.apache.lucene.analysis.Analyzer;
import org.apache.lucene.document.Document;
import org.apache.lucene.index.CorruptIndexException;
import org.apache.lucene.search.Query;
import org.apache.lucene.search.TermQuery;

import ru.yandex.msearch.config.DatabaseConfig;
import ru.yandex.msearch.util.NativePositiveLongHashSet;

public class NativeKeysCachingPrimaryKeyPart extends PrimaryKeyPartBase {
    private static final int MAP_ENTRY_SIZE = 48;
    private final NativeUnboxedKeysSet cache;
    private final AtomicInteger keysSize = new AtomicInteger(0);

    private static class NativeUnboxedKeysSet
        extends NativePositiveLongHashSet
    {
        @Override
        protected int keyHashCode(final long key) {
            return NativeCacheKey.hashCode(key);
        }

        @Override
        protected boolean keysEquals(final long a, final long b) {
            return NativeCacheKey.equals(a, b);
        }

        @Override
        protected void freeKey(final long key) {
            NativeCacheKey.free(key);
        }
    }

    public NativeKeysCachingPrimaryKeyPart(
        final AnalyzerProvider analyzerProvider,
        final IndexManager indexManager,
        final IndexAccessor indexAccessor,
        final Map<QueueShard, QueueId> queueIds,
        final long version,
        final DatabaseConfig config)
        throws IOException
    {
        super(
            analyzerProvider,
            indexManager,
            indexAccessor,
            queueIds,
            version,
            config);
        cache = new NativeUnboxedKeysSet();
    }

    @Override
    protected void cleanup() {
        cache.close();
    }

    @Override
    public int indexDocument(final HTMLDocument doc,
        final Analyzer analyzer)
        throws IOException
    {
        //reduntant check
        final PrimaryKey primaryKey = doc.primaryKey();
        final long nativeKey = primaryKey.nativeCacheKey();
        if (!cache.put(nativeKey)) {
            throw new CorruptIndexException(
                "Document for key: " + primaryKey
                + " is already in the part");
        } else {
            //null key for cloning
            primaryKey.setNativeCacheKey(0);
        }
        keysSize.addAndGet(NativeCacheKey.size(nativeKey));
//        cachePut(doc.primaryKey().cacheKey());
        return super.indexDocument(doc, analyzer);
    }

    @Override
    public boolean hasPrimaryKey(final PrimaryKey primaryKey,
        final PrimaryKeySearcher primaryKeySearcher)
        throws IOException
    {
        return cache.contains(primaryKey.nativeCacheKey());
    }

    @Override
    public Document getDocumentByPrimaryKey(final PrimaryKey primaryKey)
        throws IOException
    {
        Query query = primaryKey.query();
        Document[] docs;
        if (query instanceof TermQuery) {
            docs = fastSearch((TermQuery)query);
        } else {
            docs = search(primaryKey.query(), 2);
        }
        if (docs.length != 1) {
            if (logger.isLoggable(Level.SEVERE)) {
                logger.severe("CorruptIndexException: "
                    + "Wrong document count for key: " + primaryKey
                    + ", expected 1, got:" + docs.length);
            }
            for (int t = 0; t < 5; t++) {
                if (query instanceof TermQuery) {
                    docs = fastSearch((TermQuery)query);
                } else {
                    docs = search(primaryKey.query(), 2);
                }
                if (docs.length == 1) {
                    if (logger.isLoggable(Level.SEVERE)) {
                        logger.severe("CorruptIndexException: retry success "
                            + "Wrong document count for key: " + primaryKey
                            + ", expected 1, got:" + docs.length);
                    }
                    return docs[0];
                } else {
                    if (logger.isLoggable(Level.SEVERE)) {
                        logger.severe("CorruptIndexException: retry failed "
                            + "Wrong document count for key: " + primaryKey
                            + ", expected 1, got:" + docs.length);
                    }
                }
            }
            if (docs.length != 1) {
                throw new CorruptIndexException(
                    "Wrong document count for key: " + primaryKey
                    + ", expected 1, got:" + docs.length);
            }
        }
        return docs[0];
    }

    @Override
    public void removePrimaryKey(final PrimaryKey primaryKey) {
        cache.remove(primaryKey.nativeCacheKey());
    }

    @Override
    public boolean deleteDocument(final PrimaryKey primaryKey)
        throws IOException
    {
        final long nativeKey = primaryKey.nativeCacheKey();
        final int keySize = NativeCacheKey.size(nativeKey);
        if (!cache.remove(nativeKey)) {
            return false;
        } else {
            keysSize.addAndGet(-keySize);
            deleteDocuments(primaryKey.queryProducer());
            return true;
        }
    }

    @Override
    public long payloadSize() {
        return super.payloadSize() + keysSize.get() + cache.sizeInBytes();
    }
}
